// 6502Doc.cpp : implementation of the C6502Doc class
//

#include "stdafx.h"
#include "6502.h"

#include "6502Doc.h"
#include "6502View.h"
#include "InputBox.h"
#include "BreaksDlg.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// C6502Doc

IMPLEMENT_DYNCREATE(C6502Doc, CDocument)

BEGIN_MESSAGE_MAP(C6502Doc, CDocument)
	//{{AFX_MSG_MAP(C6502Doc)
	ON_COMMAND(ID_DEBUG_STEPINTO, OnDebugStepInto)
	ON_COMMAND(ID_DEBUG_STEPOVER, OnDebugStepOver)
	ON_COMMAND(ID_DEBUG_RUN, OnDebugRun)
	ON_COMMAND(ID_DEBUG_STOP, OnDebugStop)
	ON_COMMAND(ID_DEBUG_RESET, OnDebugReset)
	ON_COMMAND(ID_DEBUG_RUNTORETURN, OnDebugRunToReturn)
	ON_COMMAND(ID_DEBUG_RUNINITONCE, OnDebugRunInitOnce)
	ON_COMMAND(ID_DEBUG_RUNPLAYONCE, OnDebugRunPlayOnce)
	ON_COMMAND(ID_DEBUG_BREAKS, OnDebugBreaks)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// C6502Doc construction/destruction

C6502Doc::C6502Doc()
{
	m_bDontRun = TRUE;
   m_bDocOpen = m_bRunning = m_bQuitting = false;
}

C6502Doc::~C6502Doc()
{
}

/////////////////////////////////////////////////////////////////////////////
// C6502Doc serialization

void C6502Doc::Serialize(CArchive& ar)
{
	if (!ar.IsStoring())
	{
		CFile *pFile = ar.GetFile();
      DWORD  length;
      byte  *buf   = new byte[length=pFile->GetLength()];
      pFile->Read(buf, length);
      
      int ret=0;

      m_bDocOpen = true; // default to successful

      memset(m_CPU.GetMem(), 0, 65536);

      switch(m_dtType)
      {
         case DT_NOSEFART:
            {
               int i;
               for(i=0x70;i<=0x77;i++) if(buf[i]) break;
               if(i != 0x78)
               {
                  AfxMessageBox(_T("Bankswitched NSF files not supported.\nYou can force loading by selecting All Files when you load this."));
                  m_bDocOpen = false;
                  break;
               }
               WORD pos    = buf[0x08] + buf[0x09]*256;
               m_nInitAddr = buf[0x0A] + buf[0x0B]*256;
               m_nPlayAddr = buf[0x0C] + buf[0x0D]*256;
               ret = m_CPU.Load(pos, buf+0x80, length-0x80);
            }
            break;

         case DT_BINARY:
         default:
            {
               WORD pos;
               int  off;
               GetFileOffsets(&pos, &off);
               m_nInitAddr = m_nPlayAddr = off;
               ret = m_CPU.Load(pos, buf+off, length-off);
            }
            break;
      }
      if(ret)
      {
         AfxMessageBox(ID_STR_CPUERR + ret);
         m_bDocOpen = false;
      }
      delete[] buf;
      if(!m_bDocOpen) UpdateAllViews(NULL, UC_CLOSE), m_bDontRun = TRUE;
      else UpdateAllViews(NULL, UC_RESET), m_bDontRun = FALSE;
	}
}

/////////////////////////////////////////////////////////////////////////////
// C6502Doc diagnostics

#ifdef _DEBUG
void C6502Doc::AssertValid() const
{
	CDocument::AssertValid();
}

void C6502Doc::Dump(CDumpContext& dc) const
{
	CDocument::Dump(dc);
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// C6502Doc commands

void C6502Doc::DontRun(BOOL dontrun)
{
   m_bDontRun = dontrun;
}

bool C6502Doc::IsRunning()
{
   return m_bRunning;
}

bool C6502Doc::CanRun()
{
   return !m_bRunning && !m_bDontRun;
}

void C6502Doc::SetDocType(DocType dt)
{
   m_dtType = dt;
}

C6502Doc::DocType C6502Doc::GetDocType()
{
   return m_dtType;
}

bool C6502Doc::IsDocOpen()
{
   return m_bDocOpen;
}

void C6502Doc::GetProgramStats(WORD *base, WORD *len)
{
   if(base) *base = m_CPU.GetBasePC();
   if(len)  *len  = m_CPU.GetProgLen();
}

WORD C6502Doc::GetInitAddr()
{
   return m_nInitAddr;
}

WORD C6502Doc::GetPlayAddr()
{
   return m_nPlayAddr;
}

void C6502Doc::AddPCBreak(WORD pos)
{
   m_CPU.AddPCBreak(pos);
}

void C6502Doc::AddMemBreak(WORD pos, byte value, MBType op)
{
   m_CPU.AddMemBreak(pos, value, (CMyCpu::MBType)op);
}

void C6502Doc::RemPCBreak(WORD pos)
{
   m_CPU.RemPCBreak(pos);
}

void C6502Doc::RemMemBreak(WORD pos)
{
   m_CPU.RemMemBreak(pos);
}

BOOL C6502Doc::IsPCBreak(WORD pos)
{
   return m_CPU.IsPCBreak(pos);
}

BOOL C6502Doc::IsMemBreak(WORD pos)
{
   return m_CPU.IsMemBreak(pos) != -1;
}

WORD C6502Doc::GetPC()
{
   return m_CPU.GetRegs().PC;
}

void C6502Doc::SetPC(WORD pos)
{
   CMyCpu::REGS regs = m_CPU.GetRegs();
   regs.PC = pos;
   m_CPU.SetRegs(regs);
}

void C6502Doc::RunTo(WORD pos)
{
   m_bRunning = true;
   m_CPU.RunTo(pos);
   m_bRunning = false;
   if(!m_bQuitting) UpdateAllViews(NULL, UC_REPAINT);
}

void C6502Doc::Quit()
{
   m_bQuitting = true;
   OnDebugStop();
}

void C6502Doc::GetFileOffsets(WORD *loadpos, int *offset)
{
   CInputBox posbox(_T("Enter memory position to load progam into (hex):"),
                    _T("Enter position"), _T("0"), CInputBox::VerifyWORD);

   posbox.DoModal();
   *loadpos = posbox.HtoI(posbox.m_strValue);

   CInputBox offbox(_T("Enter file offset to load program from (hex):"),
                    _T("Enter offset"), _T("0"), CInputBox::VerifyWORD);
   
   offbox.DoModal();
   *offset = offbox.HtoI(offbox.m_strValue);
}

void C6502Doc::OnDebugStepInto() 
{
   m_CPU.StepInto();
   UpdateAllViews(NULL, UC_REPAINT);	
}

void C6502Doc::OnDebugStepOver() 
{
   m_bRunning = true;
   m_CPU.StepOver();
   m_bRunning = false;
   if(!m_bQuitting) UpdateAllViews(NULL, UC_REPAINT);	
}

void C6502Doc::OnDebugRun() 
{
   m_bRunning = true;
   m_CPU.Run();
   m_bRunning = false;
   if(!m_bQuitting) UpdateAllViews(NULL, UC_REPAINT);
}

void C6502Doc::OnDebugStop() 
{
   m_CPU.Stop();
}

void C6502Doc::OnDebugReset() 
{
   m_CPU.Reset();
   UpdateAllViews(NULL, UC_REPAINT);
}

void C6502Doc::OnDebugRunToReturn() 
{
   m_bRunning = true;
   m_CPU.RunToReturn();
   m_bRunning = false;
   if(!m_bQuitting) UpdateAllViews(NULL, UC_REPAINT);
}

void C6502Doc::OnDebugRunInitOnce() 
{
   WORD pc = GetPC()-1;
   m_CPU.Push(pc >> 8);
   m_CPU.Push(pc & 0xFF);
   SetPC(GetInitAddr());
   OnDebugRunToReturn();
}

void C6502Doc::OnDebugRunPlayOnce() 
{
   WORD pc = GetPC()-1;
   m_CPU.Push(pc >> 8);
   m_CPU.Push(pc & 0xFF);
   SetPC(GetPlayAddr());
   OnDebugRunToReturn();
}

void C6502Doc::OnDebugBreaks() 
{
   CBreaksDlg dlg;
   
   if(dlg.DoModal() == IDOK)
   {
      m_CPU.ClearPCBreaks();
      m_CPU.ClearMemBreaks();
      int i, len = dlg.m_vctBreaks.size();
      
      // do PC breakpoints first
      for(i=0;i<len;i++)
      {
         CBreaksDlg::BPoint bp = dlg.m_vctBreaks[i];
         if(bp.type == CBreaksDlg::BP_PC) m_CPU.AddPCBreak(bp.pos);
      }
      // then mem breakpoints
      for(i=0;i<len;i++)
      {
         CBreaksDlg::BPoint bp = dlg.m_vctBreaks[i];
         CMyCpu::MBType op;
         if(bp.type == CBreaksDlg::BP_MEMEQ) op = CMyCpu::MB_EQUALS;
         else if(bp.type == CBreaksDlg::BP_MEMCH) op = CMyCpu::MB_CHANGE;
         else continue;
         m_CPU.AddMemBreak(bp.pos, bp.value, op);
      }
      UpdateAllViews(NULL, UC_REPAINT);
   }
}
