// MyCpu.cpp: implementation of the CMyCpu class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "6502.h"
#include "MyCpu.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CMyCpu::CMyCpu()
{
   m_CPU.User = new byte[65536];
}

CMyCpu::~CMyCpu()
{
   delete[] m_CPU.User;
}

void CMyCpu::Run()
{
   MSG msg;
   
   m_bStop = false;

   while(!m_bStop)
   {
      if(m_nMemBreaks)
      {
         for(int i=0;i<100;i++) 
         {
            M6502CPU::Exec6502(&m_CPU);
            if(IsPCBreak(m_CPU.PC.W)) return;
            int i=0, len=m_nMemBreaks;
            for(;i<len;i++)
            {
               BPoint bp = *m_vctMemBP[i];
               if((bp.op == MB_CHANGE && ReadMem(bp.pos) != bp.value) ||
                  (bp.op == MB_EQUALS && ReadMem(bp.pos) == bp.value))
                     return;
            }
         }
      }
      else if(m_nPCBreaks)
      {
         for(int i=0;i<100;i++)
         {
            M6502CPU::Exec6502(&m_CPU);
            if(IsPCBreak(m_CPU.PC.W)) return;
         }
      }
      else
      {
         for(int i=0;i<100;i++) M6502CPU::Exec6502(&m_CPU);
      }

      if(::PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE))
         AfxGetApp()->PumpMessage();
   }
}

void CMyCpu::RunTo(WORD pos)
{
   MSG msg;
   
   m_bStop = false;

   while(!m_bStop)
   {
      if(m_nMemBreaks)
      {
         for(int i=0;i<100;i++)
         {
            M6502CPU::Exec6502(&m_CPU);
            if(m_CPU.PC.W == pos || IsPCBreak(m_CPU.PC.W)) return;
            int i=0, len=m_nMemBreaks;
            for(;i<len;i++)
            {
               BPoint bp = *m_vctMemBP[i];
               if((bp.op == MB_CHANGE && ReadMem(bp.pos != bp.value)) ||
                  (bp.op == MB_EQUALS && ReadMem(bp.pos == bp.value)))
                     return;
            }
         }
      }
      else if(m_nPCBreaks)
      {
         for(int i=0;i<100;i++)
         {
            M6502CPU::Exec6502(&m_CPU);
            if(m_CPU.PC.W == pos || IsPCBreak(m_CPU.PC.W)) return;
         }
      }
      else
      {
         for(int i=0;i<100;i++)
         {
            M6502CPU::Exec6502(&m_CPU);
            if(m_CPU.PC.W == pos) return;
         }
      }

      if(::PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE))
         AfxGetApp()->PumpMessage();
   }
}

void CMyCpu::RunToReturn()
{
   MSG msg;
   m_bStop = false;
   
   while(!m_bStop)
   {
      if(m_nMemBreaks)
      {
         for(int i=0;i<100;i++)
         {
            byte op = ReadMem(m_CPU.PC.W);
            M6502CPU::Exec6502(&m_CPU);
            if(op == 0x60 || IsPCBreak(m_CPU.PC.W)) return;
            int i=0, len=m_nMemBreaks;
            for(;i<len;i++)
            {
               BPoint bp = *m_vctMemBP[i];
               if((bp.op == MB_CHANGE && ReadMem(bp.pos != bp.value)) ||
                  (bp.op == MB_EQUALS && ReadMem(bp.pos == bp.value)))
                     return;
            }
            if(op == 0x20) RunToReturn();
         }
      }
      else if (m_nPCBreaks)
      {
         for(int i=0;i<100;i++)
         {
            byte op = ReadMem(m_CPU.PC.W);
            M6502CPU::Exec6502(&m_CPU);
            if(op == 0x60 || IsPCBreak(m_CPU.PC.W)) return;
            if(op == 0x20) RunToReturn();
         }
      }
      else
      {          
         for(int i=0;i<100;i++)
         {
            byte op = ReadMem(m_CPU.PC.W);
            M6502CPU::Exec6502(&m_CPU);
            if(op == 0x60) return;
            if(op == 0x20) RunToReturn();
         }
      }

      if(::PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE))
         AfxGetApp()->PumpMessage();
   }
}

void CMyCpu::StepInto()
{
   M6502CPU::Exec6502(&m_CPU);
}

void CMyCpu::StepOver()
{
   if(ReadMem(m_CPU.PC.W) == 0x20) RunTo(m_CPU.PC.W+3);
   else M6502CPU::Exec6502(&m_CPU);
}

void CMyCpu::Stop()
{
   m_bStop = true;
}

int CMyCpu::Load(WORD pos, byte *data, int len)
{
   if(pos + len > 0xFFFF) return 1; // too long
   m_BaseAddr = pos, m_ProgLen = (WORD)len;
   memcpy((byte *)m_CPU.User+pos, data, len);
   Reset();
   return 0;
}

void CMyCpu::Reset()
{
   WriteMem(0xFFFC, m_BaseAddr & 0xFF);
   WriteMem(0xFFFD, m_BaseAddr >> 8);
   M6502CPU::Reset6502(&m_CPU);
   ClearPCBreaks();
   ClearMemBreaks();
}

CMyCpu::REGS &CMyCpu::GetRegs()
{
   static REGS r;
   r.A  = m_CPU.A;
   r.P  = m_CPU.P;
   r.PC = m_CPU.PC.W;
   r.S  = m_CPU.S;
   r.X  = m_CPU.X;
   r.Y  = m_CPU.Y;
   return r;
}

void CMyCpu::SetRegs(const REGS &regs)
{
   m_CPU.A = regs.A;
   m_CPU.P = regs.P;
   m_CPU.S = regs.S;
   m_CPU.X = regs.X;
   m_CPU.Y = regs.Y;
   m_CPU.PC.W = regs.PC;
}

WORD CMyCpu::GetBasePC()
{
   return m_BaseAddr;
}

WORD CMyCpu::GetProgLen()
{
   return m_ProgLen;
}

void CMyCpu::AddPCBreak(WORD pos)
{
   void *newbp;

   if(m_mapPCBP.Lookup(pos, newbp)) m_mapPCBP.RemoveKey(pos);
   else m_nPCBreaks++, newbp = (void *)new BPoint;
   
   ((BPoint *)newbp)->pos = pos;
   m_mapPCBP.SetAt(pos, newbp);
}   

void CMyCpu::AddMemBreak(WORD pos, byte value, MBType op)
{
   int i;
   BPoint *bp;

   if((i=IsMemBreak(pos)) != -1 && m_vctMemBP[i]->value == value &&
      m_vctMemBP[i]->op == op)
         bp = m_vctMemBP[i];
   else
   {
      bp = new BPoint;
      i = m_nMemBreaks++;
      if(i == m_vctMemBP.size()) m_vctMemBP.resize(m_nMemBreaks+4);

      m_vctMemBP[i] = bp;
   }
   
   bp->pos = pos, bp->value = value, bp->op = op;
}

void CMyCpu::RemPCBreak(WORD pos)
{
   void *dead;
   if(m_mapPCBP.Lookup(pos, dead))
   {
      delete (BPoint *)dead;
      m_mapPCBP.RemoveKey(pos);
      m_nPCBreaks--;
   }
}

void CMyCpu::RemMemBreak(WORD pos)
{
   int i = IsMemBreak(pos);
   if (i == -1) return;
   int len = --m_nMemBreaks;

   delete m_vctMemBP[i];
   m_vctMemBP[i] = m_vctMemBP[len];
}

BOOL CMyCpu::IsPCBreak(WORD pos)
{
   void *dummy;
   return m_mapPCBP.Lookup(pos, dummy);
}

int CMyCpu::IsMemBreak(WORD pos)
{
   int i, len = m_nMemBreaks;
   for(i=0;i<len;i++) if(m_vctMemBP[i]->pos == pos) return i;
   return -1;
}

void CMyCpu::ClearPCBreaks()
{
   POSITION pos = m_mapPCBP.GetStartPosition();
   WORD     key;
   void    *bp;

   while(pos)
   {
      m_mapPCBP.GetNextAssoc(pos, key, bp);
      delete (BPoint *)bp;
   }

   m_mapPCBP.RemoveAll();
   m_nPCBreaks = 0;
}

void CMyCpu::ClearMemBreaks()
{
   int i, len = m_nMemBreaks;
   for(i=0;i<len;i++) delete m_vctMemBP[i];
   m_vctMemBP.resize(0);
   m_nMemBreaks = 0;
}

CMapWordToPtr const & CMyCpu::GetPCBPList()
{
   return m_mapPCBP;
}

std::vector<CMyCpu::BPoint *> const & CMyCpu::GetMemBPList(int *len)
{
   *len = m_nMemBreaks;
   return m_vctMemBP;
}

CString const & CMyCpu::DisAsm(WORD *pos, bool sizeonly)
{
   static char *OpTable[256] =
   { "BRK", "ORA",   0  ,   0  ,   0  , "ORA", "ASL",   0  , "PHP", "ORA", "ASL",   0  ,   0  , "ORA", "ASL",   0  , // 00-0F
     "BPL", "ORA",   0  ,   0  ,   0  , "ORA", "ASL",   0  , "CLC", "ORA",   0  ,   0  ,   0  , "ORA", "ASL",   0  , // 10-1F
     "JSR", "AND",   0  ,   0  , "BIT", "AND", "ROL",   0  , "PLP", "AND", "ROL",   0  , "BIT", "AND", "ROL",   0  , // 20-2F
     "BMI", "AND",   0  ,   0  ,   0  , "AND", "ROL",   0  , "SEC", "AND",   0  ,   0  ,   0  , "AND", "ROL",   0  , // 30-3F
     "RTI", "EOR",   0  ,   0  ,   0  , "EOR", "LSR",   0  , "PHA", "EOR", "LSR",   0  , "JMP", "EOR", "LSR",   0  , // 40-4F
     "BVC", "EOR",   0  ,   0  ,   0  , "EOR", "LSR",   0  , "CLI", "EOR",   0  ,   0  ,   0  , "EOR", "LSR",   0  , // 50-5F
     "RTS", "ADC",   0  ,   0  ,   0  , "ADC", "ROR",   0  , "PLA", "ADC", "ROR",   0  , "JMP", "ADC", "ROR",   0  , // 60-6F
     "BVS", "ADC",   0  ,   0  ,   0  , "ADC", "ROR",   0  , "SEI", "ADC",   0  ,   0  ,   0  , "ADC", "ROR",   0  , // 70-7F

       0  , "STA",   0  ,   0  , "STY", "STA", "STX",   0  , "DEY", "STA", "TXA",   0  , "STY", "STA", "STX",   0  , // 80-8F
     "BCC", "STA",   0  ,   0  , "STY", "STA", "STX",   0  , "TYA", "STA", "TXS",   0  ,   0  , "STA",   0  ,   0  , // 90-9F
     "LDY", "LDA", "LDX",   0  , "LDY", "LDA", "LDX",   0  , "TAY", "LDA", "TAX",   0  , "LDY", "LDA", "LDX",   0  , // A0-AF
     "BCS", "LDA",   0  ,   0  , "LDY", "LDA", "LDX",   0  , "CLV", "LDA", "TSX",   0  , "LDY", "LDA", "LDX",   0  , // B0-BF
     "CPY", "CMP",   0  ,   0  , "CPY", "CMP", "DEC",   0  , "INY", "CMP", "DEX",   0  , "CPY", "CMP", "DEC",   0  , // C0-CF
     "BNE", "CMP",   0  ,   0  ,   0  , "CMP", "DEC",   0  , "CLD", "CMP",   0  ,   0  ,   0  , "CMP", "DEC",   0  , // D0-DF
     "CPX", "SBC",   0  ,   0  , "CPX", "SBC", "INC",   0  , "INX", "SBC", "NOP",   0  , "CPX", "SBC", "INC",   0  , // E0-EF
     "BEQ", "SBC",   0  ,   0  ,   0  , "SBC", "INC",   0  , "SED", "SBC",   0  ,   0  ,   0  , "SBC", "INC",   0    // F0-FF
   };
   enum { UNK=0, IMM, ZPG, ZPX, ZPY, ABS, ABX, ABY, INX, INY, ACC, REL, NON, IND };
   static byte ModeTable[256] =
   {  NON,   INX,    0 ,    0 ,    0 ,   ZPG,   ZPG,    0 ,    NON,   IMM,   ACC,    0 ,    0 ,   ABS,   ABS,    0 ,  // 00-0F
      REL,   INY,    0 ,    0 ,    0 ,   ZPX,   ZPX,    0 ,    NON,   ABY,    0 ,    0 ,    0 ,   ABX,   ABX,    0 ,  // 10-1F
      ABS,   INX,    0 ,    0 ,   ZPG,   ZPG,   ZPG,    0 ,    NON,   IMM,   ACC,    0 ,   ABS,   ABS,   ABS,    0 ,  // 20-2F
      REL,   INY,    0 ,    0 ,    0 ,   ZPX,   ZPX,    0 ,    NON,   ABY,    0 ,    0 ,    0 ,   ABX,   ABX,    0 ,  // 30-3F
      NON,   INX,    0 ,    0 ,    0 ,   ZPG,   ZPG,    0 ,    NON,   IMM,   ACC,    0 ,   ABS,   ABS,   ABS,    0 ,  // 40-4F
      REL,   INY,    0 ,    0 ,    0 ,   ZPX,   ZPX,    0 ,    NON,   ABY,    0 ,    0 ,    0 ,   ABX,   ABX,    0 ,  // 50-5F
      NON,   INX,    0 ,    0 ,    0 ,   ZPG,   ZPG,    0 ,    NON,   IMM,   ACC,    0 ,   IND,   ABS,   ABS,    0 ,  // 60-6F
      REL,   INY,    0 ,    0 ,    0 ,   ZPX,   ZPX,    0 ,    NON,   ABY,    0 ,    0 ,    0 ,   ABX,   ABX,    0 ,  // 70-7F

       0 ,   INX,    0 ,    0 ,   ZPG,   ZPG,   ZPG,    0 ,    NON,   IMM,   NON,    0 ,   ABS,   ABS,   ABS,    0 ,  // 80-8F
      REL,   INY,    0 ,    0 ,   ZPX,   ZPX,   ZPY,    0 ,    NON,   ABY,   NON,    0 ,    0 ,   ABX,    0 ,    0 ,  // 90-9F
      IMM,   INX,   IMM,    0 ,   ZPG,   ZPG,   ZPG,    0 ,    NON,   IMM,   NON,    0 ,   ABS,   ABS,   ABS,    0 ,  // A0-AF
      REL,   INY,    0 ,    0 ,   ZPX,   ZPX,   ZPY,    0 ,    NON,   ABY,   NON,    0 ,   ABX,   ABX,   ABY,    0 ,  // B0-BF
      IMM,   INX,    0 ,    0 ,   ZPG,   ZPG,   ZPG,    0 ,    NON,   IMM,   NON,    0 ,   ABS,   ABS,   ABS,    0 ,  // C0-CF
      REL,   INY,    0 ,    0 ,    0 ,   ZPX,   ZPX,    0 ,    NON,   ABY,    0 ,    0 ,    0 ,   ABX,   ABX,    0 ,  // D0-DF
      IMM,   INX,    0 ,    0 ,   ZPG,   ZPG,   ZPG,    0 ,    NON,   IMM,   NON,    0 ,   ABS,   ABS,   ABS,    0 ,  // E0-EF
      REL,   INY,    0 ,    0 ,    0 ,   ZPX,   ZPX,    0 ,    NON,   ABY,    0 ,    0 ,    0 ,   ABX,   ABX,    0    // F0-FF
   };
   static CString ret;

   CString temp, tmp2;

   byte opcode = SafeReadMem(*pos);
   byte   mode = ModeTable[opcode];

   byte v1, v2, bytes;
   (*pos)++;

   if(!sizeonly)
   {
      ret.Format("%0.2X ", opcode);
      if(!OpTable[opcode]) return ret += "        ???";
      bytes=1;
   }
   else if(!OpTable[opcode]) return ret;

   switch(mode)
   {
      case IMM:
      case INX:
      case INY:
      case REL:
      case ZPG:
      case ZPX:
      case ZPY:
         if(!sizeonly) temp.Format("%0.2X", v1 = SafeReadMem(*pos));
         (*pos)++;
         break;

      case IND:
      case ABS:
      case ABX:
      case ABY:
         if(!sizeonly)
         {
            v1 = SafeReadMem(*pos), v2 = SafeReadMem(*pos + 1);
            temp.Format("%0.4X", v2*256+v1);
            bytes = 2;
         }
         (*pos) += 2;
         break;

      case ACC:
         if(!sizeonly)
         {
            temp  = "A";
            bytes = 0;
         }
         break;

      case NON:
         if(!sizeonly)
         {
            temp  = "";
            bytes = 0;
         }
         break;

      case UNK:
      default:
         if(!sizeonly)
         {
            temp  = "??";
            bytes = 0;
         }
         break;
   }

   if(sizeonly) return ret;

   switch(mode)
   {
      case IMM:
         temp = '#' + temp;
         break;

      case ABX:
      case ZPX:
         temp += ",X";
         break;

      case ABY:
      case ZPY:
         temp += ",Y";
         break;

      case INX:
         temp = '(' + temp + ",X)";
         break;

      case INY:
         temp = '(' + temp + "),Y";
         break;

      case IND:
         temp = '(' + temp + ')';
         break;

      case REL:
         tmp2.Format(" -> %0.4X", (signed)(*pos) + (signed char)v1);
         temp += tmp2;
         break;
   }
            
   if(bytes--)
   {
      tmp2.Format("%0.2X ", v1);
      ret += tmp2;
      if(bytes)
      {
         tmp2.Format("%0.2X ", v2);
         ret += tmp2;
      }
      else ret += "   ";
   }
   else ret += "      ";
   
   return ret += CString("  ") + OpTable[opcode] + ' ' + temp;
}