// SourceDlg.cpp : implementation file
//

#include "stdafx.h"
#include "6502.h"
#include "6502Doc.h"
#include "6502View.h"
#include "SourceDlg.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CSourceDlg dialog

CSourceDlg::CSourceDlg(CWnd* pParent /*=NULL*/)
	: CMyDialog(CSourceDlg::IDD, pParent)
{
   m_Font.CreateStockObject(SYSTEM_FIXED_FONT);

   HDC hdc;
   hdc = ::GetDC(AfxGetMainWnd()->m_hWnd);

   CDC dc;
   dc.Attach(hdc);

   CFont *pOldFont = dc.SelectObject(&m_Font);

   TEXTMETRIC tm;
   dc.GetTextMetrics(&tm);

   m_cxChar = tm.tmAveCharWidth;
   m_cyChar = tm.tmHeight + tm.tmExternalLeading;
   
   dc.SelectObject(pOldFont);
   dc.Detach();

   ::ReleaseDC(AfxGetMainWnd()->m_hWnd, hdc);

   m_pbOpData = new byte[0xA000];
   m_bDoDumb  = false;
}

CSourceDlg::~CSourceDlg()
{
   delete[] m_pbOpData;
}

BEGIN_MESSAGE_MAP(CSourceDlg, CMyDialog)
	//{{AFX_MSG_MAP(CSourceDlg)
	ON_WM_ERASEBKGND()
	ON_WM_GETMINMAXINFO()
	ON_WM_SIZE()
	ON_WM_VSCROLL()
	ON_WM_PAINT()
   ON_WM_LBUTTONDOWN()
   ON_WM_KEYDOWN()
   ON_WM_CONTEXTMENU()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CSourceDlg overrides

void CSourceDlg::OnCancel()
{
   ((CFrameWnd *)AfxGetMainWnd())->GetActiveView()->
      PostMessage(WM_COMMAND, ID_SOURCE_CLOSED);
   DestroyWindow();
}

BOOL CSourceDlg::PreTranslateMessage(MSG *pMsg)
{
   return CMyDialog::PreTranslateMessage(pMsg);
}

/////////////////////////////////////////////////////////////////////////////
// CSourceDlg commands

WORD CSourceDlg::GetCursorPos()
{
   ScrollToCursor();
   return m_nCursor;
}

void CSourceDlg::SetCursorPos(WORD pos)
{
   m_nCursor = pos;
   ScrollToCursor();
   Update(UC_REPAINT);
}

void CSourceDlg::SetCurrentPC(WORD pos)
{
   m_nRealPC = pos;
   Update(UC_REPAINT);
}

void CSourceDlg::Update(LPARAM lHint)
{
   switch(lHint)
   {
      case UC_REPAINT:
         {
            int pc = ((C6502Doc *)((CFrameWnd *)AfxGetMainWnd())->
                        GetActiveDocument())->GetPC();
            if(pc != m_nRealPC) m_nRealPC = pc, ScrollToPC();
            Invalidate();
         }
         break;

      case UC_RESET:
         ResetSource();
         Update(UC_REPAINT);
         break;

      case UC_CLOSE:
         OnCancel();
         break;
   }
}

void CSourceDlg::ResetSource()
{
   C6502Doc *Doc = (C6502Doc *)((CFrameWnd *)AfxGetMainWnd())->GetActiveDocument();
   CClientDC dc(this);

   ::InvalidateRect(m_hWnd, NULL, false);
   OnEraseBkgnd(&dc);
   dc.TextOut(0, 2, CString("Disassembling..."));

   m_wCodeArr.SetSize(0, 100);
   m_nSBEnd = 65535;
   
   WORD init;

   memset(m_pbOpData, 0x55, 0x8000);
   memset(m_pbOpData+0x8000, 0, 0x2000);
   m_pbMem = Doc->GetMemory();
   
   init = Doc->GetPC();
   DoDisAsm(Doc, Doc->GetInitAddr());
   DoDisAsm(Doc, Doc->GetPlayAddr());
   CombineRanges();

   CRect rect;
   GetClientRect(rect);
   DoResize(rect.right, rect.bottom);

   m_nPos = 0;
   WORD pc = 0;
   while(pc != init) pc += ToNextOp(pc), m_nPos++;

   SetScrollPos(SB_VERT, m_nPos);
   m_nProgTop = m_nPosPC = m_nCursor = pc;
}

void CSourceDlg::DoDisAsm(C6502Doc *Doc, WORD pos)
{
   if(InCodeArr(pos) != -1) return;
   WORD newpos=pos;
   
   while(true)
   {
      CString s=Doc->DisAsm(&newpos);//, true);
      if(newpos < pos) return;
      else AddToCodeArr(pos,newpos);

      if(IsBranch(pos))
         DoDisAsm(Doc, (signed)pos+2 + (signed char)m_pbMem[pos+1]);
      else if(IsJump(pos))
         DoDisAsm(Doc, m_pbMem[pos+1] + m_pbMem[pos+2]*256);

      if(CantContinue(pos)) return;

      pos=newpos;
   }
}

int CSourceDlg::InCodeArr(WORD pos)
{
   int i, len = m_wCodeArr.GetSize();
   for(i=0;i<len;i+=2)
      if(IsInRange(pos, m_wCodeArr[i], m_wCodeArr[i+1]-1)) return i;
   return -1;
}

void CSourceDlg::AddToCodeArr(WORD start, WORD end)
{
   int i, len = m_wCodeArr.GetSize();

   SetValid(start);
   SetToNextOp(start, end-start);
   SetToPrevOp(end,   end-start);
   m_nSBEnd -= (end-start)-1;

   for(i=0;i<len;i+=2)
   {
      WORD v1 = m_wCodeArr[i], v2 = m_wCodeArr[i+1];
      if(IsInRange(start, v1, v2))
      {
         if(end > v2) m_wCodeArr[i+1] = end;
         return;
      }
      else if(IsInRange(end, v1, v2))
      {
         if(start < v1) m_wCodeArr[i] = start;
         return;
      }
   }
   // cant expand any ranges
   m_wCodeArr.SetSize(len+2, 100);
   m_wCodeArr[len] = start, m_wCodeArr[len+1] = end;
}

void CSourceDlg::CombineRanges()
{
   int i, len = m_wCodeArr.GetSize()-2;

   for(i=0;i<len;i+=2)
   {
      WORD v2, v3,v4;
      if((v2=m_wCodeArr[i+1]) == (v3=m_wCodeArr[i+2]))
      {
         v4 = m_wCodeArr[i+3];
         m_wCodeArr[i+1] = v4;
         for(int j=i+2;j<len;j++) m_wCodeArr[j] = m_wCodeArr[j+2];
         m_wCodeArr.SetSize(len, 100);
         len -= 2;
      }
   }
}

void CSourceDlg::DoResize(int cx, int cy)
{
   SCROLLINFO si;
   si.cbSize = sizeof(si);
   si.fMask  = SIF_PAGE | SIF_RANGE;
   si.nPage  = m_cyPage = cy / m_cyChar;
   si.nMin   = 0;
   si.nMax   = m_nSBEnd;
   if(si.nMax < 0) si.nMax = 0;
   if(m_nPos > si.nMax) m_nPos = si.nMax;
   SetScrollInfo(SB_VERT, &si);
}

int CSourceDlg::AdvanceNLines(int numlines)
{
   if(numlines >= 0)
   {
      while(numlines--) m_nPosPC += ToNextOp(m_nPosPC);
   }
   else // numlines cant be 0
   {
      while(numlines++) m_nPosPC -= ToPrevOp(m_nPosPC);
   }
   return m_nPosPC=(WORD)m_nPosPC;
}

void CSourceDlg::ScrollToCursor()
{
   WORD pc = m_nCursor;
   int page = m_cyPage / 3 + 1;
   if(m_nPosPC < pc)
      while(pc-m_nPosPC > page) m_nPosPC += ToNextOp(m_nPosPC), m_nPos++;
   else if(m_nPosPC > pc)
      while(m_nPosPC-pc > -page) m_nPosPC -= ToPrevOp(m_nPosPC), m_nPos--;
   SetScrollPos(SB_VERT, m_nPos);
}

void CSourceDlg::ScrollToPC()
{
   WORD pc = m_nRealPC;
   int page = m_cyPage / 3 + 1;
   if(m_nPosPC < pc)
      while(pc-m_nPosPC > page) m_nPosPC += ToNextOp(m_nPosPC), m_nPos++;
   else if(m_nPosPC > pc)
      while(m_nPosPC-pc > -page) m_nPosPC -= ToPrevOp(m_nPosPC), m_nPos--;
   SetScrollPos(SB_VERT, m_nPos);
}

/////////////////////////////////////////////////////////////////////////////
// CSourceDlg message handlers

BOOL CSourceDlg::OnEraseBkgnd(CDC* pDC) 
{
   CRect rect;
   pDC->GetClipBox(rect);

   CBrush brush;
   brush.CreateStockObject(WHITE_BRUSH);

   pDC->FillRect(rect, &brush);
   return TRUE;
}

void CSourceDlg::OnGetMinMaxInfo(MINMAXINFO FAR* lpMMI) 
{
	CDialog::OnGetMinMaxInfo(lpMMI);
   int xwid = m_cxChar * 36;

   lpMMI->ptMinTrackSize.x = lpMMI->ptMaxTrackSize.x = lpMMI->ptMaxSize.x = xwid;
   lpMMI->ptMaxSize.y = GetSystemMetrics(SM_CYSCREEN) / 2;
}

BOOL CSourceDlg::OnInitDialog() 
{
	CDialog::OnInitDialog();

   SetScrollPos(SB_VERT, m_nPos = 0);
   ShowWindow(SW_SHOW);

	return TRUE;
}

void CSourceDlg::OnSize(UINT nType, int cx, int cy) 
{
   CDialog::OnSize(nType, cx, cy);
   DoResize(cx, cy);
}

void CSourceDlg::OnVScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
	CDialog::OnVScroll(nSBCode, nPos, pScrollBar);

   int nDelta=0;

   switch(nSBCode)
   {
      case SB_LINEUP:   nDelta--; break;
      case SB_LINEDOWN: nDelta++; break;
      case SB_PAGEUP:   nDelta = -m_cyPage; break;
      case SB_PAGEDOWN: nDelta =  m_cyPage; break;

      case SB_THUMBTRACK:
      case SB_THUMBPOSITION:
         nDelta = (WORD)nPos-m_nPos; break;
   }

   if(m_nPos + nDelta > m_nSBEnd) nDelta = m_nSBEnd-m_nPos;
   if(m_nPos + nDelta < 0) nDelta = -m_nPos;
   if(nDelta)
   {
      if(m_bDoDumb) Invalidate();
      else ScrollWindow(0, -nDelta * m_cyChar);
      m_nPos += nDelta;
      SetScrollPos(SB_VERT, AdvanceNLines(nDelta));
   }
}

void CSourceDlg::OnPaint() 
{
	CPaintDC dc(this); // device context for painting
   C6502Doc *Doc = (C6502Doc *)((CFrameWnd *)AfxGetMainWnd())->GetActiveDocument();

   CFont *pOldFont = dc.SelectObject(&m_Font);

   CRect rect;
   GetClientRect(rect);
   
   m_pbMem    = Doc->GetMemory();
   WORD start = 0;
   WORD end   = rect.bottom / m_cyChar+1;
   WORD pc    = m_nPosPC;
   WORD newpc = pc;
   int  offs  = m_nDumbOffset;

   bool reset = false, dobyte = false, unknown = false;

   int y     = 0;

   while(start < end)
   {
      CString line;
      line.Format(" %0.4X: ", pc);

      if(IsValid(pc)) line += Doc->DisAsm(&newpc);
      else
      {
         if(m_bDoDumb)
         {
            if(offs) offs--,dobyte=true;
            else
            {
               line  += Doc->DisAsm(&newpc);
               unknown = true;
            }
         }
         else dobyte=true;

         if(dobyte)
         {
            CString tmp;
            tmp.Format("%0.2X", m_pbMem[pc]);
            line  += tmp;
            newpc  = pc+1;
            dobyte = false;
         }
      }
                        
      if(reset)
         dc.SetTextColor(RGB(0,0,0)), dc.SetBkColor(RGB(255,255,255));

      if(pc == m_nRealPC)
         reset = true, dc.SetTextColor(RGB(255,255,255)), dc.SetBkColor(RGB(0,0,255));
      else if(pc == m_nCursor)
         reset = true, dc.SetTextColor(RGB(255,255,255)), dc.SetBkColor(RGB(255,0,0));
      else if(unknown)
         reset=true, unknown=false, dc.SetTextColor(RGB(0,128,48));

      if(Doc->IsPCBreak(pc)) line.SetAt(0, _T('*'));
      else if(Doc->IsMemBreak(pc)) line.SetAt(0, _T('M'));
      else if(m_nRealPC == pc) line.SetAt(0, _T('-'));
      dc.TextOut(0, y, line);
      y += m_cyChar;
      
      if(newpc < pc) break;
      pc=newpc;
      start++;
   }

   dc.SelectObject(pOldFont);
}

void CSourceDlg::OnLButtonDown(UINT nFlags, CPoint point)
{
   int  line  = point.y / m_cyChar;
   WORD pc    = m_nPosPC;
   while(line--) pc += ToNextOp(pc);
   m_nCursor  = pc;
   Invalidate(FALSE);
}

void CSourceDlg::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags)
{
   if(!CheckCursorMove(nChar, nRepCnt, nFlags)) return;
   if(!CheckDumbCtrls (nChar, nRepCnt, nFlags)) return;
   CDialog::OnKeyDown(nChar, nRepCnt, nFlags);
}

int CSourceDlg::CheckCursorMove(UINT nChar, UINT nRepCnt, UINT nFlags)
{
   WORD pc = m_nCursor;
   int  page = m_cyPage;

   switch(nChar)
   {
      case VK_UP:   pc -= ToPrevOp(pc); if(pc > m_nCursor) return 0; break;
      case VK_DOWN: pc += ToNextOp(pc); if(pc < m_nCursor) return 0; break;
      case VK_PRIOR:
         while(page--)
         {
            pc -= ToPrevOp(pc);
            if(pc > m_nCursor) { pc += ToNextOp(pc); break; }
         }
         break;
      case VK_NEXT:
         while(page--)
         {
            pc += ToNextOp(pc);
            if(pc < m_nCursor) { pc += ToPrevOp(pc); break; }
         }
         break;
      case VK_HOME:
         if(m_nPosPC != m_nProgTop)
         {
            m_nPos = pc = 0;
            WORD goal = m_nProgTop;
            while(pc != goal) pc += ToNextOp(pc), m_nPos++;
            m_nPosPC = pc;
         }
         break;

      default:
         return 1;
   }

   if(m_nCursor != pc)
   {
      m_nCursor = pc;
      ScrollToCursor();
   }
   Invalidate();
   return 0;
}

int CSourceDlg::CheckDumbCtrls (UINT nChar, UINT nRepCnt, UINT nFlags)
{
   switch(nChar)
   {
      case VK_MULTIPLY:
         m_bDoDumb = !m_bDoDumb;
         m_nDumbOffset = 0;
         if(m_bDoDumb)
         {
            CString txt;
            GetWindowText(txt);
            txt += " - Dumb On";
            SetWindowText(txt);
         }
         else SetWindowText(_T("Source Display"));
         break;
      
      case VK_ADD:
         if (m_bDoDumb) m_nDumbOffset++;
         else return 0;
         break;

      case VK_SUBTRACT:
         if(m_bDoDumb && m_nDumbOffset) m_nDumbOffset--;
         else return 0;
         break;

      case VK_DIVIDE:
         if(m_bDoDumb && m_nDumbOffset) m_nDumbOffset = 0;
         else return 0;
         break;

      default: return 1;
   }
   Invalidate();
   return 0;
}

void CSourceDlg::OnContextMenu(CWnd *pWnd, CPoint pos)
{
   CPoint point = pos;
   ScreenToClient(&point);
   OnLButtonDown(0, point);
   
   CMenu menu;
   menu.CreatePopupMenu();
   menu.AppendMenu(MF_STRING | (IsValid(m_nCursor) ? MF_GRAYED : 0),
                   1, _T("&Disassemble"));
   menu.AppendMenu(MF_SEPARATOR);
   menu.AppendMenu(MF_STRING, 2, _T("Reset Disassembly"));

   int nCmd = menu.TrackPopupMenu(TPM_LEFTALIGN | TPM_LEFTBUTTON | TPM_RIGHTBUTTON |
                                  TPM_RETURNCMD | TPM_NONOTIFY,
                                  pos.x, pos.y, this);

   C6502Doc *Doc = (C6502Doc *)((CFrameWnd *)AfxGetMainWnd())->GetActiveDocument();
   CRect rect;
   switch(nCmd)
   {
      case 1:
         DoDisAsm(Doc, m_nCursor);
         CombineRanges();

         GetClientRect(rect);
         DoResize(rect.right, rect.bottom);
         Invalidate();
         break;

      case 2:
         Update(UC_RESET);
         break;
   }
}