/*
This software was written to control a battery charger.

http://www.adammil.net/
Copyright (C) 2009 Adam Milazzo

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#define AM_BATTERY_INTERNAL
#include "battery.h"
#include "delay.h"
#include "definitions.h"
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <util/delay.h>

// TODO: measure and compensate for ADC offset error
static const double MaxVoltage = 5.04; // measured output from regulator
static const double VoltageDivider = 0.5; // measured factor from voltage divider
static const double B0 = 1.2, B1 = 3.6, B2 = 6, B3 = 7.2; // supported battery voltages
static const byte MaxPower = 255;

#define PVOLTAGE(nv,p,usingDivider) (uint16)((nv) * (p) * (usingDivider ? VoltageDivider : 1) / MaxVoltage * 1023)

#define BATTERY_VOLTAGE(batteryType, p) \
  ((batteryType) & 1 ? ((batteryType) & 2 ? PVOLTAGE(B3, p, true) : PVOLTAGE(B1, p, false)) \
                     : ((batteryType) & 2 ? PVOLTAGE(B2, p, true) : PVOLTAGE(B0, p, false)))


// charge at a the given rate until the maximum voltage level is reached, a
// timeout expires, or the voltage level begins dropping
static byte charge(byte powerLevel, uint16 timeout, bool useDeltaV) __attribute__((noinline));

// detects the presence of a battery, given a reading from readBatteryVoltage()
static bool _detectBattery(uint16 voltage) __attribute__((noinline));

// sets the power level being delivered to the battery as a value from 0-255
// which indicates a fraction of the maximum charge rate C: C * value / 255
static void setPowerLevel(byte level) __attribute__((noinline));

// gets the battery type selected by the user
static byte getBatteryType() __attribute__((noinline));

static void toggleLed();
static void turnLedOn();
static void turnLedOff();

bool detectBattery()
{
  return _detectBattery(readBatteryVoltage());
}

byte getBatteryType()
{
  return (~PINB & (BIT(PIN1) | BIT(PIN2))) >> 1;
}

bool _detectBattery(uint16 voltage)
{
  // detect the battery between 90% and 125% of its nominal voltage
  byte batteryType = getBatteryType();
  uint16 threshold = BATTERY_VOLTAGE(batteryType, 0.9);
  if(voltage < threshold) return false;
  threshold = BATTERY_VOLTAGE(batteryType, 1.25);
  return voltage <= threshold;
}

byte fastCharge()
{
  // TODO: enable deltaV when we have a good power source
  return charge(MaxPower, 5*HOUR/2, false); // charge at a 100% rate for no more than 2.5 hours
}

byte finalCharge()
{
  powerDownDelay2secs();
  return charge((MaxPower+1)/3, HOUR*3/4, false); // TODO: enable deltaV when we have a good power source
}

void selfTest()
{
  #define VOLTAGE(v) (uint16)((v) / MaxVoltage * 1023)
  // runs a simple self test by checking the voltage levels of the charger using the ADC
  uint16 voltage = readBatteryVoltage();
  // when powered off, the charging circuit should output 0.75 volts, or 0.375 through the voltage divider
  if(voltage < VOLTAGE(0.35) || voltage > VOLTAGE(0.8))
  {
    while(true) // flash LED and never return
    {
      toggleLed();
      powerDownDelay100ms();
    }
  }
  #undef VOLTAGE

  // the test was successful, so flash the light briefly to signal readiness
  turnLedOn();
  powerDownDelay100ms();
  powerDownDelay100ms();
  turnLedOff();
}

byte slowCharge()
{
  // charge at a 20% rate in the beginning with no delta-V check
  return charge((MaxPower+2)/5, 3 * MINUTE, false);
}

bool stabilize()
{
  // wait 30 seconds for the voltage to stabilize, or until the voltage stops
  // changing, whichever is greater, until a maximum of 3 minutes
  uint16 lastVoltage = 0;
  bool continueWithCharge = true;
  for(byte i=0; i < 3*MINUTE; i++)
  {
    // flash the LED slowly during stabilization
    toggleLed();

    lowPowerDelaySecs(1, SLEEP_MODE_PWR_DOWN);

    uint16 voltage = readBatteryVoltage();
    if(!_detectBattery(voltage)) // if the battery was removed, abort
    {
      continueWithCharge = false;
      break;
    }

    // if it's been 30 seconds, and the voltage hasn't changed in the last second
    if(i >= 30 && voltage == lastVoltage) break;
    lastVoltage = voltage;
  }

  turnLedOff();
  return continueWithCharge;
}

void waitForBattery()
{
  while(!detectBattery()) powerDownDelay2secs();
}

void waitUntilBatteryRemoved()
{
  while(true) // flash the LED in short blips until the battery is removed
  {
    powerDownDelay2secs();
    if(!detectBattery()) break; // break here to prevent an additional flash after the battery is removed
    turnLedOn();
    powerDownDelay100ms();
    turnLedOff();
  }
}

byte charge(byte powerLevel, uint16 timeout, bool useDeltaV)
{
  turnLedOn(); // light the LED solidly while charging is ongoing

  uint16 lastVoltage = 0;
  byte result = VoltageOrTimeLimitReached;
  while(true)
  {
    uint16 voltage = readBatteryVoltage();
    if(!detectBattery(voltage))
    {
      result = false;
      break;
    }

    // charge to 16% above nominal voltage (just under 1.4V per cell)
    byte batteryType = getBatteryType();
    uint16 maxVoltage = BATTERY_VOLTAGE(batteryType, 1.16);

    // TODO: FIXME: deltaV is currently disabled in fastCharge() because the voltage tends to
    // fluctuate +/- 20mV, which causes an early abort. perhaps this is caused by the power
    // source being unregulated?
    if(voltage >= maxVoltage || !timeout--) break;

    if(useDeltaV && voltage < lastVoltage)
    {
      result = DeltaVDetected;
      break;
    }

    lastVoltage = voltage;

    setPowerLevel(powerLevel);
    lowPowerDelaySecs(1, ~powerLevel ? SLEEP_MODE_IDLE : SLEEP_MODE_PWR_DOWN); // ~powerLevel is equivalent to powerLevel != 255
    setPowerLevel(0);
    lowPowerDelayMs(25, SLEEP_MODE_PWR_DOWN); // let the battery rest for 25 ms after each second (pulsed charging)
  }

  turnLedOff();
  return result;
}

// AVR-LibC gets the value of PRTIM0 wrong for the ATtiny13A. This was reported as bug #2898745:
// http://sourceforge.net/tracker/?func=detail&aid=2898745&group_id=68108&atid=520074
#undef PRTIM0
#define PRTIM0 1

static volatile bool _timerStopping;

ISR(TIM0_OVF_vect)
{
  if(_timerStopping)
  {
    TCCR0B = 0; // stop timer
    PRR   |= BIT(PRTIM0); // disable timer module
    _timerStopping = false;
  }
}

void setPowerLevel(byte level)
{
  bool isRunning = TCCR0B & (BIT(CS00) | BIT(CS01) | BIT(CS02));

  if(level == 0 || !~level) // equivalent to level == 0 || level == 255
  {
    if(isRunning)
    {
      _timerStopping = true; // wait for timer to finish its current cycle
      while(_timerStopping) { }
    }

    if(level == 0) DDRB &= ~BIT(DDB0); // tri-state the output pin (ie, turn off the power)
    else DDRB |= BIT(DDB0); // set the OC0A to be a sink (ie, turn on the power)
  }
  else
  {
    if(!isRunning)
    {
      PRR    &= ~BIT(PRTIM0); // enable timer module
      TCCR0A  = BIT(COM0A1) | BIT(COM0A0) | BIT(WGM01) | BIT(WGM00); // set OC0A on match, clear at top; use fast PCM mode w/ top==0xFF
      TIMSK0  = BIT(TOIE0); // enable interrupt on overflow
      DDRB   |= BIT(DDB0); // set OC0A to be an output pin. the timer will toggle it between sink (on) and source (off)
      TCCR0B  = BIT(CS02) | BIT(CS00); // start the timer w/ 1024 clock divider
    }  

    TCNT0 = 0;
    OCR0A = level;
  }

  PORTB &= ~BIT(PORTB0); // set the output pin to be a sink (when power is on) or tri-stated (when power is off)
}

void turnLedOn()
{
  PORTB |= BIT(PORTB3);
}

void turnLedOff()
{
  PORTB &= ~BIT(PORTB3);
}

void toggleLed()
{
  PINB |= BIT(PORTB3);
}
