/*
This software was written to control a battery charger.

http://www.adammil.net/
Copyright (C) 2009 Adam Milazzo

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#define AM_BATTERY_BUILDING
#include "battery.h"
#include "definitions.h"
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>

static volatile uint16 _measurement;
static byte _measurementCount;

// called when the ADC has completed a measurement
ISR(ADC_vect)
{
  if(++_measurementCount > 2) // ignore the first two measurements
  {
    uint16 newMeasurement = _measurement + ADCW; // use a temporary to prevent saves/loads every time
    if(_measurementCount == 6) // after four measurements, finish
    {
      newMeasurement = (newMeasurement+2) >> 2; // divide by 4 (with rounding) to produce the final measurement
      ADCSRA = 0; // stop and disable the ADC
    }
    _measurement = newMeasurement;
  }

  ADCSRA |= BIT(ADSC); // trigger another measurement. TODO: figure out how to use free running mode
}

uint16 readBatteryVoltage()
{
  // TODO: handle the case where interrupts are disabled
  // configure the ADC
  PRR   &= ~BIT(PRADC); // enable ADC module
  DIDR0 &= ~BIT(ADC2D); // enable ADC2 pin
  ADMUX  = BIT(MUX1); // use ADC2 (PB4), Vcc as the reference voltage, and right alignment
  ADCSRA = BIT(ADIE) | BIT(ADPS0) | BIT(ADPS1); // enable ADC interrupt, use a 8x clock divider, and disable auto trigger

  // clear our measurement data
  _measurement      = 0;
  _measurementCount = 0;

  ADCSRA |= BIT(ADEN); // enable ADC

  // prepare to go to sleep (so we can gain greater ADC accuracy)
  set_sleep_mode(SLEEP_MODE_ADC);
  sleep_enable();

  ADCSRA |= BIT(ADSC); // start measuring
  while(ADCSRA & BIT(ADEN)) sleep_cpu(); // wait until the ADC is done

  DIDR0 |= BIT(ADC2D); // disable ADC2 pin
  PRR   |= BIT(PRADC); // disable ADC module

  sleep_disable(); // disable sleep mode

  return _measurement;
}

