/*
This software was written to control a battery charger.

http://www.adammil.net/
Copyright (C) 2009 Adam Milazzo

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "delay.h"
#include "definitions.h"
#include "watchdog.h"
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <util/delay.h>

static volatile byte watchdogTimer;

// delays for the given number of milliseconds. this method should be used
// instead of _delay_ms when the delay time is not a compile-time constant
// this has 6 cycles of overhead per ms, causing the delay to be
// 4.7% longer at 128khz, 0.6% longer at 1mhz, etc.
void dynamicDelayMs(byte ms)
{
  const uint16 ticksPerMs = F_CPU / 4000;

  if(ms)
  {
    uint16 ticks;
    asm volatile
    (
      NL
      "L_1%=:"      NLT
      "ldi %A0, %2" NLT
      "ldi %B0, %3" NL
      "L_2%=:"      NLT
      "sbiw %A0, 1" NLT
      "brne L_2%="  NLT
      "subi %1, 1"  NLT
      "brne L_1%="  NLT
      : "=w" (ticks), "+r" (ms)
      : "M" (ticksPerMs % 256), "M" (ticksPerMs / 256)
    );
  }
}

// delays for the given number of seconds. this method should be used
// instead of _delay_ms when the delay time is not a compile-time constant.
/*void dynamicDelaySecs(byte secs)
{
  while(secs--) _delay_ms(1000);
}*/

ISR(WDT_vect)
{
  byte timerValue = watchdogTimer; // use a temp to prevent unnecessary saves/loads
  if(timerValue)
  {
    watchdogTimer = timerValue-1;
    resetWatchdog();
  }
  else
  {
    // disableWatchdog(); this causes tons of pushes and pops, so we'll inline it
    __asm__ __volatile__
    (
      "out %0, %1" NLT
      "out %0, __zero_reg__" NLT
      "wdr"
      : /* no outputs */
      : "I" (_SFR_IO_ADDR(WDTCR)),
        "r" (BIT(WDCE) | BIT(WDE))
    );
  }
}

void lowPowerDelay(byte timer, byte tickSize, byte sleepMode)
{
  set_sleep_mode(sleepMode);
  sleep_enable();

  watchdogTimer = timer;
  enableWatchdogInterrupt(tickSize);

  while(watchdogTimer) sleep_cpu();

  sleep_disable();
}

// delays in a low-power mode for the given number of milliseconds.
void lowPowerDelayMs(byte ms, byte sleepMode)
{
  for(; ms >= 16; ms -= 16) lowPowerDelay(1, WDTO_15MS, sleepMode); // it's actually 16ms
  dynamicDelayMs(ms); // use a standard delay for the remainder
}

// delays in a low-power mode for the given number of seconds.
void lowPowerDelaySecs(byte secs, byte sleepMode)
{
  lowPowerDelay(secs, WDTO_1S, sleepMode);
}

void powerDownDelay100ms()
{
  lowPowerDelayMs(100, SLEEP_MODE_PWR_DOWN);
}

void powerDownDelay2secs()
{
  lowPowerDelaySecs(2, SLEEP_MODE_PWR_DOWN);
}
