/*
This software was written to control a battery charger.

http://www.adammil.net/
Copyright (C) 2009 Adam Milazzo

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#ifndef AM_SERIAL_H
#define AM_SERIAL_H

#include "types.h"

#ifndef F_CPU
  #warning F_CPU is not defined. Defaulting to 1.2mhz
  #define F_CPU 1200000
#endif

#ifndef BAUD
  #warning BAUD is not defined. Defaulting to 9600
  #define BAUD 9600
#endif

// we have overhead of 15 cycles per bit, which is equal to 3.75 (~4)
// iterations of the delay loop, which take 4 cycles each.
#define _SERIAL_DELAY (((F_CPU + BAUD/2)/BAUD - 15 + 2) / 4)

#if _SERIAL_DELAY <= 0
  #warning The baud rate is too high for this CPU speed.
#elif _SERIAL_DELAY >= 65536
  #if _SERIAL_DELAY > 65536
    #warning The baud rate is too low for this CPU speed.
  #endif
  #define _SERIAL_DELAY 0 // a delay of zero is equal to 65536
#endif

void _serialSendChar(char c, byte port, uint16 delay);
void _serialSendString(char *str, byte port, uint16 delay);

// sets up a port for serial communication
void serialSetupPort(byte port);

// sends one character over a serial port. the port must have previously been
// set up with serialSetupPort()
static inline void serialSendChar(char c, byte port)
{
  _serialSendChar(c, port, _SERIAL_DELAY);
}

// sends a null-terminated string over a serial line. the port must have
// previously been set up with serialSetupPort()
static inline void serialSendString(char *str, byte port)
{
  _serialSendString(str, port, _SERIAL_DELAY);
}

#undef _SERIAL_DELAY

#endif
