using System;

namespace BuLang
{

#region Parameter
public struct Parameter
{
  public Parameter(string name, Node defaultValue)
  {
    Name = name;
    Default = defaultValue;
  }

  public readonly string Name;
  public readonly Node Default;
}
#endregion

#region Node
public abstract class Node
{
  public abstract object Evaluate();
  
  public double ToNumber()
  {
    object value = Evaluate();
    if(value is double) return (double)value;
    else
    {
      try { return Convert.ToDouble(value); }
      catch
      {
        throw new ScriptException("Expected a number, but found '"+ToString()+"'");
      }
    }
  }
}
#endregion

#region AssignNode
public sealed class AssignNode : Node
{
  public AssignNode(string name, Node value)
  {
    if(name == null || value == null) throw new ArgumentNullException();
    Name  = name;
    Value = value;
  }
  
  public readonly string Name;
  public readonly Node Value;

  public override object Evaluate()
  {
    object value = Value.Evaluate();
    LocalEnvironment.Current.Set(Name, value);
    return value;
  }

  public override string ToString()
  {
    string valueString = Value.ToString();
    return Value is FunctionNode ? valueString : Name+" = "+valueString;
  }
}
#endregion

#region BlockNode
public sealed class BlockNode : Node
{
  public BlockNode(Node[] nodes)
  {
    if(nodes == null) throw new ArgumentNullException();
    Nodes = nodes;
  }

  public override object Evaluate()
  {
    object value = null;
    foreach(Node node in Nodes) value = node.Evaluate();
    return value;
  }

  readonly Node[] Nodes;
}
#endregion

#region CallNode
public sealed class CallNode : Node
{
  public CallNode(Node function, Node[] args)
  {
    Function  = function;
    Arguments = args;
  }

  public readonly Node Function;
  public readonly Node[] Arguments;

  public override object Evaluate()
  {
    object funcValue = Function.Evaluate();
    Function scriptFunc = funcValue as Function;
    if(scriptFunc != null)
    {
      object[] values = new object[scriptFunc.Parameters.Length];
      for(int i=0,len=Math.Min(Arguments.Length,values.Length); i<len; i++)
      {
        values[i] = Arguments[i].Evaluate();
      }
      for(int i=0; i<values.Length; i++)
      {
        if(values[i] == null && scriptFunc.Parameters[i].Default != null)
        {
          values[i] = scriptFunc.Parameters[i].Default.Evaluate();
        }
      }
      return scriptFunc.Call(values);
    }
    
    IFunction func = funcValue as IFunction;
    if(func != null)
    {
      object[] values = new object[Arguments.Length];
      for(int i=0; i<values.Length; i++) values[i] = Arguments[i].Evaluate();
      return func.Call(values);
    }
    
    throw new ScriptException("Tried to convert '"+Function.ToString()+"' to a function, but couldn't");
  }

  public override string ToString()
  {
    return Function.ToString() + "()";
  }
}
#endregion

#region IdentifierNode
public sealed class IdentifierNode : Node
{
  public IdentifierNode(string name)
  {
    Name = name;
  }
  
  public readonly string Name;

  public override object Evaluate()
  {
    return LocalEnvironment.Current.Get(Name);
  }

  public override string ToString()
  {
    return Name;
  }
}
#endregion

#region IfNode
public sealed class IfNode : Node
{
  public IfNode(Node condition, Node ifTrue, Node ifFalse)
  {
    Condition = condition;
    IfTrue = ifTrue;
    IfFalse = ifFalse;
  }

  public readonly Node Condition, IfTrue, IfFalse;

  public override object Evaluate()
  {
    return Ops.IsTrue(Condition.Evaluate()) ? IfTrue.Evaluate() : (IfFalse == null ? null : IfFalse.Evaluate());
  }

  public override string ToString()
  {
    return "if";
  }
}
#endregion

#region LiteralNode
public sealed class LiteralNode : Node
{
  public LiteralNode(object value)
  {
    this.Value = value;
  }

  public override object Evaluate()
  {
    return Value;
  }

  public override string ToString()
  {
    return Ops.ToString(Value);
  }

  public readonly object Value;
}
#endregion

#region LoopNode
public abstract class LoopNode : Node
{
  public LoopNode(string loopVar, Node start, Node end, Node step)
  {
    LoopVar = loopVar;
    Start = start;
    End   = end;
    Step  = step;
  }

  public readonly string LoopVar;
  public readonly Node Start, End, Step;

  public override object Evaluate()
  {
    LocalEnvironment env = LocalEnvironment.PushNew();
    try
    {
      double start = Start.ToNumber(), end = End.ToNumber(), step;
      if(Step != null) step = Step.ToNumber();
      else step = start <= end ? 1 : -1;

      if(step == 0) throw new ScriptException("You can't have a loop (for/sigma) with a step size of zero.");

      Initialize();
      if(step >= 0)
      {
        while(start <= end)
        {
          env.Bind(LoopVar, start);
          EvaluateOnce();
          start += step;
        }
      }
      else
      {
        while(start >= end)
        {
          env.Bind(LoopVar, start);
          EvaluateOnce();
          start += step;
        }
      }
      return Finish();
    }
    finally { LocalEnvironment.Pop(); }
  }
  
  protected virtual void Initialize() { }
  protected abstract void EvaluateOnce();
  protected virtual object Finish() { return null; }
}
#endregion

#region ForNode
public sealed class ForNode : LoopNode
{
  public ForNode(string loopVar, Node start, Node end, Node step, Node body) : base(loopVar, start, end, step)
  {
    Body = body;
  }
  
  public readonly Node Body;
  
  protected override void EvaluateOnce()
  {
    Body.Evaluate();
  }

  public override string ToString()
  {
    return "for";
  }
}
#endregion

#region FunctionNode
public sealed class FunctionNode : Node
{
  public FunctionNode(string name, Parameter[] parameters, Node body)
  {
    Name = name;
    Parameters = parameters;
    Body = body;
  }
  
  public readonly string Name;
  public readonly Parameter[] Parameters;
  public readonly Node Body;

  public override object Evaluate()
  {
    return new Function(Name, Parameters, Body);
  }

  public override string ToString()
  {
    return "function "+Name;
  }
}
#endregion

#region OpNode
public sealed class OpNode : Node
{
  public OpNode(Node left, Node right, BinaryOperator op)
  {
    Left  = left;
    Right = right;
    Operator = op;
  }
  
  public readonly Node Left, Right;
  public readonly BinaryOperator Operator;

  public override object Evaluate()
  {
    return Operator.Evaluate(Left, Right);
  }
}
#endregion

#region PrintNode
public sealed class PrintNode : Node
{
  public PrintNode(Node value)
  {
    Value = value;
  }
  
  public readonly Node Value;

  public override object Evaluate()
  {
    Console.WriteLine(Ops.ToString(Value.Evaluate()));
    return null;
  }

  public override string ToString()
  {
    return "print";
  }
}
#endregion

#region SigmaNode
public sealed class SigmaNode : LoopNode
{
  public SigmaNode(string loopVar, Node start, Node end, Node step, Node body) : base(loopVar, start, end, step)
  {
    Body = body;
  }
  
  public readonly Node Body;

  protected override void Initialize()
  {
    value = 0;
  }

  protected override void EvaluateOnce()
  {
    value += Body.ToNumber();
  }

  protected override object Finish()
  {
    return value;
  }

  public override string ToString()
  {
    return "sigma";
  }

  double value;
}
#endregion

#region UnaryOpNode
public sealed class UnaryOpNode : Node
{
  public UnaryOpNode(Node node, UnaryOperator op)
  {
    Node     = node;
    Operator = op;
  }
  
  public readonly Node Node;
  public readonly UnaryOperator Operator;

  public override object Evaluate()
  {
    return Operator.Evaluate(Node);
  }
}
#endregion

} // namespace BuLang