using System;
using System.IO;
using System.Windows.Forms;

namespace BuLang
{

partial class EditorForm : Form
{
  public EditorForm()
  {
    InitializeComponent();
  }

  protected override void OnClosing(System.ComponentModel.CancelEventArgs e)
  {
    base.OnClosing(e);
    e.Cancel = !TryCloseFile();
  }

  void NewFile()
  {
    TryCloseFile();
  }

  void OpenFile()
  {
    if(TryCloseFile())
    {
      OpenFileDialog fd = new OpenFileDialog();
      fd.DefaultExt = "blf";
      fd.Filter = "Bu Language Files (*.blf)|*.blf";
      fd.Title  = "Which file would you like to open?";

      if(fd.ShowDialog() == DialogResult.OK)
      {
        try
        {
          using(StreamReader sr = new StreamReader(fd.FileName))
          {
            editor.Text = sr.ReadToEnd();
            fileName = fd.FileName;
            SetStatusText("Program loaded from "+Path.GetFileName(fileName));
          }
        }
        catch(Exception ex)
        {
          MessageBox.Show("An error occurred while opening the file.\n"+ex.Message, "Error occurred",
                          MessageBoxButtons.OK, MessageBoxIcon.Error);
        }
      }
    }
  }

  bool SaveFile(bool forceNew)
  {
    if(fileName == null || forceNew)
    {
      SaveFileDialog fd = new SaveFileDialog();
      fd.DefaultExt = "blf";
      fd.Filter = "Bu Language Files (*.blf)|*.blf";
      fd.Title = "Save program as...";

      if(fileName != null)
      {
        fd.FileName = fileName;
        fd.InitialDirectory = Path.GetDirectoryName(fileName);
      }
      
      if(fd.ShowDialog() != DialogResult.OK) return false;
      
      fileName = fd.FileName;
    }
    
    try
    {
      using(StreamWriter sw = new StreamWriter(fileName))
      {
        sw.Write(editor.Text);
      }
      fileChanged = false;
      SetStatusText("Your program has been saved as "+Path.GetFileName(fileName));
      return true;
    }
    catch(Exception ex)
    {
      MessageBox.Show("An error occurred while saving the file.\n"+ex.Message, "Error occurred",
                      MessageBoxButtons.OK, MessageBoxIcon.Error);
      return false;
    }
  }

  bool TryCloseFile()
  {
    if(fileChanged)
    {
      DialogResult res = MessageBox.Show("Save changes to your file?", "Save changes?", MessageBoxButtons.YesNoCancel,
                                         MessageBoxIcon.Question, MessageBoxDefaultButton.Button1);
      if(res == DialogResult.Cancel || res == DialogResult.Yes && !SaveFile(false))
      {
        return false;
      }
    }

    editor.Clear();
    SetStatusText("Type your program into the box above.");
    fileChanged = false;
    fileName    = null;
    return true;
  }

  void exitMenuItem_Click(object sender, EventArgs e)
  {
    Close();
  }

  void runMenuItem_Click(object sender, EventArgs e)
  {
    showingRunMessage = false;
    try
    {
      SetStatusText("Running your program now...");
      Console.WriteLine("\nRunning your program...");
      Node program = new Parser(new Scanner(new StringReader(editor.Text), "<editor>")).Parse();
      object value = program.Evaluate();
      if(value != null) Console.WriteLine(Ops.ToString(value));
      SetStatusText("Your program ran successfully.");
    }
    catch(Exception ex)
    {
      SetStatusText("An error occurred.");
      MessageBox.Show(ex.Message, "Error occurred", MessageBoxButtons.OK, MessageBoxIcon.Error);
    }
  }

  void SetStatusText(string text)
  {
    status.Text = text;
    showingRunMessage = false;
  }

  void editor_TextChanged(object sender, EventArgs e)
  {
    if(!showingRunMessage)
    {
      SetStatusText("You have edited your program. Press F5 to run it.");
      Console.WriteLine("\nYou have changed your program. You may wish to press F5 in\n"+
                        "the editor window to apply your changes.");
      showingRunMessage = true;
    }
    fileChanged = true;
  }

  void editor_KeyDown(object sender, KeyEventArgs e)
  {
    if(e.KeyCode == Keys.A && e.Modifiers == Keys.Control) // ctrl-A selects all text
    {
      editor.SelectAll();
    }
  }

  void newMenuItem_Click(object sender, EventArgs e)
  {
    NewFile();
  }

  void openMenuItem_Click(object sender, EventArgs e)
  {
    OpenFile();
  }

  void saveMenuItem_Click(object sender, EventArgs e)
  {
    SaveFile(false);
  }

  void saveAsMenuItem_Click(object sender, EventArgs e)
  {
    SaveFile(true);
  }

  void editor_KeyPress(object sender, KeyPressEventArgs e)
  {
    if(e.KeyChar == '\t')
    {
      if(editor.SelectionStart == editor.TextLength) editor.AppendText("  ");
      else
      {
        int selectionStart = editor.SelectionStart;
        editor.Text = editor.Text.Substring(0, selectionStart) + "  " + editor.Text.Substring(selectionStart);
        editor.SelectionStart = selectionStart + 2;
      }
      e.Handled = true;
    }
  }
  
  string fileName;
  bool showingRunMessage, fileChanged;
}

} // namespace BuLang