using System;

namespace BuLang
{

public static class Operator
{
  public static readonly BinaryOperator Add = AddOperator.Instance;
  public static readonly BinaryOperator Subtract = SubtractOperator.Instance;
  public static readonly BinaryOperator Multiply = MultiplyOperator.Instance;
  public static readonly BinaryOperator Divide = DivideOperator.Instance;
  public static readonly BinaryOperator Modulo = ModuloOperator.Instance;
  public static readonly BinaryOperator Exponent = ExponentOperator.Instance;

  public static readonly BinaryOperator Less = LessOperator.Instance;
  public static readonly BinaryOperator LessEq = LessEqOperator.Instance;
  public static readonly BinaryOperator More = MoreOperator.Instance;
  public static readonly BinaryOperator MoreEq = MoreEqOperator.Instance;
  public static readonly BinaryOperator Equal = EqualOperator.Instance;
  public static readonly BinaryOperator NotEqual = NotEqualOperator.Instance;

  public static readonly BinaryOperator Append = AppendOperator.Instance;

  public static readonly UnaryOperator Negate = NegateOperator.Instance;
  public static readonly UnaryOperator LogicalNegate = LogicalNegateOperator.Instance;
}

public abstract class BinaryOperator
{
  public abstract object Evaluate(Node a, Node b);
}

public abstract class UnaryOperator
{
  public abstract object Evaluate(Node a);
}

#region AddOperator
public sealed class AddOperator : BinaryOperator
{
  AddOperator() { }

  public override object Evaluate(Node a, Node b)
  {
    return a.ToNumber() + b.ToNumber();
  }

  public static readonly AddOperator Instance = new AddOperator();
}
#endregion

#region SubtractOperator
public sealed class SubtractOperator : BinaryOperator
{
  SubtractOperator() { }

  public override object Evaluate(Node a, Node b)
  {
    return a.ToNumber() - b.ToNumber();
  }

  public static readonly SubtractOperator Instance = new SubtractOperator();
}
#endregion

#region MultiplyOperator
public sealed class MultiplyOperator : BinaryOperator
{
  MultiplyOperator() { }

  public override object Evaluate(Node a, Node b)
  {
    return a.ToNumber() * b.ToNumber();
  }

  public static readonly MultiplyOperator Instance = new MultiplyOperator();
}
#endregion

#region DivideOperator
public sealed class DivideOperator : BinaryOperator
{
  DivideOperator() { }

  public override object Evaluate(Node a, Node b)
  {
    double num = a.ToNumber(), denom = b.ToNumber();
    if(denom == 0) throw new ScriptException("You tried to divide by zero. "+b+" was zero.");
    return num / denom;
  }

  public static readonly DivideOperator Instance = new DivideOperator();
}
#endregion

#region ModuloOperator
public sealed class ModuloOperator : BinaryOperator
{
  ModuloOperator() { }

  public override object Evaluate(Node a, Node b)
  {
    double num = a.ToNumber(), denom = b.ToNumber();
    if(denom == 0) throw new ScriptException("You tried to modulo (%) by zero. "+b+" was zero.");
    return Math.IEEERemainder(num, denom);
  }

  public static readonly ModuloOperator Instance = new ModuloOperator();
}
#endregion

#region ExponentOperator
public sealed class ExponentOperator : BinaryOperator
{
  ExponentOperator() { }

  public override object Evaluate(Node a, Node b)
  {
    return Math.Pow(a.ToNumber(), b.ToNumber());
  }

  public static readonly ExponentOperator Instance = new ExponentOperator();
}
#endregion

#region LessOperator
public sealed class LessOperator : BinaryOperator
{
  LessOperator() { }

  public override object Evaluate(Node a, Node b)
  {
    return a.ToNumber() < b.ToNumber();
  }

  public static readonly LessOperator Instance = new LessOperator();
}
#endregion

#region LessEqOperator
public sealed class LessEqOperator : BinaryOperator
{
  LessEqOperator() { }

  public override object Evaluate(Node a, Node b)
  {
    return a.ToNumber() <= b.ToNumber();
  }

  public static readonly LessEqOperator Instance = new LessEqOperator();
}
#endregion

#region MoreOperator
public sealed class MoreOperator : BinaryOperator
{
  MoreOperator() { }

  public override object Evaluate(Node a, Node b)
  {
    return a.ToNumber() > b.ToNumber();
  }

  public static readonly MoreOperator Instance = new MoreOperator();
}
#endregion

#region MoreEqOperator
public sealed class MoreEqOperator : BinaryOperator
{
  MoreEqOperator() { }

  public override object Evaluate(Node a, Node b)
  {
    return a.ToNumber() >= b.ToNumber();
  }

  public static readonly MoreEqOperator Instance = new MoreEqOperator();
}
#endregion

#region EqualOperator
public sealed class EqualOperator : BinaryOperator
{
  EqualOperator() { }

  public override object Evaluate(Node a, Node b)
  {
    const double epsilon = 0.0000001;
    return Math.Abs(a.ToNumber() - b.ToNumber()) < epsilon;
  }

  public static readonly EqualOperator Instance = new EqualOperator();
}
#endregion

#region NotEqualOperator
public sealed class NotEqualOperator : BinaryOperator
{
  NotEqualOperator() { }

  public override object Evaluate(Node a, Node b)
  {
    const double epsilon = 0.0000001;
    return Math.Abs(a.ToNumber() - b.ToNumber()) >= epsilon;
  }

  public static readonly NotEqualOperator Instance = new NotEqualOperator();
}
#endregion

#region AppendOperator
public sealed class AppendOperator : BinaryOperator
{
  AppendOperator() { }

  public override object Evaluate(Node a, Node b)
  {
    return Ops.ToString(a.Evaluate()) + Ops.ToString(b.Evaluate());
  }

  public static readonly AppendOperator Instance = new AppendOperator();
}
#endregion

#region NegateOperator
public sealed class NegateOperator : UnaryOperator
{
  NegateOperator() { }

  public override object Evaluate(Node a)
  {
    return -a.ToNumber();
  }
  
  public static readonly NegateOperator Instance = new NegateOperator();
}
#endregion

#region LogicalNegate
public sealed class LogicalNegateOperator : UnaryOperator
{
  LogicalNegateOperator() { }

  public override object Evaluate(Node a)
  {
    return !Ops.IsTrue(a.Evaluate());
  }
  
  public static readonly LogicalNegateOperator Instance = new LogicalNegateOperator();
}
#endregion

} // namespace BuLang