using System;
using System.Collections.Generic;
using System.IO;

namespace BuLang
{

public sealed class Parser
{
  public Parser(Scanner scanner)
  {
    if(scanner == null) throw new ArgumentNullException();
    this.scanner = scanner;
    NextToken();
  }

  public Node Parse()
  {
    List<Node> nodes = new List<Node>();
    while(!Is(TokenType.EOF))
    {
      nodes.Add(ParseStatement());
      TryConsume(TokenType.Semicolon);
    }
    return nodes.Count == 1 ? nodes[0] : new BlockNode(nodes.ToArray());
  }

  struct VarRange
  {
    public string Name;
    public Node Start, End, Step;
  }

  // statement = <functionDecl> | <assignStmt> | <ifStmt> | <forStmt> | <printStmt> | <expression>
  Node ParseStatement()
  {
    switch(token.Type)
    {
      case TokenType.For: return ParseFor();
      case TokenType.Function: return ParseFunction();
      case TokenType.If: return ParseIf();
      case TokenType.Print: return ParsePrint();
      default:
      {
        Node node = ParseExpression();
        if(node is IdentifierNode && TryConsume(TokenType.Assign))
        {
          return new AssignNode(((IdentifierNode)node).Name, ParseExpression());
        }
        else return node;
      }
    }
  }

  Node ParseCurlyBlock()
  {
    List<Node> nodes = new List<Node>();

    Consume(TokenType.LCurly);
    while(!TryConsume(TokenType.RCurly))
    {
      nodes.Add(ParseStatement());
      TryConsume(TokenType.Semicolon);
    }
    
    return nodes.Count == 1 ? nodes[0] : new BlockNode(nodes.ToArray());
  }

  VarRange ParseVarRange()
  {
    VarRange range = new VarRange();
    range.Name = ParseIdentifier();
    Consume(TokenType.Assign);
    range.Start = ParseExpression();
    Consume(TokenType.Colon);
    range.End = ParseExpression();
    if(TryConsume(TokenType.Colon)) range.Step = ParseExpression();
    return range;
  }

  // forStmt = 'for' <varRange> <curlyBlock>
  // varRange   = <identifier> '=' <expression> ':' <expression> [':' <expression>]
  // curlyBlock = '{' <statement>* '}'
  Node ParseFor()
  {
    NextToken();
    VarRange range = ParseVarRange();
    return new ForNode(range.Name, range.Start, range.End, range.Step, ParseCurlyBlock());
  }
  
  // functionDecl = 'function' <identifier> '(' <paramList> ')' <curlyBlock>
  // paramList    = <param>? | <param> (',' <param>)*
  // param        = <identifier> ('=' <expression>)
  Node ParseFunction()
  {
    NextToken();

    string name = ParseIdentifier();
    Consume(TokenType.LParen);
    
    List<Parameter> paramList = new List<Parameter>();
    while(!TryConsume(TokenType.RParen))
    {
      string paramName  = ParseIdentifier();
      Node defaultValue = TryConsume(TokenType.Assign) ? ParseExpression() : null;
      paramList.Add(new Parameter(paramName, defaultValue));
      if(!Is(TokenType.RParen)) Consume(TokenType.Comma);
    }
    
    return new AssignNode(name, new FunctionNode(name, paramList.ToArray(), ParseCurlyBlock()));
  }

  // ifStmt = 'if' <expression> <curlyBlock> ('else' (<ifStmt> | <curlyBlock>))?
  Node ParseIf()
  {
    NextToken();
    
    Node condition = ParseExpression(), ifTrue = ParseCurlyBlock(), ifFalse = null;
    if(TryConsume(TokenType.Else))
    {
      ifFalse = Is(TokenType.If) ? ParseIf() : ParseCurlyBlock();
    }
    return new IfNode(condition, ifTrue, ifFalse);
  }

  // printStmt = 'print' <expression>
  Node ParsePrint()
  {
    NextToken();
    return new PrintNode(ParseExpression());
  }

  // sigmaExpr = 'sigma' <varRange> <curlyBlock>
  Node ParseSigma()
  {
    NextToken();
    VarRange range = ParseVarRange();
    return new SigmaNode(range.Name, range.Start, range.End, range.Step, ParseCurlyBlock());
  }

  // expression = <stringExpr> ? <stringExpr> : <expression>
  Node ParseExpression()
  {
    Node node = ParseStringExpression();
    if(TryConsume(TokenType.Question))
    {
      Node ifTrue = ParseStringExpression();
      Consume(TokenType.Colon);
      node = new IfNode(node, ifTrue, ParseExpression());
    }
    return node;
  }

  // stringExpr = <subexpr> ('&' <subexpr>)*
  Node ParseStringExpression()
  {
    Node node = ParseSubExpression();
    while(TryConsume(TokenType.Append))
    {
      node = new OpNode(node, ParseSubExpression(), Operator.Append);
    }
    return node;
  }

  // subexpr = <comparand> (<compareOp> <comparand>)*
  // compareOp = '<' | '<=' | '>' | '>=' | '==' | '!='
  Node ParseSubExpression()
  {
    Node node = ParseComparand();
    while(true)
    {
      BinaryOperator op = null;
      switch(token.Type)
      {
        case TokenType.Less:     op = Operator.Less;     break;
        case TokenType.LessEq:   op = Operator.LessEq;   break;
        case TokenType.More:     op = Operator.More;     break;
        case TokenType.MoreEq:   op = Operator.MoreEq;   break;
        case TokenType.Equal:    op = Operator.Equal;    break;
        case TokenType.NotEqual: op = Operator.NotEqual; break;
      }
      if(op == null) break;
      NextToken();
      node = new OpNode(node, ParseComparand(), op);
    }
    return node;
  }

  // comparand = <term> (('+' | '-') <term>)*
  Node ParseComparand()
  {
    Node node = ParseTerm();
    while(Is(TokenType.Plus) || Is(TokenType.Minus))
    {
      BinaryOperator op = Is(TokenType.Plus) ? Operator.Add : Operator.Subtract;
      NextToken();
      node = new OpNode(node, ParseTerm(), op);
    }
    return node;
  }

  // term = <factor> (('*' | '/' | '%') <factor>*
  Node ParseTerm()
  {
    Node node = ParseFactor();
    while(Is(TokenType.Multiply) || Is(TokenType.Divide) || Is(TokenType.Modulo))
    {
      BinaryOperator op = Is(TokenType.Multiply) ? Operator.Multiply :
                          Is(TokenType.Divide)   ? Operator.Divide   : Operator.Modulo;
      NextToken();
      node = new OpNode(node, ParseFactor(), op);
    }
    return node;
  }

  // factor = <unary> ('^' <unary>)*
  Node ParseFactor()
  {
    Node node = ParseUnary();
    while(TryConsume(TokenType.Exponent))
    {
      node = new OpNode(node, ParseUnary(), Operator.Exponent);
    }
    return node;
  }

  // unary = ('+' | '-' | '!') <unary> | <primary>
  Node ParseUnary()
  {
    Node node;
    if(Is(TokenType.Plus) || Is(TokenType.Minus) || Is(TokenType.Not))
    {
      UnaryOperator op = Is(TokenType.Plus) ? null : Is(TokenType.Minus) ? Operator.Negate : Operator.LogicalNegate;
      NextToken();
      node = ParseUnary();
      if(op != null) node = new UnaryOpNode(node, op);
    }
    else
    {
      node = ParsePrimary();
    }
    return node;
  }

  // primary = <literal> | <sigmaExpr> | <identifier> ('(' <argList> ')') | '(' <expression> ')'
  // argList = <expression>? | <expression> (',' <expression>)*
  Node ParsePrimary()
  {
    Node node;
    if(Is(TokenType.Literal))
    {
      node = new LiteralNode(token.Value);
      NextToken();
    }
    else if(Is(TokenType.Identifier))
    {
      node = new IdentifierNode(ParseIdentifier());
      if(TryConsume(TokenType.LParen))
      {
        List<Node> args = new List<Node>();
        while(!TryConsume(TokenType.RParen))
        {
          args.Add(ParseExpression());
          if(!Is(TokenType.RParen)) Consume(TokenType.Comma);
        }
        node = new CallNode(node, args.ToArray());
      }
    }
    else if(TryConsume(TokenType.LParen))
    {
      node = ParseExpression();
      Consume(TokenType.RParen);
    }
    else if(Is(TokenType.Sigma))
    {
      node = ParseSigma();
    }
    else throw SyntaxError("I didn't expect to see a "+token.Type+" here.");
    return node;
  }

  string ParseIdentifier()
  {
    Expect(TokenType.Identifier);
    string name = (string)token.Value;
    NextToken();
    return name;
  }

  void Consume(TokenType type)
  {
    Expect(type);
    NextToken();
  }

  bool Is(TokenType type)
  {
    return token.Type == type;
  }

  void Expect(TokenType type)
  {
    if(!Is(type))
    {
      throw SyntaxError("I expected to find "+type+" but instead found "+token.Type);
    }
  }

  bool TryConsume(TokenType type)
  {
    if(Is(type))
    {
      NextToken();
      return true;
    }
    else return false;
  }

  TokenType NextToken()
  {
    if(!scanner.NextToken(out token))
    {
      token.Type = TokenType.EOF;
    }
    return token.Type;
  }

  SyntaxErrorException SyntaxError(string message)
  {
    return new SyntaxErrorException(token.Start, token.SourceName, message);
  }

  readonly Scanner scanner;
  Token token;
}

} // namespace BuLang