using System;

namespace BuLang
{

#region IFunction
interface IFunction
{
  object Call(params object[] args);
}
#endregion

#region ScriptException
public class ScriptException : ApplicationException
{
  public ScriptException(string message) : base(message) { }
  public ScriptException(FilePosition position, string source, string message) : base(message)
  {
    this.Position   = position;
    this.SourceName = source;
  }

  public FilePosition Position;
  public string SourceName;
}
#endregion

#region Function
public sealed class Function : IFunction
{
  public Function(string name, Parameter[] parameters, Node body)
  {
    Name = name;
    Parameters = parameters;
    Body = body;
  }

  public readonly string Name;
  public readonly Parameter[] Parameters;
  public readonly Node Body;
  
  public object Call(object[] args)
  {
    LocalEnvironment env = LocalEnvironment.PushNew();
    try
    {
      for(int i=0; i<args.Length; i++)
      {
        env.Bind(Parameters[i].Name, args[i]);
      }
      
      return Body.Evaluate();
    }
    finally { LocalEnvironment.Pop(); }
  }

  public override string ToString()
  {
    return "<function "+Name+">";
  }
}
#endregion

#region LocalEnvironment
public sealed class LocalEnvironment
{
  public LocalEnvironment(LocalEnvironment parent)
  {
    Parent = parent;
  }

  static LocalEnvironment()
  {
    Current = new LocalEnvironment(null);
    Current.Bind("ask", new AskFunction());
    Current.Bind("round", new RoundFunction());
    Current.Bind("eval", new EvalFunction());
  }

  public void Bind(string name, object value)
  {
    dict[name] = value;
  }

  public object Get(string name)
  {
    object value = dict[name];
    if(value == null && !dict.Contains(name))
    {
      if(Parent == null) throw new ScriptException("You tried to use a variable named '"+name+"' but it didn't exist.");
      value = Parent.Get(name);
    }
    return value;
  }

  public void Set(string name, object value)
  {
    if(dict.Contains(name)) dict[name] = value;
    else
    {
      LocalEnvironment env = this;
      do env = env.Parent; while(env != null && !env.dict.Contains(name));
      if(env != null) env.dict[name] = value;
      else dict[name] = value;
    }
  }

  public readonly LocalEnvironment Parent;

  public static LocalEnvironment PushNew()
  {
    return Current = new LocalEnvironment(Current);
  }

  public static void Pop()
  {
    Current = Current.Parent;
  }

  public static LocalEnvironment Current;

  sealed class AskFunction : IFunction
  {
    public object Call(object[] args)
    {
      Console.Write(args.Length == 0 ? "You wanted to ask something: " : Ops.ToString(args[0])+" ");
      return Console.ReadLine();
    }
  }
  
  sealed class EvalFunction : IFunction
  {
    public object Call(object[] args)
    {
      if(args.Length == 0) return null;
      return new Parser(new Scanner(new System.IO.StringReader(Ops.ToString(args[0])), "<eval>")).Parse().Evaluate();
    }
  }

  sealed class RoundFunction : IFunction
  {
    public object Call(object[] args)
    {
      if(args.Length == 1) return Math.Round(Ops.ToNumber(args[0]), 2);
      else if(args.Length >= 2) return Math.Round(Ops.ToNumber(args[0]), Ops.ToInt(args[1]));
      else return 0.0;
    }
  }

  readonly System.Collections.Specialized.ListDictionary dict = new System.Collections.Specialized.ListDictionary();
}
#endregion

#region Ops
public static class Ops
{
  public static bool IsTrue(object value)
  {
    if(value is bool) return (bool)value;
    else
    {
      try { return Convert.ToBoolean(value); }
      catch
      {
        throw new ScriptException("I expected a boolean (true/false) but I got '"+Ops.ToString(value)+"'");
      }
    } 
  }

  public static int ToInt(object value)
  {
    if(value is double) return (int)Math.Round((double)value);
    else
    {
      try { return Convert.ToInt32(value); }
      catch
      {
        throw new ScriptException("I expected a number but I got '"+Ops.ToString(value)+"'");
      }
    }
  }

  public static double ToNumber(object value)
  {
    if(value is double) return (double)value;
    else
    {
      try { return Convert.ToDouble(value); }
      catch
      {
        throw new ScriptException("I expected a number but I got '"+Ops.ToString(value)+"'");
      }
    }
  }

  public static string ToString(object value)
  {
    return value == null ? "[NULL]" : value.ToString();
  }
}
#endregion

} // namespace BuLang