using System;
using System.Collections;
using System.Drawing;
using System.IO;
using System.Text.RegularExpressions;
using System.Windows.Forms;

namespace Markov
{
	class MainForm : Form
	{ public MainForm() { InitializeComponent(); }

    protected override void OnLoad(EventArgs e)
    { base.OnLoad(e);
      timer.Tick += new EventHandler(OnTick); // this timer updates 'txtSug'
	    timer.Interval = 1000;
	    timer.Start();
    }

    private System.Windows.Forms.Label lblStatus;

    // This value must be at least 1. If it's set to higher values, the text
    // will be more similar to the training text.
    const int WindowSize = 2;

    // Adds a new root node
    Node AddNode(string text)
    { Node n = (Node)nodes[text];
      if(n==null) nodes[text] = n = new Node(text);
      n.Count++;
      return n;
    }

    // Learn from a string.
    // Break the text into bits, and for each set of words in the sliding
    // window of WindowSize+1 words, call the other Learn() function.
    void Learn(string text)
    { if(text=="") return;
      string[] chunks = splitre.Split(text); // split text into words
      if(chunks.Length<=WindowSize+1) Learn(chunks, 0, chunks.Length);
      else
        for(int i=0; i<=chunks.Length-(WindowSize+1); i++)
          Learn(chunks, i, WindowSize+1);
    }
    
    // Learn from a subset of an array
    void Learn(string[] chain, int index, int length)
    { Node n = AddNode(chain[index]);
      for(int i=1; i<length; i++) n = n.AddChild(chain[index+i]);
    }
    
    // Given a chain of words (which chain doesn't have to be full, but does
    // need at least one word), returns the next word at random, or returns
    // null if there is no next word.
    string NextWord(string[] words)
    { Node n = (Node)nodes[words[0]];
      if(n==null) return null; // if there's no entry for the first word, abort

      int end=0; // find the end of the chain
      for(; end<words.Length; end++) if(words[end]==null) break;
      for(int i=1; i<end; i++) // traverse the markov chain to find end node
      { n = n.GetNode(words[i]);
        if(n==null) return null; // if we don't know about that chain, abort
      }
      if(n.NumChildren==0) return null; // abort if there're no children

      // select a child node at random (but respecting probabilities)
      string word = n.RandomNode().Text;
      // add the word to the end of 'words'
      if(end<words.Length) words[end]=word;
      else
      { for(int i=0; i<end-1; i++) words[i]=words[i+1];
        words[end-1] = word;
      }
      return word; // and return it
    }

    void OnTick(object sender, EventArgs e)
    { if(!changed || thinking) return;
      changed = false;
      string txt = txtNew.Text.Trim(), lstart=txt+' ';
      if(txt=="") { txtSug.Text=""; return; }
      thinking = true; // prevent re-entry

      string[] chunks = splitre.Split(txt); // split text into words
      Node n;
      int start=Math.Max(0, chunks.Length-WindowSize); // start near the end
      txt="";

      lblStatus.Text = "Insufficient information (results will be poorer).";
      // increment 'start' until we have an unbroken chain. This is not
      // necessary but allows us to get matches is more situations.
      for(int i; start<chunks.Length-1; start++)
      { n = (Node)nodes[chunks[start]];
        for(i=start+1; i<chunks.Length; i++)
          if(n==null || n.GetNode(chunks[i])==null) goto nextStart;
        lblStatus.Text = "";
        break;
        nextStart:;
      }

      n = (Node)nodes[chunks[start]];
      if(n!=null)
      { string[] words = new string[WindowSize];
        Graphics g = this.CreateGraphics();
        float linelen = g.MeasureString(lstart, txtSug.Font).Width;

        for(int lines=0,maxlines=txtSug.Height/txtSug.Font.Height; lines<maxlines; lines++)
        { string  line = lstart;
          float length = linelen;
          int i=0, j=start;
          // fill 'words' with the words from 'start' to the end
          for(; j<chunks.Length; i++,j++) words[i]=chunks[j];
          for(; i<words.Length; i++) words[i]=null;

          while(true)
          { string word = NextWord(words);
            if(word==null ||    // if there's no next word
               (int)(length+=g.MeasureString(word+' ', txtSug.Font).Width) >=
               txtSug.Width-10) // or if the word would cause wrapping
              break;            // then we're done with this line
            line += word+' ';   // otherwise, append the word
          }
          txt += line+'\n';     // we're done with this line
        }
      }

      txtSug.Text = txt;
      thinking = false;
    }
    
	  Timer timer = new Timer();
	  Hashtable nodes = new Hashtable();
	  Regex splitre = new Regex(@"\s+", RegexOptions.Singleline);
	  bool changed, thinking;

    #region Event handlers
    private void btnLearn_Click(object sender, EventArgs e)
    { Learn(txtNew.Text.Trim()); // learn from what the user typed
      txtNew.Text = "";
      lblStatus.Text = "The sentence has been learned.";
    }

    private void mnuFileTrain_Click(object sender, System.EventArgs e)
    { OpenFileDialog ofd = new OpenFileDialog();
      ofd.ShowDialog(this); // get the filename from the user

      if(File.Exists(ofd.FileName))
      { StreamReader sr = new StreamReader(ofd.FileName);
        Learn(sr.ReadToEnd());
        sr.Close();
        lblStatus.Text = "Trained on the text file "+ofd.FileName;
      }
    }

    private void mnuFileExit_Click(object sender, EventArgs e)
    { Application.Exit(); // quit
    }

    private void txtNew_TextChanged(object sender, EventArgs e)
    { changed = true;
      lblStatus.Text = "";
    }
    #endregion
    
		#region Windows Form Designer generated code
		/**** EVERYTHING AFTER HERE CAN BE IGNORED ****/
		void InitializeComponent()
		{ 
      this.mainMenu1 = new System.Windows.Forms.MainMenu();
      this.mnuFile = new System.Windows.Forms.MenuItem();
      this.mnuFileTrain = new System.Windows.Forms.MenuItem();
      this.mnuSep = new System.Windows.Forms.MenuItem();
      this.mnuFileExit = new System.Windows.Forms.MenuItem();
      this.txtSug = new System.Windows.Forms.Label();
      this.txtNew = new System.Windows.Forms.TextBox();
      this.btnLearn = new System.Windows.Forms.Button();
      this.lblStatus = new System.Windows.Forms.Label();
      this.SuspendLayout();
      // 
      // mainMenu1
      // 
      this.mainMenu1.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
                                                                              this.mnuFile});
      // 
      // mnuFile
      // 
      this.mnuFile.Index = 0;
      this.mnuFile.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
                                                                            this.mnuFileTrain,
                                                                            this.mnuSep,
                                                                            this.mnuFileExit});
      this.mnuFile.Text = "&File";
      // 
      // mnuFileTrain
      // 
      this.mnuFileTrain.Index = 0;
      this.mnuFileTrain.Text = "&Train on text file...";
      this.mnuFileTrain.Click += new System.EventHandler(this.mnuFileTrain_Click);
      // 
      // mnuSep
      // 
      this.mnuSep.Index = 1;
      this.mnuSep.Text = "-";
      // 
      // mnuFileExit
      // 
      this.mnuFileExit.Index = 2;
      this.mnuFileExit.Text = "E&xit";
      this.mnuFileExit.Click += new System.EventHandler(this.mnuFileExit_Click);
      // 
      // txtSug
      // 
      this.txtSug.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
        | System.Windows.Forms.AnchorStyles.Left) 
        | System.Windows.Forms.AnchorStyles.Right)));
      this.txtSug.BorderStyle = System.Windows.Forms.BorderStyle.Fixed3D;
      this.txtSug.Location = new System.Drawing.Point(8, 24);
      this.txtSug.Name = "txtSug";
      this.txtSug.Size = new System.Drawing.Size(576, 176);
      this.txtSug.TabIndex = 0;
      // 
      // txtNew
      // 
      this.txtNew.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Left) 
        | System.Windows.Forms.AnchorStyles.Right)));
      this.txtNew.Location = new System.Drawing.Point(8, 208);
      this.txtNew.Name = "txtNew";
      this.txtNew.Size = new System.Drawing.Size(520, 20);
      this.txtNew.TabIndex = 2;
      this.txtNew.Text = "Enter new text here.";
      this.txtNew.TextChanged += new System.EventHandler(this.txtNew_TextChanged);
      // 
      // btnLearn
      // 
      this.btnLearn.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Right)));
      this.btnLearn.Location = new System.Drawing.Point(536, 208);
      this.btnLearn.Name = "btnLearn";
      this.btnLearn.Size = new System.Drawing.Size(48, 20);
      this.btnLearn.TabIndex = 3;
      this.btnLearn.Text = "&Learn";
      this.btnLearn.Click += new System.EventHandler(this.btnLearn_Click);
      // 
      // lblStatus
      // 
      this.lblStatus.Location = new System.Drawing.Point(8, 6);
      this.lblStatus.Name = "lblStatus";
      this.lblStatus.Size = new System.Drawing.Size(576, 16);
      this.lblStatus.TabIndex = 4;
      // 
      // MainForm
      // 
      this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
      this.ClientSize = new System.Drawing.Size(592, 233);
      this.Controls.Add(this.lblStatus);
      this.Controls.Add(this.btnLearn);
      this.Controls.Add(this.txtNew);
      this.Controls.Add(this.txtSug);
      this.Menu = this.mainMenu1;
      this.Name = "MainForm";
      this.Text = "Markov Sample";
      this.ResumeLayout(false);

    }
		#endregion

    #region Generated variables
    MainMenu mainMenu1;
    Label txtSug;
    TextBox txtNew;
    Button btnLearn;
    MenuItem mnuFile;
    MenuItem mnuFileTrain;
    MenuItem mnuSep;
    MenuItem mnuFileExit;
    #endregion
	}
}
