using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.IO;
using System.Text.RegularExpressions;
using System.Windows.Forms;

namespace Nails
{
  public partial class MainForm : Form
  {
    public MainForm()
    {
      InitializeComponent();
    }

    string[] GetInputFiles()
    {
      string[] rawFiles = txtInputFiles.Text.Split(';');
      List<string> inputFiles = new List<string>(rawFiles.Length);
      for(int i=0; i<rawFiles.Length; i++)
      {
        rawFiles[i] = rawFiles[i].Trim();
        if(!string.IsNullOrEmpty(rawFiles[i])) inputFiles.Add(rawFiles[i]);
      }
      return inputFiles.ToArray();
    }

    void ResizeFiles(string[] inputFiles, string outputDir, int width, int height, string namingScheme)
    {
      List<KeyValuePair<string,Exception>> failed = new List<KeyValuePair<string,Exception>>();

      double idealAspect = height == 0 ? 0 : (double)width / height;

      // configure the progress bar
      progress.Value   = 0;
      progress.Maximum = inputFiles.Length;

      // try to resize each image
      foreach(string inputFile in inputFiles)
      {
        if(!string.IsNullOrEmpty(inputFile))
        {
          Bitmap newImage = null;
          try
          {
            ImageFormat saveFormat;
            using(Image image = Image.FromFile(inputFile))
            {
              double aspectRatio = (double)image.Width / image.Height;
              int newWidth, newHeight;

              if(width == 0 || height != 0 && aspectRatio < idealAspect)
              {
                newWidth  = (int)Math.Round(height * aspectRatio);
                newHeight = height;
              }
              else
              {
                newWidth  = width;
                newHeight = (int)Math.Round(width / aspectRatio);
              }

              newImage = new Bitmap(newWidth, newHeight, PixelFormat.Format24bppRgb);
              newImage.Palette = image.Palette; // copy the palette, if there is one

              using(Graphics g = Graphics.FromImage(newImage))
              {
                g.CompositingMode    = CompositingMode.SourceCopy;
                g.CompositingQuality = CompositingQuality.HighQuality;
                g.InterpolationMode  = InterpolationMode.HighQualityBicubic;
                g.DrawImage(image, new Rectangle(0, 0, newWidth, newHeight));
              }

              saveFormat = image.RawFormat;
            }

            string outputFile = CreateOutputFilename(inputFile, outputDir, namingScheme);
            newImage.Save(outputFile, saveFormat);
            newImage.Dispose();
          }
          catch(Exception e)
          {
            if(newImage != null) newImage.Dispose();
            failed.Add(new KeyValuePair<string,Exception>(inputFile, e));
          }
        }

        progress.Value++;

        // process messages in the message queue so the UI doesn't look frozen
        Application.DoEvents();
      }

      if(failed.Count != 0)
      {
        string message = "The following images failed:\n";
        foreach(KeyValuePair<string, Exception> pair in failed)
        {
          message += pair.Key + ": " + pair.Value.Message + "\n";
        }
        MessageBox.Show(message, "Some files failed", MessageBoxButtons.OK, MessageBoxIcon.Warning);
      }

      progress.Value = 0;
    }

    void btnHelp_Click(object sender, EventArgs e)
    {
      MessageBox.Show("1. Select the input files and output directory.\n\n"+
                      "2. Select the thumbnail width and/or height. A blank width or height means to allow that "+
                      "dimension to stretch arbitrarily to match the other dimension. If both width and height are "+
                      "specified, the thumbnail will be at most the given width and height. In all cases, the "+
                      "original aspect ratio will be preserved.\n\n"+
                      "3. Select a naming scheme, or use the default.\n\n"+
                      "4. Click Go!", "Simple Help", MessageBoxButtons.OK);
    }

    void btnGo_Click(object sender, EventArgs e)
    {
      // first ensure that the input files are valid
      string[] inputFiles = GetInputFiles();
      foreach(string inputFile in inputFiles)
      {
        if(!File.Exists(inputFile))
        {
          MessageBox.Show("The input file '"+inputFile+"' was not found.", "File not found",
                          MessageBoxButtons.OK, MessageBoxIcon.Error);
          return;
        }
      }

      if(inputFiles.Length == 0)
      {
        MessageBox.Show("No input files were specified.", "Nothing to do", MessageBoxButtons.OK, MessageBoxIcon.Error);
        return;
      }

      // now check the output directory
      string outputDir = txtOutputDir.Text.Trim();
      if(string.IsNullOrEmpty(outputDir))
      {
        MessageBox.Show("No output directory was specified.", "No output directory",
                        MessageBoxButtons.OK, MessageBoxIcon.Error);
        return;
      }

      if(!Directory.Exists(outputDir))
      {
        MessageBox.Show("The output directory '"+outputDir+"' does not exist.", "Bad output directory",
                        MessageBoxButtons.OK, MessageBoxIcon.Error);
        return;
      }

      // now get the width and height
      int width, height;

      if(string.IsNullOrEmpty(txtWidth.Text.Trim())) width = 0;
      else if(!int.TryParse(txtWidth.Text.Trim(), out width) || width < 0)
      {
        MessageBox.Show("The thumbnail width is invalid. Please specify a positive integer.", "Invalid width",
                        MessageBoxButtons.OK, MessageBoxIcon.Error);
        return;
      }

      if(string.IsNullOrEmpty(txtHeight.Text.Trim())) height = 0;
      else if(!int.TryParse(txtHeight.Text.Trim(), out height) || height < 0)
      {
        MessageBox.Show("The thumbnail height is invalid. Please specify a positive integer.", "Invalid height",
                        MessageBoxButtons.OK, MessageBoxIcon.Error);
        return;
      }

      if(width == 0 && height == 0)
      {
        MessageBox.Show("A thumbnail width or height must be specified.", "Missing width or height!",
                        MessageBoxButtons.OK, MessageBoxIcon.Error);
        return;
      }

      string namingScheme = txtNaming.Text.Trim();
      if(string.IsNullOrEmpty(namingScheme)) namingScheme = "%f_t.%e";

      Button btnGo = (Button)sender;
      btnGo.Enabled = false;
      ResizeFiles(inputFiles, outputDir, width, height, namingScheme);
      btnGo.Enabled = true;
    }

    void btnInputBrowse_Click(object sender, EventArgs e)
    {
      using(OpenFileDialog ofd = new OpenFileDialog())
      {
        ofd.AddExtension = false;
        ofd.Filter = "Images (*.jpg;*.jpeg;*.bmp;*.gif;*.png)|*.jpg;*.jpeg;*.bmp;*.gif;*.png|All Files (*.*)|*.*";
        ofd.Multiselect = true;
        ofd.Title = "Select image files to resize.";

        if(ofd.ShowDialog() == DialogResult.OK) txtInputFiles.Text = string.Join("; ", ofd.FileNames);
      }
    }

    void btnSelectOutputDir_Click(object sender, EventArgs e)
    {
      string[] inputFiles = GetInputFiles();
      using(FolderBrowserDialog fbd = new FolderBrowserDialog())
      {
        fbd.Description = "Select where the thumbnails will be saved.";
        if(inputFiles.Length != 0) fbd.SelectedPath = Path.GetDirectoryName(inputFiles[0]);
        if(fbd.ShowDialog() == DialogResult.OK) txtOutputDir.Text = fbd.SelectedPath;
      }
    }

    static string CreateOutputFilename(string inputFile, string outputDirectory, string namingScheme)
    {
      string filename = Path.GetFileNameWithoutExtension(inputFile);
      string extension = Path.GetExtension(inputFile).TrimStart('.');

      string outputFilename = renameRe.Replace(namingScheme, delegate(Match m)
      {
        switch(char.ToLowerInvariant(m.Value[1]))
        {
          case '%': return "%";
          case 'f': return filename;
          case 'e': return extension;
          default: return string.Empty;
        }
      });

      return Path.Combine(outputDirectory, outputFilename);
    }

    static Regex renameRe = new Regex(@"%[fe%]", RegexOptions.IgnoreCase | RegexOptions.Singleline);
  }
}