/*
TerrainGen is a prototype for a terrain generator.
http://www.adammil.net/
Copyright (C) 2008 Adam Milazzo

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/* This code was a prototype for the terrain generator in Dwarf Dungeon.
 * It is very messy code.
 * 
 * It uses the following key bindings:
 * 
 * General:
 * Escape             - Quit
 * 
 * Rendering:
 * Arrow keys         - Rotate
 * Alt-arrows         - Pan
 * Ctrl-(up / down)   - Zoom
 * r                  - Reset viewpoint
 * o                  - Toggle ocean map
 * t                  - Show temperature map
 * p                  - Show precipitation map
 * d                  - Show drainage map
 * l                  - Change light angle
 * 1-5                - Set rendering detail
 * 
 * Generation:
 * g                  - Generate another world
 * e                  - Toggle higher-res terrain and erosion steps
 * h                  - Toggle association between height and cragginess
 * Alt-(1-4)          - Set world size
 * 6-0 and Alt-(6-0)  - Set fractal detail level
 * -                  - Reset erosion cycles to zero
 * =                  - Increase erosion cycles by 50
 * R                  - Toggle final rainfall (to create lakes)
 */

using System;
using System.Globalization;
using System.Threading;

using AdamMil.Mathematics.Random;
using AdamMil.Mathematics.Geometry.ThreeD;
using GameLib.Events;
using GameLib.Video;
using GameLib.Interop.OpenGL;
using System.Collections.Generic;
using System.Drawing;

namespace TerrainGen
{

static class Program
{
  static Program()
  {
    CPUCount = Math.Max(1,
      Microsoft.Win32.Registry.LocalMachine.OpenSubKey(@"HARDWARE\DESCRIPTION\System\CentralProcessor").SubKeyCount);
  }

  /// <summary>Gets the number of logical CPUs on the user's machine.</summary>
  public static readonly int CPUCount;

  struct Quad
  {
    public Quad(int x, int y, int x2, int y2)
    {
      X = (ushort)x;
      Y = (ushort)y;
      X2 = (ushort)x2;
      Y2 = (ushort)y2;
      NX = NY = NZ = 0;
    }
    public ushort X, Y, X2, Y2;
    public float NX, NY, NZ;
  }

  [Flags]
  enum OceanSide
  {
    None=0, Left=1, Right=2, Top=4, Bottom=8
  }

  const int WaterThreshold = 32;

  static float[,] heightMap;
  static byte[,] tempMap, rainMap, drainMap;
  static bool[,] oceanMap;

  static System.Collections.Generic.List<Quad> quads = new System.Collections.Generic.List<Quad>();

  static void InitOpenGL()
  {
    Video.SetGLMode(800, 600, 0);

    GL.glEnable(GL.GL_DEPTH_TEST); // z-buffering
    GL.glDepthFunc(GL.GL_LEQUAL);

    GL.glDisable(GL.GL_BLEND);
    GL.glDisable(GL.GL_TEXTURE_2D); // texture mapping
    GL.glHint(GL.GL_PERSPECTIVE_CORRECTION_HINT, GL.GL_NICEST);

    GL.glClearColor(0, 0, 0, 1); // misc stuff
    GL.glDisable(GL.GL_DITHER);
    GL.glEnable(GL.GL_CULL_FACE);

    GL.glViewport(0, 0, Video.Width, Video.Height);

    GL.glMatrixMode(GL.GL_PROJECTION); // matrices
    GL.glLoadIdentity();
    //GLU.gluOrtho2D(0, 256, 256, 0);
    GLU.gluPerspective(45.0f, (double)Video.Width/Video.Height, 1f, 1000.0f);
    //GL.glFrustum(-SizeAtNear, SizeAtNear, SizeAtNear, -SizeAtNear, NearZ, FarZ);

    GL.glMatrixMode(GL.GL_MODELVIEW);
    GL.glLoadIdentity();

    GL.glShadeModel(GL.GL_SMOOTH); // lighting
    GL.glEnable(GL.GL_LIGHTING);
    GL.glEnable(GL.GL_LIGHT0);
    GL.glLightModeli(GL.GL_LIGHT_MODEL_TWO_SIDE, GL.GL_FALSE);
    GL.glColorMaterial(GL.GL_FRONT_AND_BACK, GL.GL_AMBIENT_AND_DIFFUSE);
    GL.glEnable(GL.GL_COLOR_MATERIAL);

    GL.glLightfv(GL.GL_LIGHT0, GL.GL_AMBIENT, new float[] { .5f,.5f,.5f, 1 });
    GL.glLightfv(GL.GL_LIGHT0, GL.GL_DIFFUSE, new float[] { 1, 1, 1, 1 });
  }

  const float HeightScale = 0.05f;
  static float yangle = -30, zangle = 0, xoff = 0, yoff = 0, zoom = -50;

  static bool showDrainMap, showOceanMap, showTempMap, showRainMap;

  static float _halfSize, _mul;
  static void Render()
  {
    GL.glLoadIdentity();
    GL.glTranslatef(0, 0, -3.5f);

    GL.glClear(GL.GL_COLOR_BUFFER_BIT | GL.GL_DEPTH_BUFFER_BIT);
    if(doLighting) GL.glLightPosition(GL.GL_LIGHT0, 1, 1, 1, 0);
    else GL.glLightPosition(GL.GL_LIGHT0, 0, 0, 1, 0);
    GL.glTranslatef(xoff, yoff, zoom);
    GL.glRotatef(yangle, 1, 0, 0);
    GL.glRotatef(zangle, 0, 0, 1);

    GL.glBegin(GL.GL_QUADS);
    _halfSize = renderMap.GetLength(0) * 0.5f;
    _mul = doExpand ? 2f/ExpansionFactor : 2;
    foreach(Quad quad in quads)
    {
      GL.glNormal3f(quad.NX, quad.NY, quad.NZ);

      RenderQuadVertex(quad.X, quad.Y);
      RenderQuadVertex(quad.X, quad.Y2);
      RenderQuadVertex(quad.X2, quad.Y2);
      RenderQuadVertex(quad.X2, quad.Y);
    }
    GL.glEnd();

    Video.Flip();
  }

  static byte BlendColor(byte a1, byte a2, int min, int max, int value)
  {
    float scale = (float)(value - min) / (max - min);
    return (byte)(a1 + scale * (a2 - a1) + 0.5f);
  }

  static void BlendColor(byte a1, byte b1, byte a2, byte b2, out byte a, out byte b, int min, int max, int value)
  {
    float scale = (float)(value - min) / (max - min);
    a = (byte)(a1 + scale * (a2 - a1) + 0.5f);
    b = (byte)(b1 + scale * (b2 - b1) + 0.5f);
  }

  static void BlendColor(byte r1, byte g1, byte b1, byte r2, byte g2, byte b2, out byte r, out byte g, out byte b,
                         int min, int max, int value)
  {
    float scale = (float)(value - min) / (max - min);
    r = (byte)(r1 + scale * (r2 - r1) + 0.5f);
    g = (byte)(g1 + scale * (g2 - g1) + 0.5f);
    b = (byte)(b1 + scale * (b2 - b1) + 0.5f);
  }

  static void RenderQuadVertex(int x, int y)
  {
    int height = GetHeight(x, y), sx=x, sy=y;

    if(doExpand)
    {
      sx /= ExpansionFactor;
      sy /= ExpansionFactor;
    }

    byte r, g, b;

    if(showTempMap)
    {
      const int MinTemp = -18+128, MidTemp = 10+128, MaxTemp = 49+128;
      const int c1 = (MinTemp + MidTemp) / 2, c2 = MidTemp, c3 = (MidTemp + MaxTemp) / 2;
      int temp = tempMap[sx, sy];

      if(temp < c1) { r = 0; b = 255; g = BlendColor(0, 255, MinTemp, c1, temp); } // blue to cyan
      else if(temp < c2) { r = 0; g = 255; b = BlendColor(255, 0, c1, c2, temp); } // cyan to green
      else if(temp < c3) { g = 255; b = 0; r = BlendColor(0, 255, c2, c3, temp); } // green to yellow
      else { r = 255; b = 0; g = BlendColor(255, 0, c3, MaxTemp, temp); } // yellow to red
    }
    else if(showRainMap)
    {
      const int MinRain = 0, MidRain = 100, MaxRain = 200;
      const int c1 = (MinRain + MidRain) / 2, c2 = MidRain, c3 = (MidRain + MaxRain) / 2;
      int rain = Math.Min(MaxRain, (int)rainMap[sx, sy]);

      if(rain < c1) { BlendColor(128, 128, 255, 0, out r, out b, MinRain, c1, rain); g = r; } // gray to yellow
      else if(rain < c2) { g = 255; b = 0; r = BlendColor(255, 0, c1, c2, rain); } // yellow to green
      else if(rain < c3) { r = 0; g = 255; b = BlendColor(0, 255, c2, c3, rain); } // green to cyan
      else { r = 0; b = 255; g = BlendColor(255, 0, c3, MaxRain, rain); } // cyan to blue
    }
    else if(showDrainMap)
    {
      const int MinDrain = 0, MidDrain = 100, MaxDrain = 200;
      const int c1 = (MinDrain + MidDrain) / 2, c2 = MidDrain, c3 = (MidDrain + MaxDrain) / 2;
      int drain = Math.Min(MaxDrain, (int)drainMap[sx, sy]);

      if(drain < c1) { BlendColor(128, 128, 255, 0, out r, out b, MinDrain, c1, drain); g = r; } // gray to yellow
      else if(drain < c2) { g = 255; b = 0; r = BlendColor(255, 0, c1, c2, drain); } // yellow to green
      else if(drain < c3) { r = 0; g = 255; b = BlendColor(0, 255, c2, c3, drain); } // green to cyan
      else { r = 0; b = 255; g = BlendColor(255, 0, c3, MaxDrain, drain); } // cyan to blue
    }
    else
    {
      if(showOceanMap && oceanMap[sx, sy]) { r = b = 255; g = 0; }
      else if(height < WaterThreshold) { r = 0; BlendColor(0, 255, 0xaa, 0, out g, out b, 0, WaterThreshold, height); }
      else if(height < 155) BlendColor(0, 0xaa, 0, 0xcc, 0x99, 0, out r, out g, out b, WaterThreshold, 155, height);
      else BlendColor(0xcc, 0x99, 0, 0xdd, 0xdd, 0xdd, out r, out g, out b, 155, 255, height);

      if(extraWater != null)
      {
        int water = (int)(extraWater[x+1, y+1]*2 + 0.5f);
        if(water >= WaterThreshold) { r = g = 0; b = 255; }
        else BlendColor(r, g, b, 0, 0, 255, out r, out g, out b, 0, 32, water);
      }
    }

    GL.glColor3ub(r, g, b);
    GL.glVertex3f((x-_halfSize)*_mul, (_halfSize-y)*_mul, height*HeightScale);
  }

  static bool doHF = true;
  static double GetHeightValueFactor(double n)
  {
    if(!doHF) return 1;

    if(n < 0) n = -n;
    if(n > 1) n = 1;

    return 0.221789 + 0.269516*n + 0.951876*(n*n);
  }

  static double GetInitialFrequency(int variance)
  {
    if(variance == 0) return 0;
    else
    {
      // this cubic polynomial approximately maps 100 -> 1/8, 75 -> 1/11.25, 50 -> 1/16, 25 -> 1/32, and 1 -> 1/64
      double v = variance * 0.01;
      return 0.0145306 + 0.0562843*v + 0.0813575*(v*v) - 0.0276583*(v*v*v);
    }
  }

  static double GetInitialHeightFrequency(int size, int variance)
  {
    if(variance == 0) return 0;
    else
    {
      double freq = GetInitialFrequency(variance);
      if(size > 128)
      {
        double lowerfreq = freq * 0.75;
        freq = freq + (size-128)*(1.0/128)*(lowerfreq - freq);
      }
      return freq;
    }
  }

  static double GetOceanScale(int size, OceanSide oceanSides, int x, int y)
  {
    double scale, quarterSize = size * 0.25;

    if((oceanSides & OceanSide.Left)  != 0 && x < quarterSize ||
       (oceanSides & OceanSide.Right) != 0 && x > size-quarterSize)
    {
      double dx = (x <= quarterSize ? quarterSize - x : x - (size-quarterSize)) / (double)quarterSize;
      scale = 2*(dx*dx*dx) - 3*(dx*dx) + 1;
    }
    else scale = 1;

    if((oceanSides & OceanSide.Top) != 0 && y < quarterSize ||
       (oceanSides & OceanSide.Bottom) != 0 && y > size-quarterSize)
    {
      double dy = (y <= quarterSize ? quarterSize - y : y - (size-quarterSize)) / (double)quarterSize;
      scale *= 2*(dy*dy*dy) - 3*(dy*dy) + 1;
    }

    return scale;
  }

  static bool doLighting=true;

  const int ExpansionFactor = 16;

  static void CreateHeightMap(WorldGeneratorOptions options, CoherentNoise noise, bool expanded)
  {
    int size = expanded ? options.Size*ExpansionFactor : options.Size, chunkSize = size / CPUCount;

    // choose some ocean edges at random
    OceanSide oceanSides = 0;
    for(int i=0; i<options.OceanEdges; i++)
    {
      OceanSide mask;
      do
      {
        mask = (OceanSide)(1 << rand.Next(0, 4));
      } while((oceanSides & mask) != 0);
      oceanSides |= mask;
    }

    heightMap = new float[size, size];

    // split the work across the system's CPUs
    Thread[] extraThreads = new Thread[CPUCount-1];
    for(int i=0; i<extraThreads.Length; i++)
    {
      extraThreads[i] = new Thread((ParameterizedThreadStart)delegate(object start)
                          { CreateHeightMap((int)start*chunkSize, chunkSize, options, noise, oceanSides, expanded); });
      extraThreads[i].Start(i);
    }

    CreateHeightMap(chunkSize*(CPUCount-1), size-chunkSize*(CPUCount-1), options, noise, oceanSides, expanded);

    for(int i=0; i<extraThreads.Length; i++)
    {
      extraThreads[i].Join();
    }
  }

  static int iters=6;
  static void CreateHeightMap(int xStart, int count, WorldGeneratorOptions options, CoherentNoise noise,
                              OceanSide oceanSides, bool expanded)
  {
    double elevScale  = (options.Elevation.Maximum - options.Elevation.Minimum) * 0.01;
    double elevOffset = options.Elevation.Minimum * 0.01;
    double xfreq = GetInitialHeightFrequency(options.Size, options.ElevationVariance.X);
    double yfreq = GetInitialHeightFrequency(options.Size, options.ElevationVariance.Y);
    int size = options.Size;

    if(expanded)
    {
      xfreq *= (1.0/ExpansionFactor);
      yfreq *= (1.0/ExpansionFactor);
      size  *= ExpansionFactor;
    }

    for(int x=xStart, xend=xStart+count; x<xend; x++)
    {
      for(int y=0; y<size; y++)
      {
        double oceanScale = GetOceanScale(size, oceanSides, x, y);
        double n = ((noise.Perlin(x*xfreq, y*yfreq) + 1) * 0.5 * elevScale + elevOffset) * oceanScale;

        double fmul=2, vmul = 0.5;
        for(int i=0; i<iters; fmul*=2, vmul*=0.5, i++)
        {
          double offset = noise.Perlin(x*(xfreq*fmul), y*(yfreq*fmul)) * vmul * oceanScale;
          if(i != 0) offset *= GetHeightValueFactor(n); // don't apply the height factor to the first iteration because
          n += offset;                                  // it tends to prevent large lakes from forming
        }

        if(n < 0) n = 0;
        else if(n > 1) n = 1;

        heightMap[x, y] = (float)(n*255);
      }
    }
  }

  static void CreateOceanMap(WorldGeneratorOptions options)
  {
    oceanMap = new bool[options.Size, options.Size];

    // detect water squares on the four sides and mark them as ocean
    for(int i=0; i<options.Size; i++)
    {
      FillOcean(i, 0, options.Size);
      FillOcean(i, options.Size-1, options.Size);
      FillOcean(0, i, options.Size);
      FillOcean(options.Size-1, i, options.Size);
    }
  }

  static void FillOcean(int x, int y, int size)
  {
    if(heightMap[x, y] >= WaterThreshold || oceanMap[x, y]) return;

    Stack<System.Drawing.Point> points = new Stack<System.Drawing.Point>();
    points.Push(new System.Drawing.Point(x, y));
    do
    {
      System.Drawing.Point pt = points.Pop();

      if(heightMap[pt.X, pt.Y] < WaterThreshold && !oceanMap[pt.X, pt.Y])
      {
        oceanMap[pt.X, pt.Y] = true;

        if(pt.X > 0)
        {
          if(pt.Y > 0) points.Push(new System.Drawing.Point(pt.X-1, pt.Y-1));
          points.Push(new System.Drawing.Point(pt.X-1, pt.Y));
          if(pt.Y < size-1) points.Push(new System.Drawing.Point(pt.X-1, pt.Y+1));
        }

        if(pt.X < size-1)
        {
          if(pt.Y > 0) points.Push(new System.Drawing.Point(pt.X+1, pt.Y-1));
          points.Push(new System.Drawing.Point(pt.X+1, pt.Y));
          if(pt.Y < size-1) points.Push(new System.Drawing.Point(pt.X+1, pt.Y+1));
        }
      }
    } while(points.Count != 0);
  }

  /// <summary>Given a temperature in degrees celsius, returns the temperature in degrees kelvin.</summary>
  static double GetKelvins(double celsius)
  {
    return celsius + 273.15;
  }

  /// <summary>Given air pressure in kPa, returns the steam point (boiling point) of water in degrees kelvin.</summary>
  static double GetSteamPointOfWater(double airPressure)
  {
    // adapted from http://en.wikipedia.org/wiki/Boiling_point (Clausius-Clapeyron equation)
    const double GasConst = 8.314472, HeatOfVaporization = 40650;
    const double LogSaturationPressureAt1Atm = 4.618329, BoilingPointAt1Atm = 373.12;

    return 1 / ((GasConst/HeatOfVaporization) * (LogSaturationPressureAt1Atm - Math.Log(airPressure)) +
                1/BoilingPointAt1Atm);
  }

  /// <summary>Given altitude in meters, returns the air pressure in kPa. Valid up to 11,000 meters.</summary>
  static double GetAirPressure(double altitudeInMeters)
  {
    // adapted from http://en.wikipedia.org/wiki/Atmospheric_pressure
    const double SeaLevelPressure = 101.325, StandardTemp = 288.15, Gravity = 9.80665, USGasConst = 8314.32;
    const double AirMolarMass = 28.9644, LapseRate = -0.0065;

    return SeaLevelPressure * Math.Pow(StandardTemp / (StandardTemp + LapseRate * altitudeInMeters),
                                       Gravity * AirMolarMass / (USGasConst * LapseRate));
  }

  /// <summary>Given a temperature in degrees Celsius, less than or equal to zero, returns the saturation pressure of
  /// water vapor within air at that temperature, over ice, in kPa.
  /// </summary>
  static double GetSaturationPressureOverIce(double temperature)
  {
    // adapted from http://en.wikipedia.org/wiki/Goff-Gratch_equation
    const double IcePoint = 273.15, PressureAtIcePoint = 0.61071; // temperature in K and pressure in kPa

    double kelvins = GetKelvins(temperature), tempRatio = IcePoint / kelvins;
    return Math.Pow(10, -9.09718 * (tempRatio-1) - 3.56654 * Math.Log10(tempRatio) +
                        0.876793 * (1 - kelvins * (1/IcePoint))) * PressureAtIcePoint;
  }

  /// <summary>Given air temperature in degrees Celsius and an altitude in meters, returns the saturation pressure of
  /// water vapor within air at that temperature and altitude, over land, in kPa.
  /// </summary>
  static double GetSaturationPressureOverLand(double temperature, double altitudeInMeters)
  {
    // the formula below doesn't take into account altitude, and so doesn't work well at elevations above 5000 meters
    // or so. it's probably better to just use the method for over water
    return GetSaturationPressureOverWater(temperature, altitudeInMeters);

    // adapted from http://en.wikipedia.org/wiki/Boiling_point (Clausius-Clapeyron equation)
    /*const double GasConst = 8.314472, HeatOfVaporization = 40650;
    const double LogSaturationPressureAt1Atm = 4.618329, BoilingPointAt1Atm = 373.12;

    return Math.Exp(HeatOfVaporization/GasConst * (1/BoilingPointAt1Atm - 1/GetKelvins(temperature)) +
                    LogSaturationPressureAt1Atm);*/
  }

  /// <summary>Given air temperature in degrees Celsius and an altitude in meters, returns the saturation pressure of
  /// water vapor within air at that temperature and altitude, over water, in kPa.
  /// </summary>
  static double GetSaturationPressureOverWater(double temperature, double altitudeInMeters)
  {
    // adapted from http://en.wikipedia.org/wiki/Goff-Gratch_equation
    const double SteamPoint = 373.15, PressureAtSteamPoint = 101.3246;
    double kelvins = GetKelvins(temperature), tempRatio = SteamPoint / kelvins;
    return Math.Pow(10, -7.90298 * (tempRatio-1) + 5.02808 * Math.Log10(tempRatio) -
                        (1.3816/10000000) * (Math.Pow(10, 11.344 * (1 - kelvins * (1/SteamPoint))) - 1) +
                        (8.1328/1000) * (Math.Pow(10, -3.49149 * (tempRatio - 1)) - 1)) * PressureAtSteamPoint;
  }

  static double GetSaturationPressure(int x, int y)
  {
    int temperature = tempMap[x, y] - 128;
    float height = heightMap[x, y];
    double altitude = GetHeightInMeters(height);

    return temperature <= 0 ? GetSaturationPressureOverIce(temperature) :
           height < WaterThreshold ? GetSaturationPressureOverWater(temperature, altitude) :
           GetSaturationPressureOverLand(temperature, altitude);
  }

  static void CreateDrainageMap(WorldGeneratorOptions options, CoherentNoise noise)
  {
    const double BaseDrainage = 0.5; // we'll use a base humidity of 50%

    drainMap = new byte[options.Size, options.Size];

    double xfreq = GetInitialFrequency(options.DrainageVariance.X);
    double yfreq = GetInitialFrequency(options.DrainageVariance.Y);
    double xVarianceScale = options.DrainageVariance.X * (1/60.0), qxv = xVarianceScale * 0.5;
    double yVarianceScale = options.DrainageVariance.Y * (1/60.0), qyv = yVarianceScale * 0.5;

    for(int y=0; y<options.Size; y++)
    {
      double yScale = (noise.Perlin(y*yfreq) * qyv + 1);

      for(int x=0; x<options.Size; x++)
      {
        double scale = (noise.Perlin(x*xfreq, y*xfreq) * qxv + 1) * yScale;
        drainMap[x, y] = (byte)Math.Round(scale * BaseDrainage * 255);
      }
    }
  }

  static void CreatePrecipitationMap(WorldGeneratorOptions options, CoherentNoise noise)
  {
    const double BaseHumidity = 0.5; // we'll use a base humidity of 50%

    rainMap = new byte[options.Size, options.Size];

    // calculate the base relative humidity in each tile
    double xfreq = GetInitialFrequency(options.RainVariance.X);
    double yfreq = GetInitialFrequency(options.RainVariance.Y);
    double xVarianceScale = options.RainVariance.X * (1/60.0), qxv = xVarianceScale * 0.5;
    double yVarianceScale = options.RainVariance.Y * (1/60.0), qyv = yVarianceScale * 0.5;

    for(int y=0; y<options.Size; y++)
    {
      double yScale = (noise.Perlin(y*yfreq) * qyv + 1);

      for(int x=0; x<options.Size; x++)
      {
        double scale = (noise.Perlin(x*xfreq, y*xfreq) * qxv + 1) * yScale;
        rainMap[x, y] = (byte)Math.Round(scale * BaseHumidity * 100);
      }
    }

    // now blow breezes across the map to simulate orographic precipitation and rain shadows
    for(int i=0; i<options.Size; i++)
    {
      BlowRainBreezes(options, i, 0, 0, 1);
      BlowRainBreezes(options, i, options.Size-1, 0, -1);
      BlowRainBreezes(options, 0, i, 1, 0);
      BlowRainBreezes(options, options.Size-1, i, -1, 0);
    }
  }

  static double GetHeightInMeters(float height)
  {
    // heights above the water threshold scale from 0 to 8000 meters
    return height <= WaterThreshold ? 0 : (height-WaterThreshold) * (8000.0/(255-WaterThreshold));
  }

  static void BlowRainBreezes(WorldGeneratorOptions options, int x, int y, int xo, int yo)
  {
    double prevPressure = rainMap[x, y] * 0.01 * GetSaturationPressure(x, y), falloff = 1 / (options.Size * 0.4);
    do
    {
      double saturationPressure = GetSaturationPressure(x, y), pressure = rainMap[x, y] * 0.01 * saturationPressure;

      if(oceanMap[x, y]) // set relative humidity over ocean tiles to at least 90%
      {
        prevPressure = pressure = Math.Max(pressure, 0.9 * saturationPressure);
      }
      else if(heightMap[x, y] < WaterThreshold) // interpolate between current pressure and 90% over other water tiles
      {
        pressure = Math.Max(pressure, pressure + 0.25 * (0.9*saturationPressure - pressure));
      }

      pressure = prevPressure + falloff * (pressure - prevPressure);
      rainMap[x, y] = (byte)options.Rainfall.Clip((int)Math.Round(pressure*100 / saturationPressure));
      prevPressure = Math.Min(pressure, saturationPressure);

      x += xo;
      y += yo;
    } while(x >= 0 && y >= 0 && x < options.Size && y < options.Size);
  }

  static void CreateTemperatureMap(WorldGeneratorOptions options, CoherentNoise noise)
  {
    tempMap = new byte[options.Size, options.Size];

    double xfreq = GetInitialFrequency(options.TemperatureVariance.X);
    double yfreq = GetInitialFrequency(options.TemperatureVariance.Y);
    double xVarianceScale = options.TemperatureVariance.X * (1/50.0), qxv = xVarianceScale * 0.5;
    double yVarianceScale = options.TemperatureVariance.Y * (1/65.0), qyv = yVarianceScale * 0.5;

    for(int y=0; y<options.Size; y++)
    {
      // produce a curve to make the temperature colder at the poles and warmer in the center
      double yScale = Math.Sin((double)y/options.Size * Math.PI);
      yScale = yScale * yVarianceScale + (1 - yVarianceScale);  // reduce the effect of the sine wave based on yVarianceScale
      yScale *= noise.Perlin(y*yfreq) * qyv + 1; // add some noise to it

      for(int x=0; x<options.Size; x++)
      {
        double scale = (noise.Perlin(x*xfreq,y*xfreq) * qxv + 1) * yScale;

        float height = heightMap[x, y];
        if(height < WaterThreshold-2) // reduce temperature variation over deep water
        {
          // move it 50% of the way towards the average temperature (or 2/3 of the average for ocean)
          scale = (oceanMap[x, y] ? (1.0/3) : 0.5) + 0.5*scale;
        }

        // reduce the temperature at higher elevations (> 1000 m)
        if(height > 30) scale = scale * (0.8/(255-30)) * (255-height) + 0.2;

        double temperature = options.Temperature.Minimum + scale * 0.5 * (options.Temperature.Difference);
        tempMap[x, y] = (byte)(options.Temperature.Clip((int)Math.Round(temperature)) + 128);
      }
    }

    // now simulate the effect of ocean breezes
    for(int i=0; i<options.Size; i++)
    {
      BlowTemperatureBreezes(i, 0, 0, 1, options.Size);
      BlowTemperatureBreezes(i, options.Size-1, 0, -1, options.Size);
      BlowTemperatureBreezes(0, i, 1, 0, options.Size);
      BlowTemperatureBreezes(options.Size-1, i, -1, 0, options.Size);
    }
  }

  static void BlowTemperatureBreezes(int x, int y, int xo, int yo, int size)
  {
    if(!oceanMap[x, y]) return;

    int temp = tempMap[x, y], points = 10;
    do
    {
      int tempDiff = temp - tempMap[x, y];

      if(tempDiff < 0)
      {
        tempDiff = Math.Max(-points, (tempDiff-1)/2);
        tempMap[x, y] += (byte)tempDiff;
        points += (tempDiff-1) / 2;
        temp   -= tempDiff / 2;
      }
      else if(tempDiff > 0)
      {
        tempDiff = Math.Min(points, (tempDiff+1)/2);
        tempMap[x, y] += (byte)tempDiff;
        points -= (tempDiff+1) / 2;
        temp   -= tempDiff / 2;
      }

      if(oceanMap[x, y]) points = Math.Min(50, points+2);

      x += xo;
      y += yo;

    } while(points != 0 && x >= 0 && y >= 0 && x < size && y < size);
  }

  static float GetSoilSoftness(float altitude)
  {
    return altitude * (0.25f * 1.0f / 255);
  }

  static float[,] extraWater;
  static bool runRivers;
  static void ErodeTerrain(WorldGeneratorOptions options)
  {
    int size = heightMap.GetLength(0);
    
    // we'll create the arrays with padding on the edges so we can avoid many bounds checks
    float[,] altitude = new float[size+2, size+2], sediment = new float[size+2, size+2];
    float[,] water = new float[size+2, size+2];
    extraWater = water;

    for(int x=1; x <= size; x++)
    {
      for(int y=1; y <= size; y++) altitude[x, y] = heightMap[x-1, y-1];
      
      // copy the altitude measurements on the edges to the adjecent padding cells
      altitude[0, x] = altitude[1, x];
      altitude[size+1, x] = altitude[size, x];
      altitude[x, 0] = altitude[x, 1];
      altitude[x, size+1] = altitude[x, size];
    }

    // and set the four padding corners as well
    altitude[0,      0]      = altitude[1,    1];
    altitude[0,      size+1] = altitude[1,    size];
    altitude[size+1, 0]      = altitude[size, 1];
    altitude[size+1, size+1] = altitude[size, size];

    for(int i=0; i<options.ErosionCycles; i++)
    {
      DoHydraulicErosion(altitude, sediment, water, size, i < options.ErosionCycles/2);
      DoThermalErosion(altitude, size);
      if((i & 15) == 0) RenderErosion(altitude, size);
    }

    for(int i=0; i<options.ErosionCycles; i++)
    {
      DoHydraulicErosion(altitude, sediment, water, size, false);
      if((i & 15) == 0) RenderErosion(altitude, size);
    }

    for(int i=0; i<5; i++)
    {
      DoThermalErosion(altitude, size);
    }

    if(runRivers)
    {
      for(int i=0; i < options.ErosionCycles*2; i++)
      {
        RunRivers(altitude, water, size, i < options.ErosionCycles);

        if((i&15) == 0)
        {
          CreateQuads();
          Render();
        }
      }
    }
  }

  static void DoHydraulicErosion(float[,] altitude, float[,] sediment, float[,] water, int size, bool addWater)
  {
    const float DepositionFraction = 0.1f, RainFactor = 0.01f;

    for(int x=1; x <= size; x++)
    {
      for(int y=1; y <= size; y++)
      {
        // drop water on the vertex
        if(addWater) water[x,y] += rainMap[(x-1)/ExpansionFactor, (y-1)/ExpansionFactor] * (RainFactor/100);
        
        float softness = GetSoilSoftness(altitude[x,y]), oav = water[x, y] + altitude[x,y]; // oav is overall altitude at the vertex

        // calculate the overall altitude differences of the surrounding vertices
        float oad0 = oav - water[x-1,y-1] - altitude[x-1,y-1], oad1 = oav - water[x-1,y]   - altitude[x-1,y],
              oad2 = oav - water[x-1,y+1] - altitude[x-1,y+1], oad3 = oav - water[x,  y-1] - altitude[x,  y-1],
              oad4 = oav - water[x,  y+1] - altitude[x,  y+1], oad5 = oav - water[x+1,y-1] - altitude[x+1,y-1],
              oad6 = oav - water[x+1,y]   - altitude[x+1,y],   oad7 = oav - water[x+1,y+1] - altitude[x+1,y+1];

        // calculate the total positive difference so we can move water and sediment in proper proportion
        float itotalOAD;
        itotalOAD = oad0 > 0 ? oad0 : 0;
        if(oad1 > 0) itotalOAD += oad1;
        if(oad2 > 0) itotalOAD += oad2;
        if(oad3 > 0) itotalOAD += oad3;
        if(oad4 > 0) itotalOAD += oad4;
        if(oad5 > 0) itotalOAD += oad5;
        if(oad6 > 0) itotalOAD += oad6;
        if(oad7 > 0) itotalOAD += oad7;
        if(itotalOAD > 0) itotalOAD = 1 / itotalOAD;

        // now actually move the water and sediment
        // TODO: move some of these parameters into an object that can be passed by reference?
        float origSediment = sediment[x, y];
        MoveWater(altitude, sediment, water, x, y, -1, -1, oad0, itotalOAD, origSediment, softness);
        MoveWater(altitude, sediment, water, x, y, -1,  0, oad1, itotalOAD, origSediment, softness);
        MoveWater(altitude, sediment, water, x, y, -1,  1, oad2, itotalOAD, origSediment, softness);
        MoveWater(altitude, sediment, water, x, y,  0, -1, oad3, itotalOAD, origSediment, softness);
        MoveWater(altitude, sediment, water, x, y,  0,  1, oad4, itotalOAD, origSediment, softness);
        MoveWater(altitude, sediment, water, x, y,  1, -1, oad5, itotalOAD, origSediment, softness);
        MoveWater(altitude, sediment, water, x, y,  1,  0, oad6, itotalOAD, origSediment, softness);
        MoveWater(altitude, sediment, water, x, y,  1,  1, oad7, itotalOAD, origSediment, softness);

        // now deposit some of whatever sediment remains
        float sedimentToDeposit = sediment[x,y] * DepositionFraction;
        altitude[x, y] += sedimentToDeposit;
        sediment[x, y] -= sedimentToDeposit;
      }
    }
  }

  static void MoveWater(float[,] altitude, float[,] sediment, float[,] water,
                        int x, int y, int xo, int yo, float oad, float itotalOAD, float origSediment, float softness)
  {
    const float SedimentCapacityFactor = 5;

    float wv = water[x, y], wd = wv < oad ? wv : oad;
    if(wd > 0)
    {
      float fraction = oad * itotalOAD;

      wd *= fraction;
      water[x, y]       -= wd;
      water[x+xo, y+yo] += wd;

      float sv = origSediment * fraction, sedCapacity = wd * SedimentCapacityFactor;
      if(sv < sedCapacity)
      {
        float soilConverted   = softness * (sedCapacity - sv);
        sediment[x+xo, y+yo] += sv + soilConverted;
        altitude[x, y]       -= soilConverted;
        sediment[x, y]       -= sv;
      }
      else
      {
        sediment[x+xo, y+yo] += sedCapacity;
        sediment[x, y]       -= sedCapacity;
      }
    }
  }

  const float TalusAngle = 4f;

  static void DoThermalErosion(float[,] altitude, int size)
  {
    for(int x=1; x <= size; x++)
    {
      for(int y=1; y <= size; y++)
      {
        float av = altitude[x,y];

        // calculate the altitude differences of the surrounding vertices
        float ad0 = av - altitude[x-1,y-1], ad1 = av - altitude[x-1,y],
              ad2 = av - altitude[x-1,y+1], ad3 = av - altitude[x,  y-1],
              ad4 = av - altitude[x,  y+1], ad5 = av - altitude[x+1,y-1],
              ad6 = av - altitude[x+1,y],   ad7 = av - altitude[x+1,y+1];

        // calculate the total positive difference so we can move rock in proper proportion
        float itotalAD;
        itotalAD = ad0 > TalusAngle ? ad0 : 0;
        if(ad1 > TalusAngle) itotalAD += ad1;
        if(ad2 > TalusAngle) itotalAD += ad2;
        if(ad3 > TalusAngle) itotalAD += ad3;
        if(ad4 > TalusAngle) itotalAD += ad4;
        if(ad5 > TalusAngle) itotalAD += ad5;
        if(ad6 > TalusAngle) itotalAD += ad6;
        if(ad7 > TalusAngle) itotalAD += ad7;
        if(itotalAD > 0) itotalAD = 1 / itotalAD;

        // now actually move the rock
        MoveRock(altitude, x, y, -1, -1, ad0, itotalAD);
        MoveRock(altitude, x, y, -1,  0, ad1, itotalAD);
        MoveRock(altitude, x, y, -1,  1, ad2, itotalAD);
        MoveRock(altitude, x, y,  0, -1, ad3, itotalAD);
        MoveRock(altitude, x, y,  0,  1, ad4, itotalAD);
        MoveRock(altitude, x, y,  1, -1, ad5, itotalAD);
        MoveRock(altitude, x, y,  1,  0, ad6, itotalAD);
        MoveRock(altitude, x, y,  1,  1, ad7, itotalAD);
      }
    }
  }

  static void MoveRock(float[,] altitude, int x, int y, int xo, int yo, float ad, float itotalAD)
  {
    const float ErosionFactor = 0.1f;

    if(ad > TalusAngle)
    {
      float toMove = (ad - TalusAngle) * (ad * itotalAD) * ErosionFactor;
      altitude[x, y] -= toMove;
      altitude[x+xo, y+yo] += toMove;
    }
  }

  static void CreateErosionHeightmap(float[,] altitude, int size)
  {
    for(int x=1; x <= size; x++)
    {
      for(int y=1; y <= size; y++)
      {
        float newHeight = altitude[x, y];
        heightMap[x-1, y-1] = (newHeight < 0 ? 0 : newHeight > 255 ? 255 : newHeight);
      }
    }
  }

  static void RenderErosion(float[,] altitude, int size)
  {
    CreateErosionHeightmap(altitude, size);
    CreateQuads();
    Render();
  }

  static void RunRivers(float[,] altitude, float[,] water, int size, bool addWater)
  {
    const float RainFactor = 0.04f;

    for(int x=1; x <= size; x++)
    {
      for(int y=1; y <= size; y++)
      {
        // drop water on the vertex
        if(addWater) water[x,y] += rainMap[(x-1)/ExpansionFactor, (y-1)/ExpansionFactor] * (RainFactor/100);
        
        float oav = water[x, y] + altitude[x,y]; // oav is overall altitude at the vertex

        // calculate the overall altitude differences of the surrounding vertices
        float oad0 = oav - water[x-1,y-1] - altitude[x-1,y-1], oad1 = oav - water[x-1,y]   - altitude[x-1,y],
              oad2 = oav - water[x-1,y+1] - altitude[x-1,y+1], oad3 = oav - water[x,  y-1] - altitude[x,  y-1],
              oad4 = oav - water[x,  y+1] - altitude[x,  y+1], oad5 = oav - water[x+1,y-1] - altitude[x+1,y-1],
              oad6 = oav - water[x+1,y]   - altitude[x+1,y],   oad7 = oav - water[x+1,y+1] - altitude[x+1,y+1];

        // calculate the total positive difference so we can move water and sediment in proper proportion
        float itotalOAD;
        itotalOAD = oad0 > 0 ? oad0 : 0;
        if(oad1 > 0) itotalOAD += oad1;
        if(oad2 > 0) itotalOAD += oad2;
        if(oad3 > 0) itotalOAD += oad3;
        if(oad4 > 0) itotalOAD += oad4;
        if(oad5 > 0) itotalOAD += oad5;
        if(oad6 > 0) itotalOAD += oad6;
        if(oad7 > 0) itotalOAD += oad7;
        if(itotalOAD > 0) itotalOAD = 1 / itotalOAD;

        MoveWater(water, x, y, -1, -1, oad0, itotalOAD);
        MoveWater(water, x, y, -1,  0, oad1, itotalOAD);
        MoveWater(water, x, y, -1,  1, oad2, itotalOAD);
        MoveWater(water, x, y,  0, -1, oad3, itotalOAD);
        MoveWater(water, x, y,  0,  1, oad4, itotalOAD);
        MoveWater(water, x, y,  1, -1, oad5, itotalOAD);
        MoveWater(water, x, y,  1,  0, oad6, itotalOAD);
        MoveWater(water, x, y,  1,  1, oad7, itotalOAD);
      }
    }
  }

  static void MoveWater(float[,] water, int x, int y, int xo, int yo, float oad, float itotalOAD)
  {
    float wv = water[x, y], wd = wv < oad ? wv : oad;
    if(wd > 0)
    {
      wd *= oad * itotalOAD;
      water[x, y]       -= wd;
      water[x+xo, y+yo] += wd;
    }
  }
  static void GenerateTerrain(WorldGeneratorOptions options)
  {
    if(options.TerrainSeed == 0) options.TerrainSeed = new Random().Next();
    rand = new Random(options.TerrainSeed);

    CoherentNoise noise = new CoherentNoise();
    
    noise.Reseed(rand);
    CreateHeightMap(options, noise, false);
    
    CreateOceanMap(options);

    noise.Reseed(rand);
    CreateTemperatureMap(options, noise);

    noise.Reseed(rand);
    CreatePrecipitationMap(options, noise);

    noise.Reseed(rand);
    CreateDrainageMap(options, noise);

    rand = new Random(options.TerrainSeed);
    noise.Reseed(rand);
    if(doExpand)
    {
      CreateHeightMap(options, noise, true);
      renderMap  = heightMap;
      extraWater = null;
      if(options.ErosionCycles != 0) ErodeTerrain(options);
    }

    //renderMap = tempMap;
    //renderMap = rainMap;
    renderMap = heightMap;

    CreateQuads();
  }

  static bool doExpand;
  static float[,] renderMap;
  static int quality;

  static void CreateQuads()
  {
    int divisor = quality == 0 ? 128 : quality == 1 ? 256 : quality == 2 ? 512 : quality == 3 ? 1024 : 2048;

    quads.Clear();
    CreateQuads(0, 0, renderMap.GetLength(0)-1, renderMap.GetLength(1)-1, Math.Max(1, renderMap.GetLength(0)/divisor));

    float halfSize = renderMap.GetLength(0) * 0.5f, mul = doExpand ? 1.0f/8 : 2;
    for(int i=0; i<quads.Count; i++)
    {
      Quad quad = quads[i];
      Vector v1 = new Vector((quad.X-halfSize)*mul, (quad.Y-halfSize)*mul, renderMap[quad.X, quad.Y]*HeightScale);
      Vector v2 = new Vector((quad.X2-halfSize)*mul, (quad.Y-halfSize)*mul, renderMap[quad.X2, quad.Y]*HeightScale);
      Vector v3 = new Vector((quad.X-halfSize)*mul, (quad.Y2-halfSize)*mul, renderMap[quad.X, quad.Y2]*HeightScale);
      Vector n  = (v2-v1).CrossProduct(v3-v1).Normal;
      quad.NX = (float)n.X;
      quad.NY = (float)n.Y;
      quad.NZ = (float)n.Z;
      quads[i] = quad;
    }
  }

  static int GetHeight(int x, int y)
  {
    /*float height = renderMap[x,y];
    if(extraWater != null) height += extraWater[x+1,y+1];
    return (int)(height + 0.5f);*/
    return (int)(renderMap[x, y]+0.5f);
  }

  static void CreateQuads(int x1, int y1, int x2, int y2, int threshold)
  {
    int xd = x2-x1, yd = y2-y1;

    int height = GetHeight(x1, y1);
    bool isFlat = height == GetHeight(x2, y1) && height == GetHeight(x1, y2) && height == GetHeight(x2, y2);

    for(int x=x1; x <= x2 && isFlat; x++)
    {
      for(int y=y1; y <= y2; y++)
      {
        if(GetHeight(x, y) != height)
        {
          isFlat = false;
          break;
        }
      }
    }

    // if the heights are equal, or the space we're considering is really small, create the quad
    if(isFlat || xd <= threshold || yd <= threshold)
    {
      quads.Add(new Quad(x1, y1, x2, y2));
    }
    else // otherwise, call ourselves recursively with a smaller space
    {
      int midx = x1 + xd/2, midy = y1 + yd/2;
      CreateQuads(x1, y1, midx, midy, threshold);
      CreateQuads(midx, y1, x2, midy, threshold);
      CreateQuads(x1, midy, midx, y2, threshold);
      CreateQuads(midx, midy, x2, y2, threshold);
    }
  }

  static Random rand = new Random();

  static void Main()
  {
    Events.Initialize();
    Video.Initialize();
    GameLib.Input.Keyboard.EnableKeyRepeat(200, 25);
    InitOpenGL();

    WorldGeneratorOptions options = new WorldGeneratorOptions();
    options.TerrainSeed = 44501;
    options.Size = 32;
    options.OceanEdges = 2;
    options.ErosionCycles = 0;
    GenerateTerrain(options);

    Render();

    Events.PumpEvents(delegate(Event e)
    {
      if(e.Type == EventType.Repaint) Render();
      else if(e.Type == EventType.Keyboard)
      {
        KeyboardEvent ke = (KeyboardEvent)e;
        if(ke.Down)
        {
          if(ke.Key == GameLib.Input.Key.Escape) return false;
          else if(ke.Key == GameLib.Input.Key.Left)
          {
            if(ke.HasAny(GameLib.Input.KeyMod.Alt)) xoff -= 1f;
            else zangle -= 4f;
          }
          else if(ke.Key == GameLib.Input.Key.Right)
          {
            if(ke.HasAny(GameLib.Input.KeyMod.Alt)) xoff += 1f;
            else zangle += 4f;
          }
          else if(ke.Key == GameLib.Input.Key.Up)
          {
            if(ke.HasAny(GameLib.Input.KeyMod.Alt)) yoff += 1f;
            else if(ke.HasAny(GameLib.Input.KeyMod.Ctrl)) zoom += 1f;
            else yangle -= 3f;
          }
          else if(ke.Key == GameLib.Input.Key.Down)
          {
            if(ke.HasAny(GameLib.Input.KeyMod.Alt)) yoff -= 1f;
            else if(ke.HasAny(GameLib.Input.KeyMod.Ctrl)) zoom -= 1f;
            else yangle += 3f;
          }
          else if(ke.Char == 'o') showOceanMap = !showOceanMap;
          else if(ke.Char == 't')
          {
            showTempMap = !showTempMap;
            showDrainMap = showRainMap = false;
          }
          else if(ke.Char == 'p')
          {
            showRainMap = !showRainMap;
            showDrainMap = showTempMap = false;
          }
          else if(ke.Char == 'd')
          {
            showDrainMap = !showDrainMap;
            showRainMap = showTempMap = false;
          }
          else if(ke.Char == 'r')
          {
            xoff = yoff = zangle = 0;
            yangle = -30;
            zoom = -50;
          }
          else if(ke.Char == 'g')
          {
            options.TerrainSeed = 0;
            GenerateTerrain(options);
          }
          else if(ke.Char == 'e')
          {
            doExpand = !doExpand;
            GenerateTerrain(options);
          }
          else if(ke.Char == 'l')
          {
            doLighting = !doLighting;
          }
          else if(ke.Char == 'h')
          {
            doHF = !doHF;
            GenerateTerrain(options);
          }
          else if(ke.Char == '1' || ke.Char == '2' || ke.Char == '3' || ke.Char == '4' || ke.Char == '5')
          {
            if(!ke.HasAny(GameLib.Input.KeyMod.Alt))
            {
              quality = ke.Char - '1';
              CreateQuads();
            }
            else if(ke.Char != '5')
            {
              options.Size = (int)Math.Pow(2, ke.Char - '1' + 4);
              GenerateTerrain(options);
            }
          }
          else if(ke.Char == '6' || ke.Char == '7' || ke.Char == '8' || ke.Char == '9' || ke.Char == '0')
          {
            iters = (ke.Char == '0' ? 4 : ke.Char - '6') + (ke.HasAny(GameLib.Input.KeyMod.Alt) ? 5 : 0);
            GenerateTerrain(options);
          }
          else if(ke.Char == '-')
          {
            if(options.ErosionCycles != 0)
            {
              options.ErosionCycles = 0;
              GenerateTerrain(options);
            }
          }
          else if(ke.Char == '=')
          {
            options.ErosionCycles += 50;
            GenerateTerrain(options);
          }
          else if(ke.Char == 'R')
          {
            runRivers = !runRivers;
            GenerateTerrain(options);
          }
          Render();
        }
      }
      return e.Type != EventType.Quit;
    });

    Video.Deinitialize();
    Events.Deinitialize();
  }
}

} // namespace DwarfDungeon