/*
TerrainGen is a prototype for a terrain generator.
http://www.adammil.net/
Copyright (C) 2008 Adam Milazzo

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

using System;

namespace TerrainGen
{

#region SmallRange
/// <summary>A small range type used by <see cref="WorldGeneratorOptions"/>.</summary>
struct SmallRange
{
  /// <summary>Initializes a new <see cref="SmallRange"/> with values from -128 to 127.</summary>
  public SmallRange(int min, int max)
  {
    this.min = this.max = 0; // prevent a compiler error saying that all fields must be assigned to...
    SetRange(min, max);
  }

  /// <summary>Gets the difference between the maximum and minimum values.</summary>
  public int Difference
  {
    get { return Maximum - Minimum; }
  }

  /// <summary>Gets the upper end of the range, inclusive.</summary>
  public int Maximum
  {
    get { return max; }
  }

  /// <summary>Gets the lower end of the range, inclusive.</summary>
  public int Minimum
  {
    get { return min; }
  }

  /// <summary>Clips the given value to the range and returns it.</summary>
  public int Clip(int value)
  {
    return value < min ? min : value > max ? max : value;
  }

  /// <summary>Sets the upper and lower ends of the range, inclusive, with values from -128 to 127.</summary>
  public void SetRange(int min, int max)
  {
    if(min > max) throw new ArgumentException("The minimum must be less than or equal to the maximum.");
    if(min < sbyte.MinValue || max > sbyte.MaxValue)
    {
      throw new ArgumentOutOfRangeException("SimpleRange values must be from -128 to 127.");
    }

    this.min = (sbyte)min;
    this.max = (sbyte)max;
  }

  sbyte min, max;
}
#endregion

#region SmallUnsignedRange
/// <summary>A small range type used by <see cref="WorldGeneratorOptions"/>.</summary>
struct SmallUnsignedRange
{
  /// <summary>Initializes a new <see cref="SmallRange"/> with values from 0 to 255.</summary>
  public SmallUnsignedRange(int min, int max)
  {
    this.min = this.max = 0; // prevent a compiler error saying that all fields must be assigned to...
    SetRange(min, max);
  }

  /// <summary>Gets the difference between the maximum and minimum values.</summary>
  public int Difference
  {
    get { return Maximum - Minimum; }
  }

  /// <summary>Gets the upper end of the range, inclusive.</summary>
  public int Maximum
  {
    get { return max; }
  }

  /// <summary>Gets the lower end of the range, inclusive.</summary>
  public int Minimum
  {
    get { return min; }
  }

  /// <summary>Clips the given value to the range and returns it.</summary>
  public int Clip(int value)
  {
    return value < min ? min : value > max ? max : value;
  }

  /// <summary>Sets the upper and lower ends of the range, inclusive, with values from -128 to 127.</summary>
  public void SetRange(int min, int max)
  {
    if(min > max) throw new ArgumentException("The minimum must be less than or equal to the maximum.");
    if(min < 0 || max > 255) throw new ArgumentOutOfRangeException("SimpleRange values must be from 0 to 255.");

    this.min = (byte)min;
    this.max = (byte)max;
  }

  byte min, max;
}
#endregion

#region Variance
/// <summary>A type used by <see cref="WorldGeneratorOptions"/> to describe the variance of terrain features on the
/// X and Y axes.
/// </summary>
struct Variance
{
  /// <summary>Initializes a new <see cref="Variance"/> object with the given variances along the X and Y axes.</summary>
  public Variance(int x, int y)
  {
    this.x = this.y = 0; // prevent a compiler error saying that all fields must be assigned to...
    X = x;
    Y = y;
  }

  /// <summary>Gets or sets the variance along the X axis.</summary>
  public int X
  {
    get { return x; }
    set
    {
      Validate(value);
      x = (byte)value;
    }
  }

  /// <summary>Gets or sets the variance along the Y axis.</summary>
  public int Y
  {
    get { return y; }
    set
    {
      Validate(value);
      y = (byte)value;
    }
  }

  byte x, y;

  static void Validate(int value)
  {
    if(value < 0 || value > 100) throw new ArgumentOutOfRangeException("Variance values must be from 0 to 100.");
  }
}
#endregion

sealed class WorldGeneratorOptions
{
  public SmallRange Drainage
  {
    get { return drainage; }
    set
    {
      ValidatePercentRange(value, "Drainage");
      drainage = value;
    }
  }

  public Variance DrainageVariance
  {
    get { return drainVariance; }
    set { drainVariance = value; }
  }

  public SmallRange Elevation
  {
    get { return elevation; }
    set
    {
      ValidatePercentRange(value, "Elevation");
      elevation = value;
    }
  }

  public Variance ElevationVariance
  {
    get { return elevVariance; }
    set { elevVariance = value; }
  }

  public SmallUnsignedRange Rainfall
  {
    get { return rainfall; }
    set { rainfall = value; }
  }

  public Variance RainVariance
  {
    get { return rainVariance; }
    set { rainVariance = value; }
  }

  public SmallRange Temperature
  {
    get { return temperature; }
    set
    {
      if(value.Minimum < -100 || value.Maximum > 100)
      {
        throw new ArgumentOutOfRangeException("Temperature range must be within -100 and 100 degrees celsius.");
      }
      temperature = value;
    }
  }

  public Variance TemperatureVariance
  {
    get { return tempVariance; }
    set { tempVariance = value; }
  }

  public int ErosionCycles
  {
    get { return erosionCycles; }
    set
    {
      if(value < 0) throw new ArgumentOutOfRangeException("ErosionCycles cannot be negative.");
      erosionCycles = value;
    }
  }

  public int OceanEdges
  {
    get { return oceanEdges; }
    set
    {
      if(value < 0 || value > 4) throw new ArgumentOutOfRangeException("OceanEdges must be from 0 to 4.");
      oceanEdges = value;
    }
  }

  public int Size
  {
    get { return size; }
    set
    {
      if(value < 16 || value > 128) throw new ArgumentOutOfRangeException("Size must be from 16 to 128 tiles.");
      size = value;
    }
  }

  public int TerrainSeed
  {
    get { return terrainSeed; }
    set { terrainSeed = value; }
  }

  static void ValidatePercentRange(SmallRange range, string propertyName)
  {
    if(range.Minimum < 0 || range.Maximum > 100)
    {
      throw new ArgumentOutOfRangeException(propertyName + " range must be within 0 and 100 percent.");
    }
  }

  int size=64, terrainSeed, erosionCycles=200, oceanEdges=1;
  SmallRange elevation=new SmallRange(0, 100), temperature=new SmallRange(-10, 49), drainage=new SmallRange(0, 100);
  SmallUnsignedRange rainfall=new SmallUnsignedRange(0, 255);
  Variance elevVariance=new Variance(50, 50), rainVariance=new Variance(50, 50), tempVariance=new Variance(50, 50);
  Variance drainVariance=new Variance(50, 50);
}

} // namespace DwarfDungeon