ALIGN 4
FileHeaders FileHead MAXFILES dup(<>)

CurrentVSN  dd 0           ; serial number of the current disk (we hope)
FATCache    db 4608 dup(0) ; the fat cache itself
FATCached   db 0           ; is the FAT cache valid?
FATCacheDirty db 9 dup(0)  ; each flag is set if that fat block has changed

; GetFreeSpace - returns free space on disk
; INPUT:
; none
; OUTPUT:
; EAX - free space in bytes
; carry flag set on error
;
GetFreeSpace:
push  cx
push  dx
sub   cx,   cx       ; 0 bytes

mov   eax, FATCANUSE ; start search at beginning

@@loopstart:
call  ReadFATEntry   ; try to read that entry
jc    @@return       ; if error, quit
cmp   dx,   FATEMPTY ; if it's an empty record, increment count
jne   @@loopinc      ; else try next...
inc   cx
@@loopinc:
inc   ax             ; next entry
cmp   ax,   MAXFAT   ; if we're not done restart
jne   @@loopstart

@@return:
mov   ax,   cx
shl   eax,  9        ; multiply by size of cluster (512)
pop   dx
pop   cx
ret


; ----------------------------------------------------------------------------
; LoadFATCache - loads the fat cache from the disk.. doesnt write out previous
; INPUT:
; none
; OUTPUT:
; Carry set on error
;
LoadFATCache:
push  bp
sub   sp,   512
mov   bp,   sp
push  es
pushad

push  ss
pop   es
mov   bx,   bp

mov   al,   1     ; read the boot sector into our buffer
sub   cx,   cx
call  ReadSectors
mov   eax,  [es:bx+39]
cmp   [FATCached],   0h          ; not cached at all!
je    @@updateVSN
cmp   eax, [CurrentVSN]   ; if the serial numbers match, we can leave
je    @@return
@@updateVSN:
mov   [CurrentVSN],  eax

push  cs    ; init fat cache dirty flags to all clean
pop   es
mov   di,   offset FATCacheDirty
mov   al,   0
mov   cx,   9
rep   stosb

mov   bx,   offset FATCache
mov   al,   9                 ; read 9 sectors
mov   cx,   1                 ; starting at sector 1
mov   [FATCached],   0h       ; assume failure
call  ReadSectors
jc    @@return
mov   [FATCached],   1h

@@return:
popad
pop   es
add   sp,   512
pop   bp
ret


; CommitFATCache - write the changed parts of the FAT cache to disk
; INPUT:
; none
; OUTPUT:
; Carry set on error
CommitFATCache:
cmp   [FATCached], 0     ; if no cache, quit with error
je    @@nocache
push  ds
push  es
pushad

push  cs
pop   es
mov   bx,   offset FATCache

push  cs                   ; ds:si = dirty flags
pop   ds
mov   si,   offset FATCacheDirty

mov   cx,   1              ; starting at sector 1

@@loopstart:
cmp   cx,   FATCANUSE      ; sector FATCANUSE is past fat.. we're done
je    @@loopend
cmp   [byte si], 0h        ; is this sector dirty?
je    @@loopinc
mov   al,   1              ; write 1 sector
call  WriteSectors         ; write it!
jc    @@return             ; error writing... return error
mov   [byte si], 0h        ; mark it as clean
@@loopinc:
inc   cx                   ; next sector
inc   si                   ; next dirty flag
add   bx,   512            ; next cache block
jmp   @@loopstart          ; restart
@@loopend:

@@return:                  ; carry clear on no error
popad
pop   es
pop   ds
ret
@@nocache:                 ; no cache! quit with error
stc
ret


; FindFreeFAT - finds the next available FAT entry
; INPUT:
; none
; OUTPUT:
; AX - available fat entry or FATEMPTY if none are available
; Carry set on error
;
FindFreeFAT:
push  dx
sub   ax,   ax       ; start search at beginning
@@loopstart:
call  ReadFATEntry   ; try to read that entry
jc    @@return       ; if error, quit
cmp   dx,   FATEMPTY ; if it's an empty record, we're done
jne   @@loopinc      ; else try next...
jmp   @@loopend
@@loopinc:
inc   ax             ; next entry
cmp   ax,   MAXFAT   ; if we're not done restart
jne   @@loopstart
@@loopend:
cmp   ax,   MAXFAT   ; did we find one?
jne   @@found
mov   ax,   FATEMPTY ; nope, return FATEMPTY
@@found:
clc
@@return:
pop   dx
ret


; ReadFATEntry - returns the value of a given FAT entry (from the FAT cache)
; INPUT:
; AX - fat entry to read
; OUTPUT:
; DX - fat entry value (not valid if error.. duh)
; Carry set on error
;
ReadFATEntry:
cmp   [FATCached], 0h    ; error if no cache is set up
je    @@error
cmp   ax,   MAXFAT       ; or if AX is out of bounds
jae   @@error

push  ds
push  bx
push  ax

push  cs
pop   ds
mov   bx,   offset FATCache

mov   dx,   ax                ; byte index into FAT = AX * 3 / 2
shl   ax,   1
add   ax,   dx
shr   ax,   1                 ; ax = that byte index, dx = original ax
add   bx,   ax                ; add the offset to our fat cache pointer

test  dx,   1                 ; if fat entry is ODD
jnz   @@odd
@@even:                       ; else it must be even
mov   dx,   [bx]
and   dh,   0Fh               ; ours is in the lower nibble (clears carry)
jmp   @@return
@@odd:                        ; it was odd!
mov   dx,   [bx]
shr   dx,   4                 ; shift into final position
clc
@@return:                     ; return success!
pop   ax
pop   bx
pop   ds
ret
@@error:                      ; error occurred, return failure
stc
ret


; WriteFATEntry - writes a value into the fat
; INPUT:
; AX - fat entry to write to
; DX - fat entry value
; OUTPUT:
; Carry set on error
;
WriteFATEntry:
cmp   [FATCached], 0h    ; error if no cache is set up
je    @@error
cmp   ax,   MAXFAT       ; or if AX is out of bounds
jae   @@error

push  ds
push  dx
push  bx
push  ax

push  cs
pop   ds

mov   bx,   ax                ; byte index into FAT = AX * 3 / 2
shl   bx,   1
add   bx,   ax
shr   bx,   1                 ; bx = that byte index
push  bx
add   bx,   offset FATCache   ; add the offset to our fat cache pointer

test  ax,   1                 ; if fat entry is ODD
jnz   @@odd
@@even:                       ; else it must be even
and   [word bx],  0F000h      ; mask off old value
or    [bx],  dx               ; OR the new value in
jmp   @@return
@@odd:                        ; it was odd!
shl   dx,   4
and   [word bx],  0Fh         ; mask of all but the lowest nibble
or    [bx], dx                ; OR the new value in
@@return:                     ; return success!
pop   bx
shr   bx,   9
add   bx,   offset FATCacheDirty
mov   [byte bx],  1
pop   ax
pop   bx
pop   dx
pop   ds
ret
@@error:                      ; error occurred, return failure
stc
ret


; GetFileBlock - returns the address of a file block from its handle
; INPUT:
; AX - file handle
; OUTPUT:
; ES:BX - file block
; Carry set on error
GetFileBlock:
cmp   ax,   MAXFILES
jae   @@error
test  ax,   ax
jz    @@error
push  ax
push  dx

inc   ax

push  cs                   ; es:bx = offset of array
pop   es
mov   bx,   offset FileHeaders

mov   dx,   FILEHEADSIZE+512  ; ax = index into array
mul   dx
add   bx,   ax                ; bx = index into segment (can set carry)

pop   dx
pop   ax
ret
@@error:
stc
ret


; OpenFileHandle - opens a file handle from a FAT entry
; INPUT:
; DX - fat entry
; OUTPUT:
; Carry set on error
; AX - file handle
;
OpenFileHandle:
push  ds
push  es
push  bx
push  cx

cmp   dx,   FATCANUSE                           ; if fat entry is invalid
jb    @@error                                   ; quit with an error
cmp   dx,   MAXFAT
jae   @@error
call  GetFreeFileHandle
jc    @@error                                   ; no free handles!
call  GetFileBlock
jc    @@error
call  LoadFATCache                              ; refresh FAT Cache
jc    @@error

push  es                                        ; ds:bx --> file block
pop   ds

mov   [(FileHead PTR bx).Pointer],     0h
      ; doesnt set Length field.. cant be determined yet
mov   [(FileHead PTR bx).Entry],       dx
mov   [(FileHead PTR bx).Current],     dx
mov   cl,   [CurrentProcess]
mov   [(FileHead PTR bx).ProcessID],   cl
mov   [(FileHead PTR bx).Attr],        0h

push  ax
add   bx,   FILEHEADSIZE         ; advance bx to start of buffer
mov   cx,   dx                   ; cx = dx = fat entry = sector to read from
mov   al,   1                    ; one sector
call  ReadSectors                ; fill buffer
pop   ax
jc    @@error
jmp   @@return

@@error:
stc
sub   ax,   ax
@@return:
pop   cx
pop   bx
pop   es
pop   ds
ret


; CloseFileHandle - commits and closes a file handle
; INPUT:
; AX - file handle
; OUTPUT:
; Carry set on error, and handle is not closed (error can be "not open")
;
CloseFileHandle:
push  es
push  bx
call  GetFileBlock      ; if it's a valid handle, fetch the address
jc    @@return
call  ValidFileBlock
jc    @@return

call  CommitFileBlock   ; commit to disk if needed
jc    @@return

mov   [(FileHead PTR es:bx).Entry], FATEMPTY ; close the handle
call  CommitFATCache

@@return:
pop   bx
pop   es
ret


; ValidFileBlock  - determines whether a given file block is valid (open)
; INPUT:
; ES:BX - address of block
; OUTPUT:
; Carry set on invalid block
;
ValidFileBlock:
cmp   [(FileHead PTR es:bx).Entry], FATCANUSE   ; is it a valid block?
jb    @@bad
cmp   [(FileHead PTR es:bx).Entry], MAXFAT      ; valid block check #2
jae   @@bad
cmp   [(FileHead PTR es:bx).Current], FATCANUSE ; is the buffer block valid?
jb    @@bad
cmp   [(FileHead PTR es:bx).Current], MAXFAT    ; buffer block check #2
jae   @@bad
clc
ret
@@bad:
stc
ret


; CommitFileBlock - writes a file block to disk if needed
; INPUT:
; ES:BX - address of block
; OUTPUT:
; Carry set on error
PROC  CommitFileBlock  NEAR
push  ax
push  bx
push  cx
push  ds

call  ValidFileBlock
jc    @@return                         ; file block is closed!

push  es    ; ds:bx --> block
pop   ds

test  [(FileHead PTR bx).Attr],  1000b ; is the block dirty?
jz    @@return                         ; nope, we're done

or    [(FileHead PTR bx).Attr], 100b   ; set the changed flag
                                       ; ok, we need to write it out

mov   cx,   [(FileHead PTR bx).Current]; cx = sector to write to
add   bx,   FILEHEADSIZE               ; advance BX to start of buffer
mov   al,   1                          ; write one sector
call  WriteSectors                     ; write it!

@@return:
pop   ds
pop   cx
pop   bx
pop   ax
ret
ENDP  CommitFileBlock


; GetFreeFileHandle - finds a free file handle
; INPUT:
; none
; OUTPUT:
; Carry set on error
; AX - file handle
;
GetFreeFileHandle:
push  es
push  bx

mov   ax,   1
@@loopstart:
cmp   ax,   MAXFILES
ja    @@error
call  GetFileBlock
jc    @@error
call  ValidFileBlock
jc    @@found     ; if error, this block must be closed
inc   ax
jmp   @@loopstart

@@error:   
stc
jmp   @@return
@@found:
clc
@@return:
pop   es
pop   bx
ret


; LFBtoSector - converts a logical file block to the corresponding sector
;               if the file is too small, it is enlarged if possible
; INPUT:
; EAX   - number of the logical file block
; ES:BX - address of file block
; OUTPUT:
; Carry set on error
; ECX - absolute sector
;
PROC  LFBtoSector  NEAR
LOCAL block:DWORD =stack_size
STACKFRAME
pushad

call  ValidFileBlock                      ; make sure it's open
jc    @@error

mov   [block], eax
sub   eax,  eax
mov   ax,   [(FileHead PTR es:bx).Entry]  ; cur    = File.Entry;
sub   ebx,  ebx                           ; blocks = 0;
sub   edx,  edx

@@loopstart:                              ; while(blocks < block)
cmp   ebx,  [block]                       ; {
jae   @@loopend
call  ReadFATEntry                        ;    val = ReadFat(cur);
jc    @@error                             ;    if (error) return error;
cmp   dx,   FATEOF                        ;    if (val == FATEOF)
jnz   @@updatecur                         ;    {
inc   ax                                  ;       val = ReadFat(cur+1)
call  ReadFATEntry 
jc    @@findfat                           ;       if (error || val)
test  dx,   dx
jz    @@valeqcur
@@findfat:
call  FindFreeFAT                         ;          val = FindFreeFat();
mov   dx,   ax
@@valeqcur:                               ;       else val = cur + 1;
mov   dx,   ax
@@checkval:                               ;       if (!val) return error;
dec   ax
test  dx,   dx
jz    @@error
call  WriteFATEntry                       ;       WriteFat(cur, val);
jc    @@error
mov   ax,   dx                            ;       WriteFat(val, FATEOF);
mov   dx,   FATEOF
call  WriteFATEntry
mov   dx,   ax
@@updatecur:                              ;    }
mov   ax,   dx                            ;    cur = val
inc   ebx                                 ;    blocks++;
jmp   @@loopstart                         ; }
@@loopend:
clc
jmp   @@return

@@error:
stc
@@return:                                 ; return cur;
mov   [block], eax
popad
mov   ecx,  [block]
ENDSTACKFRAME
ret
ENDP  LFBtoSector


; UpdateFileCache - updates the file cache with data from a new LFB
; INPUT:
; EAX - logical file block
; ES:BX - file block address
; OUTPUT:
; Carry set on error
;
UpdateFileCache:
pushad
call  LFBtoSector
jc    @@return

cmp   cx,   [(FileHead PTR es:bx).Current]   ; do we need to update?
je    @@return

call  CommitFileBlock                        ; yeah.. write the old one
jc    @@return

mov   al,   1                                ; read in the new one
add   bx,   FILEHEADSIZE
call  ReadSectors
jc    @@return

mov   [(FileHead PTR es:bx - FILEHEADSIZE).Current],  cx  ; write new index
@@return:
popad
ret

