;----------------------------------------------------------------------------
; FAT starts at sector 1 and each entry is 12 bits.  First FAT entry is the
; start of the root directory.  Fat entries of 000 indicate an EOF chain.
; Entries of 001 indicate an unuseable cluster. A directory is merely is
; stored the same way a file is.
;----------------------------------------------------------------------------

; ReadSectors - reads one or more sectors from the floppy disk into memory
; INPUT:
; ES:BX - block to write into
; AL    - number of sectors to read (no more than will fit in current segment)
; CX    - sector number
; OUTPUT:
; Carry flag set on error
; AH = error code
; AL = number of sectors read
ReadSectors:
push  bx
push  cx
push  dx

mov   dx,   ax
mov   dh,   dl
@@readloop:
call  ReadSectorsRaw
jc    @@error
sub   dl,   al
jz    @@good
mov   ah,   0
add   cx,   ax
shl   ax,   9
add   bx,   ax
mov   al,   dl
jmp   @@readloop

@@good:
clc
sub   dh,   dl
@@return:
mov   al,   dh
pop   dx
pop   cx
pop   bx
ret
@@error:
sub   dl,   al
stc
jmp   @@return


; WriteSectors - writes one or more sectors onto the floppy disk from memory
; INPUT:
; ES:BX - block to read from
; AL    - number of sectors to read (no more than will fit in current segment)
; CX    - sector number
; OUTPUT:
; Carry flag set on error
; AH = error code
; AL = number of sectors read
WriteSectors:
push  bx
push  cx
push  dx

mov   dx,   ax
mov   dh,   dl
@@readloop:
call  WriteSectorsRaw
jc    @@error
sub   dl,   al
jz    @@good
mov   ah,   0
add   cx,   ax
shl   ax,   9
add   bx,   ax
mov   al,   dl
jmp   @@readloop

@@good:
clc
sub   dh,   dl
@@return:
mov   al,   dh
pop   dx
pop   cx
pop   bx
ret
@@error:
sub   dl,   al
stc
jmp   @@return


; ----------------------------------------------------------------------------
; ReadSectorsRaw - used internally (assumes 1.44mb floppy)
; INPUT:
; ES:BX - block to write into
; AL    - number of sectors to read (1 to 128)
; CX    - sector number
; OUTPUT:
; carry flag set on error.. AH = error code
;
ReadSectorsRaw:
pushad
and   ecx,  0FFFFh
xchg  eax,  ecx   ; eax = sector #, cl = # sectors to read
mov   ch,   0     ; cx  = # sectors to read
push  cx          ; save number of sectors
push  bx          ; save buffer offset
mov   bp,   3     ; 3 tries

sub   edx,   edx  ; edx = 0
mov   ebx,   edx  ; ebx = 0
mov   bx,    18   ; ds:[Sectors_Per_Track]
div   ebx         ; eax = AbsSect / Sect_p_Track; edx = AbsSect % Sect_p_Track
mov   cx,    dx   ;
inc   cx          ; cx = Bios Sector

sub   edx,   edx  ; ready for another divide (edx = 0)
mov   ebx,   edx  ; ebx = 0
mov   bx,    2    ; ds:[Number_Of_Heads]
div   ebx         ; edx = Bios Head eax = Bios Track
mov   bx,    dx   ; bx  = Bios Head

mov   dx,    ax  ; dx  = copy of track
shr   dx,    2
and   dl,    11000000b ; dl = part of track ready to be added to CL
add   cl,    dl  ; cl = sector and part of track
mov   ch,    al  ; ch = other part of track

mov   dl,    0   ; dl = drive
mov   dh,    bl  ; dh = head

pop   bx         ; bx = buffer offset
@@retry:
pop   ax         ; al = number of sectors to read
push  ax
mov   ah,    02h ; ah = 02h (read sector function of int 13h)
int   13h
jc    @@error
pop   ax
popad
ret

@@error:
dec   bp
jz    @@fail
push  ax
push  dx
sub   ax,   ax
sub   dx,   dx
int   13h
pop   dx
pop   ax
jmp   @@retry
@@fail:
pop   ax
popad
ret


; WriteSectorsRaw - used internally (assumes 1.44mb floppy)
; INPUT:
; ES:BX - block to read from
; AL    - number of sectors to read (1 to 128)
; CX    - sector number
; OUTPUT:
; carry flag set on error.. AH = error code
;
WriteSectorsRaw:
pushad
and   ecx,  0FFFFh
xchg  eax,  ecx   ; eax = sector #, cl = # sectors to read
mov   ch,   0     ; cx  = # sectors to read
push  cx          ; save number of sectors
push  bx          ; save buffer offset
mov   bp,   3     ; 3 tries

sub   edx,   edx  ; edx = 0
mov   ebx,   edx  ; ebx = 0
mov   bx,    18   ; ds:[Sectors_Per_Track]
div   ebx         ; eax = AbsSect / Sect_p_Track; edx = AbsSect % Sect_p_Track
mov   cx,    dx   ;
inc   cx          ; cx = Bios Sector

sub   edx,   edx  ; ready for another divide (edx = 0)
mov   ebx,   edx  ; ebx = 0
mov   bx,    2    ; ds:[Number_Of_Heads]
div   ebx         ; edx = Bios Head eax = Bios Track
mov   bx,    dx   ; bx  = Bios Head

mov   dx,    ax  ; dx  = copy of track
shr   dx,    2
and   dl,    11000000b ; dl = part of track ready to be added to CL
add   cl,    dl  ; cl = sector and part of track
mov   ch,    al  ; ch = other part of track

mov   dl,    0   ; dl = drive
mov   dh,    bl  ; dh = head

pop   bx         ; bx = buffer offset
@@retry:
pop   ax         ; al = number of sectors to read
push  ax
mov   ah,    03h ; ah = 03h (write sector function of int 13h)
int   13h
jc    @@error
pop   ax
popad
ret

@@error:
dec   bp
jz    @@fail
push  ax
push  dx
sub   ax,   ax
sub   dx,   dx
int   13h
pop   dx
pop   ax
jmp   @@retry
@@fail:
pop   ax
popad
ret

