﻿/*
cap2mod is a simple mod kit for the game Capitalism 2. dbmod allows editing
of the 1STD.SET database. It was written by Adam Milazzo on February 28th,
2013. This source code is released into the public domain.

For more information, feel free to contact me. http://www.adammil.net/
*/

using System;
using System.Globalization;
using System.IO;
using AdamMil.IO;
using AdamMil.Utilities;
using Microsoft.Office.Interop.Excel;
using BinaryReader = AdamMil.IO.BinaryReader;
using BinaryWriter = AdamMil.IO.BinaryWriter;

namespace Cap2Mod.DBMod
{

class Program
{
  static int Main(string[] args)
  {
    if(args.Length != 0) args[0] = args[0].ToLowerInvariant();
    if(args.Length != 3 || args[0] != "import" && args[0] != "export")
    {
      Console.WriteLine("USAGE: dbmod {import|export} 1std.set data.xlsx");
      return 1;
    }

    Application excel = null;
    Workbook workbook = null;
    try
    {
      Console.WriteLine("Opening game database...");
      Database database;
      using(Stream stream = File.OpenRead(args[1]))
      using(BinaryReader reader = new BinaryReader(stream))
      {
        database = new Database();
        database.Load(reader);
      }

      Console.WriteLine("Opening Excel workbook...");
      excel = new Application();
      workbook = excel.Workbooks.Open(Path.GetFullPath(args[2]), Type.Missing, args[0] == "import");
      if(args[0] == "import")
      {
        Import(workbook, database);
        Console.WriteLine("Saving game database...");
        MemoryStream ms = new MemoryStream();
        using(BinaryWriter writer = new BinaryWriter(ms)) database.Save(writer);
        using(Stream stream = new FileStream(args[1], FileMode.Create, FileAccess.Write)) ms.CopyTo(stream, true);
      }
      else
      {
        Export(database, workbook);
        Console.WriteLine("Saving Excel workbook...");
        workbook.Save();
      }

      return 0;
    }
    catch(Exception ex)
    {
      Console.WriteLine("ERROR: " + ex.GetType().Name + ": " + ex.Message);
      return 2;
    }
    finally
    {
      if(workbook != null) workbook.Close(false);
      if(excel != null) excel.Quit();
    }
  }

  static bool AreEqual(Column column, object excelValue, ref object dbValue)
  {
    if(object.Equals(excelValue, dbValue)) return true;
    if(column.Type == ColumnType.String && excelValue is double)
    {
      double dblValue;
      if(InvariantCultureUtility.TryParse(dbValue as string, out dblValue) && dblValue == (double)excelValue) return true;
    }
    else if(column.Type == ColumnType.Bool && dbValue is bool)
    {
      dbValue = (bool)dbValue ? "T" : "F";
      return object.Equals(excelValue, dbValue);
    }
    return false;
  }

  static void Export(Database db, Workbook workbook)
  {
    foreach(Worksheet sheet in workbook.Worksheets)
    {
      foreach(Table table in db.Tables)
      {
        if(sheet.Name == table.Name)
        {
          Console.WriteLine("Exporting " + table.Name + "...");
          int[] columnMap = MakeColumnMap(sheet, table);
          Range cells = sheet.Cells;
          for(int ri=0; ri<table.Rows.Count; ri++)
          {
            for(int ci=0; ci<table.Columns.Count; ci++)
            {
              int eci = columnMap[ci];
              if(eci != 0)
              {
                Range r = (Range)cells[ri+2, eci];
                object value = table.Rows[ri][table.Columns[ci].Name];
                if(!AreEqual(table.Columns[ci], r.Value, ref value)) r.Value = value; // setting a value is slow, so only do it if they're different
              }
            }
          }

          if(sheet.UsedRange.Rows.Count-1 > table.Rows.Count)
          {
            sheet.UsedRange.Range[sheet.UsedRange.Cells[table.Rows.Count+2, 1],
                                  sheet.UsedRange.Cells[sheet.UsedRange.Rows.Count, sheet.UsedRange.Columns.Count]].ClearContents();
          }
          break;
        }
      }
    }
  }

  static void Import(Workbook workbook, Database db)
  {
    foreach(Worksheet sheet in workbook.Worksheets)
    {
      foreach(Table table in db.Tables)
      {
        if(sheet.Name == table.Name)
        {
          Console.WriteLine("Importing " + table.Name + "...");
          int[] columnMap = MakeColumnMap(sheet, table);
          table.Rows.Clear();
          Range range = sheet.UsedRange;
          for(int ri=2, width=range.Columns.Count, height=range.Rows.Count; ri <= height; ri++)
          {
            Row row = new Row();
            for(int ci=0; ci<table.Columns.Count; ci++)
            {
              int eci = columnMap[ci];
              if(eci != 0)
              {
                object value = ((Range)range.Cells[ri, eci]).Value;
                if(value != null)
                {
                  switch(table.Columns[ci].Type)
                  {
                    case ColumnType.Bool:
                    {
                      string strValue = value as string;
                      if(strValue != null) strValue = strValue.Trim();
                      value = string.IsNullOrEmpty(strValue) ? null : (object)(char.ToUpperInvariant(strValue[0]) == 'T');
                      break;
                    }
                    case ColumnType.Number:
                    {
                      if(!(value is double))
                      {
                        string strValue = value as string;
                        if(strValue == null)
                        {
                          try { value = Convert.ToDouble(value, CultureInfo.InvariantCulture); }
                          catch { }
                        }
                        else
                        {
                          double dblValue;
                          if(InvariantCultureUtility.TryParse(strValue, out dblValue)) value = dblValue;
                        }
                      }
                      break;
                    }
                    case ColumnType.String:
                    {
                      value = value as string ?? Convert.ToString(value, CultureInfo.InvariantCulture);
                      break;
                    }
                  }
                }
                row[table.Columns[ci].Name] = value;
              }
            }
            table.Rows.Add(row);
          }
        }
      }
    }
  }

  static int[] MakeColumnMap(Worksheet sheet, Table table)
  {
    Range cells = sheet.Cells;
    int[] columnMap = new int[table.Columns.Count];
    for(int i=1, width=sheet.UsedRange.Columns.Count; i<=width; i++)
    {
      string columnName = ((Range)cells[1, i]).Value as string;
      for(int j=0; j<table.Columns.Count; j++)
      {
        if(table.Columns[j].Name == columnName)
        {
          columnMap[j] = i;
          break;
        }
      }
    }
    return columnMap;
  }
}

} // namespace Cap2Mod.DBMod
