﻿/*
cap2mod is a simple mod kit for the game Capitalism 2. iresedit allows editing
the I_*.RES/P_*.RES pairs in the game's 'Resource' directory. It was written
by Adam Milazzo on September 9th, 2013. This source code is released into the
public domain.

For more information, feel free to contact me. http://www.adammil.net/
*/

using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Text;
using System.Drawing.Imaging;

namespace Cap2Mod.IResEdit
{

static class Program
{
  struct Entry
  {
    public Entry(string name, int offset) { Name = name; Offset = offset; }
    public string Name;
    public int Offset;
  }

  static int Main(string[] args)
  {
    if(args.Length != 0) args[0] = args[0].ToLowerInvariant();
    if(args.Length != 3 || args[0] != "import" && args[0] != "export")
    {
      Console.WriteLine("USAGE: iresedit {import|export} i_foo.res directory");
      return 1;
    }

    try
    {
      if(!File.Exists(args[1]))
      {
        Console.WriteLine("ERROR: Image file does not exist: " + args[1]);
        return 2;
      }

      string palettePath;
      Color[] palette;
      Dictionary<string, Color[]> namedPalettes;
      if(!ReadPalettes(args[1], out palettePath, out palette, out namedPalettes)) return 2;

      if(args[0] == "export")
      {
        Directory.CreateDirectory(args[2]);
        using(BinaryReader reader = new BinaryReader(File.OpenRead(args[1])))
        {
          Entry[] entries = ReadDatabaseEntries(reader);
          foreach(Entry e in entries)
          {
            Color[] pal = palette;
            if(namedPalettes != null && !namedPalettes.TryGetValue(e.Name, out pal))
            {
              Console.WriteLine("WARN: Embedded image " + e.Name + " was ignored due to a missing palette.");
              continue;
            }

            reader.BaseStream.Position = e.Offset;
            using(Bitmap bmp = ReadImage(reader.BaseStream, pal)) bmp.Save(Path.Combine(args[2], e.Name + ".png"), ImageFormat.Png);
            Console.WriteLine(e.Name);
          }
        }

        Console.WriteLine("Export complete.");
      }
      else
      {
        using(BinaryReader reader = new BinaryReader(new FileStream(args[1], FileMode.Open, FileAccess.ReadWrite)))
        using(BinaryReader palReader =
                namedPalettes == null ? null : new BinaryReader(new FileStream(palettePath, FileMode.Open, FileAccess.ReadWrite)))
        {
          Entry[] entries = ReadDatabaseEntries(reader), palEntries = palReader == null ? null : ReadDatabaseEntries(palReader);
          foreach(string imagePath in Directory.GetFiles(args[2], "*.png"))
          {
            string fileName = Path.GetFileNameWithoutExtension(imagePath);
            int entryIndex = -1;
            for(int i=0; i<entries.Length; i++)
            {
              if(string.Equals(entries[i].Name, fileName, StringComparison.OrdinalIgnoreCase)) { entryIndex = i; break; }
            }
            if(entryIndex == -1)
            {
              Console.WriteLine("WARN: Image " + Path.GetFileName(imagePath) + " ignored (no matching embedded image)");
              continue;
            }

            if(namedPalettes != null && !namedPalettes.ContainsKey(entries[entryIndex].Name))
            {
              Console.WriteLine("WARN: Embedded image " + entries[entryIndex] + " ignored (missing palette)");
              continue;
            }

            try
            {
              using(Image image = Image.FromFile(imagePath))
              {
                Bitmap bmp = image as Bitmap;
                if(bmp == null) throw new InvalidDataException("not a bitmap");
                if(bmp.PixelFormat != PixelFormat.Format8bppIndexed) throw new InvalidDataException("not 8-bit indexed");

                reader.BaseStream.Position = entries[entryIndex].Offset;
                int width  = reader.BaseStream.ReadByte() | (reader.BaseStream.ReadByte()<<8);
                int height = reader.BaseStream.ReadByte() | (reader.BaseStream.ReadByte()<<8);
                if(width != bmp.Width || height != bmp.Height)
                {
                  throw new InvalidDataException("size must be " + bmp.Width.ToString() + "x" + bmp.Height.ToString());
                }
                WriteImage(reader.BaseStream, bmp);

                if(palReader != null)
                {
                  foreach(Entry e in palEntries)
                  {
                    if(string.Equals(e.Name, fileName, StringComparison.OrdinalIgnoreCase))
                    {
                      palReader.BaseStream.Position = e.Offset + 8;
                      foreach(Color color in bmp.Palette.Entries)
                      {
                        palReader.BaseStream.WriteByte(color.R);
                        palReader.BaseStream.WriteByte(color.G);
                        palReader.BaseStream.WriteByte(color.B);
                      }
                      break;
                    }
                  }
                }
              }

              Console.WriteLine(entries[entryIndex].Name);
            }
            catch(Exception ex)
            {
              Console.WriteLine("WARN: Image " + Path.GetFileName(imagePath) + " ignored (" + ex.Message + ")");
            }
          }
        }
        Console.WriteLine("Import complete.");
      }

      return 0;
    }
    catch(Exception ex)
    {
      Console.WriteLine("ERROR: " + ex.GetType().Name + ": " + ex.Message);
      return 2;
    }
  }

  static Entry[] ReadDatabaseEntries(BinaryReader reader)
  {
    Entry[] entries = new Entry[reader.ReadInt16()];
    for(int i=0; i<entries.Length; i++)
    {
      byte[] nameBytes = reader.ReadBytes(9);
      int nameLength = nameBytes.Length;
      while(nameBytes[nameLength-1] == 0) nameLength--;
      entries[i] = new Entry(Encoding.ASCII.GetString(nameBytes, 0, nameLength), reader.ReadInt32());
    }
    return entries;
  }

  static unsafe Bitmap ReadImage(Stream imgFile, Color[] colors)
  {
    int width = imgFile.ReadByte() | (imgFile.ReadByte()<<8), height = imgFile.ReadByte() | (imgFile.ReadByte()<<8);
    if(height <= 0) throw new InvalidDataException("Invalid image file.");

    Bitmap bmp = new Bitmap(width, height, PixelFormat.Format8bppIndexed);
    ColorPalette palette = bmp.Palette; // get a copy of the palette
    for(int i=0; i<colors.Length; i++) palette.Entries[i] = colors[i];
    bmp.Palette = palette;

    BitmapData data = bmp.LockBits(new Rectangle(0, 0, width, height), ImageLockMode.WriteOnly, PixelFormat.Format8bppIndexed);
    byte* row = (byte*)data.Scan0.ToPointer();
    for(int y=0; y<height; row += data.Stride, y++)
    {
      for(int x=0; x<width; x++)
      {
        int value = imgFile.ReadByte();
        if(value < 0) throw new InvalidDataException("Invalid image data.");
        row[x] = (byte)value;
      }
    }
    bmp.UnlockBits(data);
    return bmp;
  }

  static Color[] ReadPalette(Stream paletteFile)
  {
    Color[] palette = new Color[256];
    paletteFile.Seek(8, SeekOrigin.Current);
    for(int i=0; i<palette.Length; i++)
    {
      int r = paletteFile.ReadByte(), g = paletteFile.ReadByte(), b = paletteFile.ReadByte();
      if(b < 0) throw new InvalidDataException("Invalid palette file.");
      palette[i] = Color.FromArgb(r, g, b);
    }
    return palette;
  }

  static bool ReadPalettes(string imagePath, out string palettePath, out Color[] palette, out Dictionary<string,Color[]> namedPalettes)
  {
    palette       = null;
    namedPalettes = null;

    string fileName = Path.GetFileName(imagePath);
    palettePath = fileName.StartsWith("I_", StringComparison.OrdinalIgnoreCase) ?
      Path.Combine(Path.GetDirectoryName(imagePath), "P_" + fileName.Substring(2)) : null;
    bool singlePalette = false;
    if(palettePath == null || !File.Exists(palettePath))
    {
      /* Most I_*.RES files are lacking the corresponding P_*.RES file, but still contain images that we could load
       * if only we knew which palette to use. The "standard" palette stored in pal_std.res works for many of them,
       * but not all. It's possible that the ones for which it doesn't work are not used in Cap2, i.e. they're left
       * over from a previous game built using the same engine, since there's a lot of that stuff, but I'll disable
       * the feature just in case.
      string altPath = Path.Combine(Path.GetDirectoryName(imagePath), "pal_std.res");
      if(!File.Exists(altPath))
      {
        Console.WriteLine("ERROR: Missing palette file: " + palettePath + " or " + altPath);
        return false;
      }
      palettePath   = altPath;
      singlePalette = true;*/
      Console.WriteLine("ERROR: Missing palette file: " + palettePath);
      return false;
    }

    if(singlePalette)
    {
      using(FileStream paletteFile = File.OpenRead(palettePath)) palette = ReadPalette(paletteFile);
    }
    else
    {
      namedPalettes = new Dictionary<string, Color[]>();
      using(BinaryReader reader = new BinaryReader(File.OpenRead(palettePath)))
      {
        Entry[] entries = ReadDatabaseEntries(reader);
        for(int i=0; i<entries.Length; i++)
        {
          reader.BaseStream.Position = entries[i].Offset;
          namedPalettes[entries[i].Name] = ReadPalette(reader.BaseStream);
        }
      }
    }
    return true;
  }

  static unsafe void WriteImage(Stream stream, Bitmap bmp)
  {
    BitmapData data = bmp.LockBits(new Rectangle(0, 0, bmp.Width, bmp.Height), ImageLockMode.ReadOnly, PixelFormat.Format8bppIndexed);
    byte[] imgBytes = new byte[bmp.Width * bmp.Height];
    byte* row = (byte*)data.Scan0.ToPointer();
    for(int y=0, i=0; y<bmp.Height; row += data.Stride, y++)
    {
      for(int x=0; x<bmp.Width; i++, x++) imgBytes[i] = row[x];
    }
    bmp.UnlockBits(data);
    stream.Write(imgBytes, 0, imgBytes.Length);
  }
}

} // namespace Cap2Mod.IResEdit
