﻿/*
cap2mod is a simple mod kit for the game Capitalism 2. prodedit allows more
advanced editing of the I_ITEM.RES/P_ITEM.RES files in the game's 'Resource'
than that allowed by iresedit. It was written by Adam Milazzo on
September 10th, 2013. This source code is released into the public domain.

For more information, feel free to contact me. http://www.adammil.net/
*/

using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Text;
using System.Drawing.Imaging;

namespace Cap2Mod.ProdEdit
{

static class Program
{
  struct Entry
  {
    public Entry(string name, int offset) { Name = name; Offset = offset; }
    public string Name;
    public int Offset;
  }

  static int Main(string[] args)
  {
    if(args.Length != 0) args[0] = args[0].ToLowerInvariant();
    if(args.Length != 3 || args[0] != "import" && args[0] != "export")
    {
      Console.WriteLine("USAGE: prodedit {import|export} i_item.res directory");
      return 1;
    }

    try
    {
      if(!File.Exists(args[1]))
      {
        Console.WriteLine("ERROR: Image file does not exist: " + args[1]);
        return 2;
      }

      string palettePath;
      Dictionary<string, Color[]> namedPalettes;
      if(!ReadPalettes(args[1], out palettePath, out namedPalettes)) return 2;

      if(args[0] == "export")
      {
        Directory.CreateDirectory(args[2]);
        using(BinaryReader reader = new BinaryReader(File.OpenRead(args[1])))
        {
          Entry[] entries = ReadDatabaseEntries(reader);
          foreach(Entry e in entries)
          {
            Color[] palette = null;
            if(namedPalettes != null && !namedPalettes.TryGetValue(e.Name, out palette))
            {
              Console.WriteLine("WARN: Embedded image " + e.Name + " was ignored due to a missing palette.");
              continue;
            }

            reader.BaseStream.Position = e.Offset;
            using(Bitmap bmp = ReadImage(reader.BaseStream, palette)) bmp.Save(Path.Combine(args[2], e.Name + ".png"), ImageFormat.Png);
            Console.WriteLine(e.Name);
          }
        }

        Console.WriteLine("Export complete.");
      }
      else
      {
        string[] imagePaths = Directory.GetFiles(args[2], "*.png");
        using(BinaryWriter imgWriter = new BinaryWriter(new FileStream(args[1], FileMode.Open, FileAccess.ReadWrite)))
        using(BinaryWriter palWriter = new BinaryWriter(new FileStream(palettePath, FileMode.Open, FileAccess.ReadWrite)))
        {
          imgWriter.Write((ushort)imagePaths.Length);
          palWriter.Write((ushort)imagePaths.Length);

          const int PaletteSize = 776, ImageSize = 120*120 + 2*2;
          int headerSize = (imagePaths.Length+1) * 13 + 2;
          for(int i=0; i <= imagePaths.Length; i++)
          {
            string name = i < imagePaths.Length ? Path.GetFileNameWithoutExtension(imagePaths[i]).ToUpperInvariant() : "";
            if(name.Length == 0 || Encoding.ASCII.GetByteCount(name) > 8)
            {
              throw new ArgumentException(imagePaths[i] + " must have a name from 1 to 8 low ASCII characters (excluding the extension).");
            }
            byte[] nameBytes = new byte[9];
            Encoding.ASCII.GetBytes(name, 0, name.Length, nameBytes, 0);

            imgWriter.Write(nameBytes, 0, nameBytes.Length);
            imgWriter.Write(headerSize + i*ImageSize);

            palWriter.Write(nameBytes, 0, nameBytes.Length);
            palWriter.Write(headerSize + i*PaletteSize);
          }

          foreach(string imagePath in imagePaths)
          {
            using(Image image = Image.FromFile(imagePath))
            {
              Bitmap bmp = image as Bitmap;
              if(bmp == null) throw new InvalidDataException(imagePath + " is not a bitmap.");
              if(bmp.PixelFormat != PixelFormat.Format8bppIndexed) throw new InvalidDataException(imagePath + " is not 8-bit indexed.");
              if(bmp.Width != 120 || bmp.Height != 120) throw new InvalidDataException(imagePath + " is not 120x120 in size.");

              WriteImage(imgWriter, bmp);
              palWriter.Write(768 + 8); // the number of bytes in the palette
              palWriter.Write(0xB123); // i don't know what this means, but all the game palettes have it

              Color[] pal = bmp.Palette.Entries;
              for(int i=0; i<pal.Length; i++)
              {
                Color color = pal[i];
                palWriter.Write(color.R);
                palWriter.Write(color.G);
                palWriter.Write(color.B);
              }
              for(int i=pal.Length; i<256; i++)
              {
                palWriter.Write((byte)0);
                palWriter.Write((byte)0);
                palWriter.Write((byte)0);
              }
            }

            Console.WriteLine(Path.GetFileNameWithoutExtension(imagePath));
          }
        }
        Console.WriteLine("Import complete.");
      }

      return 0;
    }
    catch(Exception ex)
    {
      Console.WriteLine("ERROR: " + ex.GetType().Name + ": " + ex.Message);
      return 2;
    }
  }

  static Entry[] ReadDatabaseEntries(BinaryReader reader)
  {
    Entry[] entries = new Entry[reader.ReadInt16()];
    for(int i=0; i<entries.Length; i++)
    {
      byte[] nameBytes = reader.ReadBytes(9);
      int nameLength = nameBytes.Length;
      while(nameBytes[nameLength-1] == 0) nameLength--;
      entries[i] = new Entry(Encoding.ASCII.GetString(nameBytes, 0, nameLength), reader.ReadInt32());
    }
    return entries;
  }

  static unsafe Bitmap ReadImage(Stream imgFile, Color[] colors)
  {
    int width = imgFile.ReadByte() | (imgFile.ReadByte()<<8), height = imgFile.ReadByte() | (imgFile.ReadByte()<<8);
    if(height <= 0) throw new InvalidDataException("Invalid image file.");

    Bitmap bmp = new Bitmap(width, height, PixelFormat.Format8bppIndexed);
    ColorPalette palette = bmp.Palette; // get a copy of the palette
    for(int i=0; i<colors.Length; i++) palette.Entries[i] = colors[i];
    bmp.Palette = palette;

    BitmapData data = bmp.LockBits(new Rectangle(0, 0, width, height), ImageLockMode.WriteOnly, PixelFormat.Format8bppIndexed);
    byte* row = (byte*)data.Scan0.ToPointer();
    for(int y=0; y<height; row += data.Stride, y++)
    {
      for(int x=0; x<width; x++)
      {
        int value = imgFile.ReadByte();
        if(value < 0) throw new InvalidDataException("Invalid image data.");
        row[x] = (byte)value;
      }
    }
    bmp.UnlockBits(data);
    return bmp;
  }

  static Color[] ReadPalette(Stream paletteFile)
  {
    Color[] palette = new Color[256];
    paletteFile.Seek(8, SeekOrigin.Current);
    for(int i=0; i<palette.Length; i++)
    {
      int r = paletteFile.ReadByte(), g = paletteFile.ReadByte(), b = paletteFile.ReadByte();
      if(b < 0) throw new InvalidDataException("Invalid palette file.");
      palette[i] = Color.FromArgb(r, g, b);
    }
    return palette;
  }

  static bool ReadPalettes(string imagePath, out string palettePath, out Dictionary<string,Color[]> namedPalettes)
  {
    namedPalettes = null;

    string fileName = Path.GetFileName(imagePath);
    palettePath = fileName.StartsWith("I_", StringComparison.OrdinalIgnoreCase) ?
      Path.Combine(Path.GetDirectoryName(imagePath), "P_" + fileName.Substring(2)) : null;
    if(palettePath == null || !File.Exists(palettePath))
    {
      Console.WriteLine("ERROR: Missing palette file: " + palettePath);
      return false;
    }

    namedPalettes = new Dictionary<string, Color[]>();
    using(BinaryReader reader = new BinaryReader(File.OpenRead(palettePath)))
    {
      Entry[] entries = ReadDatabaseEntries(reader);
      for(int i=0; i<entries.Length; i++)
      {
        reader.BaseStream.Position = entries[i].Offset;
        namedPalettes[entries[i].Name] = ReadPalette(reader.BaseStream);
      }
    }
    return true;
  }

  static unsafe void WriteImage(BinaryWriter writer, Bitmap bmp)
  {
    writer.Write((ushort)bmp.Width);
    writer.Write((ushort)bmp.Height);

    BitmapData data = bmp.LockBits(new Rectangle(0, 0, bmp.Width, bmp.Height), ImageLockMode.ReadOnly, PixelFormat.Format8bppIndexed);
    byte[] imgBytes = new byte[bmp.Width * bmp.Height];
    byte* row = (byte*)data.Scan0.ToPointer();
    for(int y=0, i=0; y<bmp.Height; row += data.Stride, y++)
    {
      for(int x=0; x<bmp.Width; i++, x++) imgBytes[i] = row[x];
    }
    bmp.UnlockBits(data);
    writer.Write(imgBytes);
  }
}

} // namespace Cap2Mod.ProdEdit
