﻿using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.Linq;

namespace du
{

static class Program
{
  static void Main(string[] args)
  {
    // parse the command line parameters. this is very rudimentary...
    long threshold = args.Length >= 1 ? ParseSizeString(args[0]) : 1;

    string directoryName = args.Length >= 2 ? args[1] : Environment.CurrentDirectory;
    DirectoryInfo directory = new DirectoryInfo(Path.GetFullPath(directoryName));

    bool sortOwn = args.Length >= 3 && args[2].ToLowerInvariant() == "own";

    Dictionary<string, Info> folderSizes = new Dictionary<string, Info>();
    ScanDirectory(directory, directory, folderSizes);

    // filter the directory names to the ones that meet the threshold
    List<KeyValuePair<string, Info>> filteredPairs = folderSizes.Where(pair => pair.Value.TotalSize >= threshold).ToList();

    // calculate the length of the root directory name, which will be the length that we'll strip off from the subdirectory names
    // so that we can show them relative to the root directory
    int rootNameLength = directory.FullName.Length;
    // if the length doesn't include a trailing slash, then add one to it, in order to remove the slash from the subdirectory
    // names too
    if(!directory.FullName.EndsWith(Path.DirectorySeparatorChar.ToString()) &&
       !directory.FullName.EndsWith(Path.AltDirectorySeparatorChar.ToString()))
    {
      rootNameLength++;
    }

    // find the maximum name length (ignoring the part that we will strip off)
    int maxNameLength = filteredPairs.Max(pair => pair.Key.Length - rootNameLength);

    // render names at least equal in length to the column, but not so long that they'd exceed the window width
    int nameSize = Math.Max(4, Math.Min(maxNameLength, Console.WindowWidth - 25));

    // render the headers
    Console.WriteLine("Name".PadRight(nameSize) + "  Total size  Own size");
    Console.WriteLine(new string('-', nameSize) + "  ----------  --------");

    foreach(KeyValuePair<string,Info> pair in filteredPairs.OrderByDescending(p => sortOwn ? p.Value.Size : p.Value.TotalSize))
    {
      // strip off the root directory name to get the display name. if this is the root directory, display it as a period
      string displayName = pair.Key.Length <= rootNameLength ? "." : pair.Key.Substring(rootNameLength);
      if(displayName.Length > nameSize) // if the display name is too long, then we'll need to truncate it
      {
        if(displayName.Length - nameSize <= 3) // if it's too long by at most 3 characters, just add an ellipsis to the end
        {
          displayName = displayName.Substring(0, nameSize-3) + "...";
        }
        else // otherwise, add an ellipsis to the center
        {
          int sideLength = (nameSize-3) / 2; // calculate the length of either side (this could be too small by 1/2, but oh well)
          displayName = displayName.Substring(0, sideLength) + "..." +
                        displayName.Substring(displayName.Length - sideLength, sideLength);
        }
      }

      // render the row
      Console.WriteLine(displayName.PadRight(nameSize) + "  " + GetSizeString(pair.Value.TotalSize).PadLeft(10) + "  " +
                        GetSizeString(pair.Value.Size));
    }
  }

  sealed class Info
  {
    public long Size, TotalSize;
  }

  // common unit sizes in bytes
  const long KB = 1024, MB = KB*1024, GB = MB*1024;

  /// <summary>Converts a data length into a short, readable string.</summary>
  static string GetSizeString(long size)
  {
    if(size < KB) return size.ToString() + "b";
    else if(size < MB) return ((double)size/KB).ToString("f2") + " kb";
    else if(size < GB) return ((double)size/MB).ToString("f2") + " mb";
    else return ((double)size/GB).ToString("f2") + " gb";
  }

  /// <summary>Parses a size string, such as one returned from <see cref="GetSizeString"/>.</summary>
  static long ParseSizeString(string value)
  {
    value = value.Trim();

    // find the last digit, and assume that the number is everything before that
    int numEnd = value.LastIndexOfAny(new char[] { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9' });
    if(numEnd != -1) // if there's a number...
    {
      double size;
      if(double.TryParse(value.Substring(0, numEnd+1), NumberStyles.AllowLeadingWhite | NumberStyles.Float,
                         CultureInfo.InvariantCulture, out size)) // if it can be parsed as a valid floating point number...
      {
        // then take the rest of the argument as the suffix, and multiply the parsed value by the suffix
        string suffix = value.Substring(numEnd+1).Trim().ToLowerInvariant();
        if(!string.IsNullOrEmpty(suffix))
        {
          switch(suffix)
          {
            case "m": case "mb": size *= MB; break;
            case "k": case "kb": size *= KB; break;
            case "g": case "gb": size *= GB; break;
            case "b": break;
            default: goto badValue;
          }
        }
      }

      return (long)Math.Round(size); // round to a byte threshold
    }

    badValue: throw new ArgumentException("Invalid size string: " + value);
  }

  /// <summary>Scans a directory recursively and adds all of its folder sizes to <paramref name="folderSizes"/>.</summary>
  static void ScanDirectory(DirectoryInfo root, DirectoryInfo directory, Dictionary<string, Info> folderSizes)
  {
    try
    {
      // calculate the directory's size by summing the sizes of all of its files
      long size = 0;
      foreach(FileInfo file in directory.GetFiles()) size += file.Length;

      // add the new folder info
      folderSizes[directory.FullName] = new Info() { Size = size, TotalSize = size };

      // if this isn't the root directory, add the size to all of the parent folders as well, to calculate their total size
      if(directory != root)
      {
        for(DirectoryInfo ancestor = directory.Parent; ancestor != null; ancestor = ancestor.Parent)
        {
          folderSizes[ancestor.FullName].TotalSize += size;
          if(ancestor.FullName == root.FullName) break;
        }
      }

      // then recurse into subdirectories
      foreach(DirectoryInfo subdirectory in directory.GetDirectories())
      {
        ScanDirectory(root, subdirectory, folderSizes);
      }
    }
    catch(UnauthorizedAccessException) { } // if a directory can't be accessed, ignore it
  }
}

}
