#include <iostream.h>
#include <stdlib.h>
#include <string.h>
#include "Debug.h"
#include "Defines.h"
#include "Eclair.h"

#include <stdio.h>
Eclair::Eclair()
{
  m_pROM        = NULL;
  m_DrawFxn     = NULL;
  m_SoundFxn    = NULL;
  m_pMem        = new byte[65536];
  m_pScans      = new byte[256*NUM_SCANS];
  m_CPU.Emu     = this;
  m_CPU.User    = m_pMem;
  m_CPU.IPeriod = CYCLES_PER_SCANLINE;
  m_CPU.Trace   = 0;
  m_CPU.TrapBadOps = 1;
}

Eclair::~Eclair()
{
  delete[] m_pMem;
  delete[] m_pScans;
  if(m_pROM) delete[] m_pROM;
}

void Eclair::Reset()
{
  m_DoReset = true;
}

int Eclair::LoadROM(istream &file)
{
  long romlen;

  if(m_pROM) { delete[] m_pROM; m_pROM=NULL; }

  file.seekg(0, ios::end);
  romlen = file.tellg()-BANKSIZE;
  file.seekg(0, ios::beg);
  m_NumBanks = romlen/BANKSIZE;

  m_pROM = new byte[romlen];
  file.read(m_pMem+BANK0, BANKSIZE);
  file.read(m_pROM, romlen);
  m_pMem[BANK0+0x3F] = 0; // null terminate rom name, just in case..

  memcpy(m_pMem+BANKX, m_pMem+BANK0, BANKSIZE);
  memcpy(m_pMem+BANKY, m_pMem+BANK0, BANKSIZE);

  ResetSystem();

  return 0;
}

char const * Eclair::RomName()
{
  return (char const *)m_pROM;
}

void Eclair::SetDraw(DRAWFUNCT fxn)
{
  m_DrawFxn = fxn;
}

void Eclair::SetSound(SOUNDFUNCT fxn)
{
  m_SoundFxn = fxn;
}

int Eclair::DoScan()
{
  RunZ80(&m_CPU);

  if(m_Scanline<NUM_SCANS && m_DrawFxn) DoDraw(m_Scanline);
  else if(m_Scanline == NUM_SCANS+VBLANK_SCANS)
  {
    if(m_DrawFxn)  DoDraw(255);
    if(m_SoundFxn) m_SoundFxn(m_SndChans, m_PCMChan);
    m_Scanline = 0;
    return 1;
  }
  m_Scanline++;
  return 0;
}

void Eclair::Press(int controller, BUTTON btn, int down)
{
  byte mask = 1<<btn;
  if(down) m_Control[controller] |= mask;
  else m_Control[controller] &= ~mask;
}

void Eclair::ResetSystem()
{
  m_XBank = m_YBank = m_HPanning = m_VPanning = 0;
  m_Control[0] = m_Control[1] = 0;
  m_Scanline = 0;
  m_Viewport.c.left  = m_Viewport.c.top = 0;
  m_Viewport.c.right = 64;
  m_Viewport.c.bottom = NUM_SCANS/4;
  m_PCMChan.freq  = m_PCMChan.offset = m_PCMChan.start = m_PCMChan.end = 0;
  m_PCMChan.info.type = 0;
  m_DoReset = false;
  m_VPBkgnd = m_NVPBkgnd = 0;
  m_TPitch  = 96;
  m_SoundDirty = true;

  for(int i=0;i<7;i++)
  {
    m_SndChans[i].freq = 0;
    m_SndChans[i].vol  = 0;
    m_SndChans[i].info.type = 0;
    m_SndChans[i].info.PW   = 0;
  }

  ResetZ80(&m_CPU);
  m_CPU.PC.W = m_pMem[BANKX-2] + m_pMem[BANKX-1]*256 + BANK0;
  m_CPU.SP.W = BASEMAX & 0xFFF0;
  m_CPU.I    = INTTAB/256;

  memset(m_pMem+0x0100, 0, 0x6F00);
  memset(m_pScans, 0, 256*NUM_SCANS);

  // default interrupt handler (EI, RETI)
  m_pMem[0] = 0xFB, m_pMem[1] = 0xED, m_pMem[2] = 0x4D;
  // and the NMI handler with RETN
  m_pMem[0x66] = 0xED, m_pMem[0x67] = 0x45;
  m_pMem[INTTAB+0x38]=0x38, m_pMem[INTTAB+0x66]=0x66;
  srand(0);

  RunZ80(&m_CPU); // prime DoScan() [so the interrupt comes before the draw]
}

void Eclair::DrawSprite(byte *dest, int num, int x, int y, int scanline, byte info)
{
  int yi = scanline-y;
  if((unsigned)yi > 3 || x < -3) return;
  int times, xi=0;

  if(x<0) times=4+x, xi=-x;
  else if(x>252) times=256-x,dest+=x;
  else times=4,dest+=x;

  if(info & 0x04) yi = 3-yi; // vertical flip
  byte *src = m_pMem + SPRTILE + num*16 + yi*4 + xi;

  if(info & 0x02) // horizontal flip
  { for(int i=0;i<times;i++) if(src[3-i]) dest[i] = src[3-i];
  }
  else
  { for(int i=0;i<times;i++) if(src[i]) dest[i] = src[i];
  }
}

void Eclair::DoDraw(int scanline)
{
  if(scanline == 255)
  {
    m_DrawFxn(m_pScans, m_Palette, scanline);
    return;
  }

  register int i;
  byte *scans = m_pScans+scanline*256;
  byte *mem;
  int left = m_Viewport.c.left, right  = m_Viewport.c.right;
  int top  = m_Viewport.c.top,  bottom = m_Viewport.c.bottom;
  if(right<left) { int t=right; right=left,left=t; }
  if(bottom<top) { int t=bottom; bottom=top,top=t; }
  int sl=left*4, sr=right*4, st=top*4, sb=bottom*4;

  // background colors
  if(scanline >= st && scanline < sb)
  {
    if(m_NVPBkgnd)
    {
      if(sl) memset(scans, m_NVPBkgnd, sl);
      if(sr!=256) memset(scans+sr, m_NVPBkgnd, 256-sr);
    }
    if(m_VPBkgnd) memset(scans+sl, m_VPBkgnd, sr-sl);
  }
  else if(m_NVPBkgnd) memset(scans, m_NVPBkgnd, 256);

  // background sprites
  mem = m_pMem+SPRITES;
  for(i=0;i<256;mem+=4,i++)
  {
    byte info = mem[3];
    if((info & 0x09) != 0x09) continue; // not used or not background sprite
    int x = mem[1], y = mem[2];
    if(!(info & 0x10)) // not absolute position
      x-=m_HPanning, y-=m_VPanning;
    DrawSprite(scans, mem[0], x, y, scanline, info);
  }

  // viewport tiles
  if(scanline >= st && scanline < sb)
  {
    int tr, ts,te; // tile row, start, end (in virtual screen coords)
    tr = (scanline-st+m_VPanning)/4;
    if(tr*m_TPitch > SPRITES-IMGDATA) goto SkipTiles;
    ts = m_HPanning/4, te=(m_HPanning+3)/4 + (right-left);
    int sx = sl-m_HPanning%4;
    mem = m_pMem + IMGDATA + tr*m_TPitch + ts;
    byte *srcbase = m_pMem+BAKTILE + ((scanline+m_VPanning)%4)*4;
    byte *dest=scans+sx;
    if(te>m_TPitch) te=m_TPitch;
    int wid=te-ts;
    if(mem+wid > m_pMem+SPRITES)
      wid -= mem+wid - m_pMem - SPRITES;

    if(wid)
    {
      int n=mem[0], times=4;
      if(n)
      {
        int xi=0;
        if(sx<sl) times=4-(xi=(sl-sx)),dest+=xi,sx+=xi;
        byte *src = srcbase + n*16 + xi;
        for(int j=0;j<times;j++) if(src[j]) dest[j]=src[j];
      }
      dest+=times;
      sx+=times;
      wid--;
    }

    if(wid)
    {
/*
      for(i=1;i<wid;i++)
      {
        int n=mem[i];
        if(n)
        {
          byte *src = srcbase + n*16;
          if(src[0]) dest[0]=src[0];
          if(src[1]) dest[1]=src[1];
          if(src[2]) dest[2]=src[2];
          if(src[3]) dest[3]=src[3];
        }
        dest+=4;
      }
*/
      __asm
      {
        push ebp
        mov esi, [srcbase]
        push esi
        mov ecx, [wid]    ; ecx = width (assumed nonzero)
        sub eax, eax
        mov edi, [dest]   ; edi = dest
        mov ebp, [mem]    ; ebp = mem
        mov ebx, 1        ; ebx = tile index
        sub edi, 4        ; prime edi, because addition comes at start of loop
        copy:
        dec ecx
        jz  short end
        mov al, [ebp + ebx] ; al = tile number
        inc ebx             ; increment tile index
        add edi, 4          ; move to correct pixel for writing
        or  al, al          ; if (al == 0)
        jz  short copy      ;   continue;
        mov esi, [esp]      ; esi = srcbase + tilenum*16
        shl eax, 4          ; eax = tile number * 16
        add esi, eax

        mov al, [esi]       ; al = source pixel
        mov dl, al          ; dl = source pixel
        neg al              ; al = al ? 00 : FF
        sbb al, al          
        not al
        and [edi], al       ; dest &= mask
        xor [edi], dl       ; dest ^= dest pixel

        mov al, [esi+1]
        mov dl, al
        neg al
        sbb al, al
        not al
        and [edi+1], al
        xor [edi+1], dl

        mov al, [esi+2]
        mov dl, al
        neg al
        sbb al, al
        not al
        and [edi+2], al
        xor [edi+2], dl

        mov al, [esi+3]
        mov dl, al
        neg al
        sbb al, al
        not al
        and [edi+3], al
        xor [edi+3], dl
        
        sub eax, eax
        jmp short copy

        end:
        pop esi
        add edi, 4
        pop ebp
        mov i, ebx
        mov dest, edi
      }
      sx+=(wid-1)*4;

      int n=mem[i], times=4;
      if(n)
      {
        int xi=0;
        if(sx>sr-4) times=sr-sx;
        byte *src = srcbase + n*16 + xi;
        for(int j=0;j<times;j++) if(src[j]) dest[j]=src[j];
      }
    }
  }
  SkipTiles:

  // foreground sprites
  mem = m_pMem+SPRITES;
  for(i=0;i<256;mem+=4,i++)
  {
    byte info = mem[3];
    if((info & 0x09) != 0x01) continue; // if background sprite or not used
    int x = mem[1], y = mem[2];
    if(!(info & 0x10)) // not absolute position
      x-=m_HPanning, y-=m_VPanning;
    DrawSprite(scans, mem[0], x, y, scanline, info);
  }

  // output the scanline
  m_DrawFxn(scans, m_Palette, scanline);
}
