#include <stdlib.h>
#include <memory.h>
#include "MyZ80.h"
#include "Eclair.h"
#include "Debug.h"
#include "Defines.h"

byte _InZ80(Z80 *R, register word Port)
{
  if(Port >= 0x10 && Port <= 0x47) // one of ours
  {
    Eclair *emu = R->Emu;
    if(Port < 0x2C) // sound channel
    {
      SndChannel *chan = &emu->m_SndChans[(Port-0x10)/4];
      int part = (Port-0x10)%4;
      switch(part)
      {
        case 0: // freq low
          return LOBYTE(chan->freq);
        case 1: // freq high
          return HIBYTE(chan->freq);
        case 2: // volume
          return chan->vol;
        case 3: // info
          return (chan->info.type << 6) | chan->info.PW;
      }
    }
    else if(Port < 0x35) // PCM channel
    {
      PCMChannel *chan = &emu->m_PCMChan;
      int part = (Port - 0x2C);
      switch(part)
      {
        case 0: // freq low
          return LOBYTE(chan->freq);
        case 1: // freq high
          return HIBYTE(chan->freq);
        case 2: // offset low
          return LOBYTE(chan->offset);
        case 3: // offset high
          return HIBYTE(chan->offset);
        case 4: // start low
          return LOBYTE(chan->start);
        case 5: // start high
          return HIBYTE(chan->start);
        case 6: // end low
          return LOBYTE(chan->end);
        case 7: // end high
          return HIBYTE(chan->end);
        case 8: // info byte
          return chan->info.type;
      }
    }
    else if(Port < 0x37) // controller
      return emu->m_Control[Port-0x35];
    else if(Port < 0x3B) // viewport
      return emu->m_Viewport.values[Port-0x37];
    else
    {
      switch((byte)Port)
      {
        case 0x3B: // tile pitch
          return emu->m_TPitch;
        case 0x3C: // horizontal panning
          return emu->m_HPanning;
        case 0x3D: // vertical panning
          return emu->m_VPanning;
        case 0x3E: // current scanline
          {
            int line = emu->m_Scanline;
            return line<NUM_SCANS ? (byte)line : 255;
          }
        case 0x3F:
          return emu->m_XBank;
        case 0x40:
          return emu->m_YBank;
        case 0x41:
          return emu->m_PalEntry;
        case 0x42:
          return LOBYTE(emu->m_Palette[emu->m_PalEntry]);
        case 0x43:
          return HIBYTE(emu->m_Palette[emu->m_PalEntry++]);
        case 0x45:
          return (byte)(rand()>>7);
        case 0x46:
          return emu->m_VPBkgnd;
        case 0x47:
          return emu->m_NVPBkgnd;
      }
    }
  }
  return 0; // unhandled
}

void _OutZ80(Z80 *R, register word Port,register byte Value)
{
#ifndef NDEBUG
  if(Port >= 0x10 && Port <= 0x52) // one of ours
#else
  if(Port >= 0x10 && Port <= 0x47) // one of ours
#endif
  {
    Eclair *emu = R->Emu;
    if(Port < 0x2C) // sound channel
    {
      SndChannel *chan = &emu->m_SndChans[(Port-0x10)/4];
      int part = (Port-0x10)%4;
      emu->m_SoundDirty = true;
      switch(part)
      {
        case 0: // freq low
          SETLOBYTE(chan->freq, Value); break;
        case 1: // freq high
          SETHIBYTE(chan->freq, Value); break;
        case 2: // volume
          chan->vol = Value; break;
        case 3: // info
          chan->info.type = Value >> 6;
          chan->info.PW   = Value & 0x3F;
          break;
      }
    }
    else if(Port < 0x35) // PCM channel
    {
      PCMChannel *chan = &emu->m_PCMChan;
      int part = (Port - 0x2C);
      emu->m_SoundDirty = true;
      switch(part)
      {
        case 0: // freq low
          SETLOBYTE(chan->freq, Value);
          chan->inc = (chan->freq<<16)/22050;
          break;
        case 1: // freq high
          SETHIBYTE(chan->freq, Value);
          chan->inc = (chan->freq<<16)/22050;
          break;
        case 2: // offset low
          SETLOBYTE(chan->offset, Value);
          chan->off = chan->offset<<16; break;
        case 3: // offset high
          SETHIBYTE(chan->offset, Value);
          chan->off = chan->offset<<16; break;
        case 4: // start low
          SETLOBYTE(chan->start, Value); break;
        case 5: // start high
          SETHIBYTE(chan->start, Value); break;
        case 6: // end low
          SETLOBYTE(chan->end, Value); break;
        case 7: // end high
          SETHIBYTE(chan->end, Value); break;
        case 8: // info byte
          chan->info.type = Value;
          if(!(Value&0x02)) chan->forward=true; // not pingpong or reverse
          break;
      }
    }
    else if(Port < 0x37) return;
    else if(Port < 0x3B) // viewport
      emu->m_Viewport.values[Port-0x37] = Value;
    else
    {
      switch((byte)Port)
      {
        case 0x3B:
          emu->m_TPitch = Value; break;
        case 0x3C: // horizontal panning
          emu->m_HPanning = Value; break;
        case 0x3D: // vertical panning
          emu->m_VPanning = Value; break;
        case 0x3F: // bankswitch X
          if(Value <= emu->m_NumBanks && Value != emu->m_XBank)
          {
            emu->m_XBank=Value;
            memcpy(emu->m_pMem+BANKX,
                   Value?emu->m_pROM+(Value<<13)+(Value<<12):emu->m_pMem+BANK0, BANKSIZE);
          }
          break;
        case 0x40: // bankswitch Y
          if(Value <= emu->m_NumBanks && Value != emu->m_YBank)
          {
            emu->m_YBank=Value;
            memcpy(emu->m_pMem+BANKY,
                   Value?emu->m_pROM+(Value<<13)+(Value<<12):emu->m_pMem+BANK0, BANKSIZE);
          }
          break;
        case 0x41:
          emu->m_PalEntry = Value; break;
        case 0x42:
          {
            int i = emu->m_PalEntry;
            SETLOBYTE(emu->m_Palette[i], Value);
          }
          break;
        case 0x43:
          {
            int i = emu->m_PalEntry++;
            SETHIBYTE(emu->m_Palette[i], Value);
          }
          break;
        case 0x44:
          srand(Value); break;
        case 0x46:
          emu->m_VPBkgnd = Value; break;
        case 0x47:
          emu->m_NVPBkgnd = Value; break;
      #ifdef ROMDEBUG
        case 0x50:
          DebugOut("%.2X", Value);
          break;
        case 0x51:
          DebugOut("%c", Value);
          break;
        case 0x52:
          DebugOut("%d", Value);
          break;
      #endif
      }
    }
  }
}

word LoopZ80(register Z80 *R)
// called once per scanline!
{
  Eclair *emu = R->Emu;

  if(emu->m_DoReset) // reset button pressed
  {
    emu->m_Scanline = 0;
    emu->ResetSystem();
    return INT_IRQ;
  }
  else
  {
    if(emu->m_Scanline < NUM_SCANS) return 0x00; // scanline interrupt
    else if(emu->m_Scanline == NUM_SCANS+VBLANK_SCANS) return INT_NMI;
  }
  return INT_QUIT;
}

