#include <dsound.h>
#include <math.h>
#include "Debug.h"
#include "IntrFace\CDDraw.h"
#include "Output.h"
#include "Defines.h"
#include "Eclair.h"

#define BUFLEN (22100/4*2)
#define PI2 6.28318530717958647692528676655901
extern CDDraw g_Cdraw;
extern HWND   g_hWnd;
extern Eclair g_Emu;
static CDDSurface *Vscreen;
static LPDIRECTSOUND pDS;
static LPDIRECTSOUNDBUFFER Primary, Second;

static int InitSound();
static void FillSound(short *buf, DWORD samples, SndChannel const chans[7], PCMChannel &PCMchan);
static inline bool InRange(int r1, int r2, int v)
{ if(r1<r2) return v>=r1 && v<r2;
  if(r1>r2) return v>=r1 || v<r2;
  return r1==v;
}
static inline int Dist(int from, int to)
{ if(to>=from) return to-from;
  return BUFLEN-from+to;
}
static int Sine[22050];

int InitOutput()
{
  if(g_Cdraw.MakeSurface(320, 240, CDD_SAMEDEPTH, &Vscreen, CDD_SYSTEM))
    return 1;
  g_Cdraw.ClearSurface(Vscreen);
  
  if(InitSound())
    return 2;

  g_Cdraw.Lock(Vscreen);

  for(int i=0;i<22050;i++) Sine[i] = (int)(sin(i * PI2 / 22050) * 255);
  return 0;
}

void DeinitOutput()
{
  Second->Stop();
  Second->Release();
  Primary->Release();
  pDS->Release();

  g_Cdraw.Unlock(Vscreen);
  g_Cdraw.KillSurface(&Vscreen);
}

void Draw(byte const *scan, word const palette[256], int scanline)
{
  if(scanline != 255)
  {
    word *line = (word*)(Vscreen->Mem + (scanline+8)*Vscreen->Pitch) + 32;
    for(int i=0;i<256;i++)
    {
      byte c=scan[i];
      if(c) line[i] = palette[c];
    }
  }
  else // flush to screen
  {
    g_Cdraw.Unlock(Vscreen);
    g_Cdraw.WaitForRetrace();
    g_Cdraw.Blit(&g_Cdraw.Primary, Vscreen, NULL, NULL);
    g_Cdraw.Lock(Vscreen);
  }
}

void Sound(SndChannel const chans[7], PCMChannel &PCMchan)
{
  HRESULT hRet;
  static DWORD nextwrite=1;
  DWORD play, write, locklen;
  short *mem1, *mem2;
  DWORD len1,  len2;

  Second->GetCurrentPosition(&play, &write);
  if(InRange(play, write, nextwrite)) nextwrite=write;
  if(nextwrite > play) locklen = BUFLEN-nextwrite + play-1;
  else locklen = play-nextwrite-1;
  if((int)locklen<0) return;

  if(locklen > 368*2) locklen = 368*2;
  while(nextwrite > BUFLEN) nextwrite -= BUFLEN;
  if(Dist(play, nextwrite) > 368*4) return;
  //DebugOut("p,w %d,%d %d-%d %d %c\n", play, write, nextwrite, nextwrite+locklen>BUFLEN?nextwrite+locklen-BUFLEN:nextwrite+locklen, Dist(play, nextwrite), write==nextwrite?'B':' ');

  hRet = Second->Lock(nextwrite, locklen, (void**)&mem1, &len1, (void**)&mem2, &len2, 0);
  if(hRet == DSERR_BUFFERLOST)
    hRet = Second->Lock(nextwrite, locklen, (void**)&mem1, &len1, (void**)&mem2, &len2, 0);
  if(FAILED(hRet)) return;

  nextwrite += locklen;

  FillSound(mem1, len1/2, chans, PCMchan);
  if(mem2) FillSound(mem2, len2/2, chans, PCMchan);

  Second->Unlock(mem1, len1, mem2, len2);
}

void FillSound(short *mem, DWORD samples, SndChannel const chans[7], PCMChannel &PCMchan)
{
  static int poses[7], lastrand[7];

  bool usepcm = PCMchan.start != PCMchan.end;
  unsigned long offset, start, end, inc;
  int looptype;
  bool forward;
  
  offset = PCMchan.off;
  forward = PCMchan.forward;
  end = PCMchan.end<<16;
  if(usepcm)
  {
    inc = PCMchan.inc, looptype = PCMchan.info.type;
    start = PCMchan.start<<16;
  }

  for(DWORD i=0;i<samples;i++)
  {
    short final=0;
    for(int chan=0;chan<7;chan++)
    {
      int vol  = chans[chan].vol;
      if(!vol) continue;
      int freq = chans[chan].freq;
      int pwid = chans[chan].info.PW;
      int pos  = poses[chan];
      SNDTYPE type = (SNDTYPE)chans[chan].info.type;
      switch(type)
      {
        case SND_SQUARE:
          if(pos<22050*(pwid+1)/128) final+=vol;
          else final-=vol;
          break;

        case SND_SINE:
          final += Sine[pos]*vol/255;
          break;

        case SND_TRI:
          {
            int middle=22050*pwid/128;
            if(pos<middle) final+=(pos*vol*2)/middle-vol;
            else final+=vol-(((pos-middle)*vol*2)/(44100-middle));
          }
          break;

        case SND_NOISE:
          if(pos+freq > 22050) final += (lastrand[chan] = rand()%(vol*2)-vol);
          else final += lastrand[chan];
          break;
      }
      
      pos += freq;
      while(pos > 22049) pos -= 22050;
      poses[chan] = pos;
    }
    if(usepcm)
    {
      final+=(signed char)g_Emu.ReadMem((word)(offset>>16))*2;
      if(forward)
      {
        offset+=inc;
        if(offset>=end)
        {
          switch(looptype)
          {
            case PCM_NONE:
              PCMchan.end = PCMchan.start;
              usepcm=false;
              break;
            
            case PCM_NORMAL:
              offset=start;
              break;

            case PCM_PINGPONG:
              offset=end-(1<<16);
              forward=false;
              break;
          }
        }
      }
      else
      {
        offset-=inc;
        if(offset<start)
        {
          switch(looptype)
          {
            case PCM_REVERSE:
              offset=end-(1<<16);
              break;
            
            case PCM_PINGPONG:
              offset=start;
              forward=true;
          }
        }
      }
    }
    mem[i] = final*16;
  }
  PCMchan.off = offset;
  PCMchan.offset = (word)(offset>>16);
  PCMchan.end = (word)(end>>16);
  PCMchan.forward = forward;
}

int InitSound()
{
  HRESULT hRet;
  DSBUFFERDESC dsbd;
  WAVEFORMATEX WFX;

  hRet = DirectSoundCreate(NULL, &pDS, NULL);
  if (FAILED(hRet)) return 1;

  hRet = pDS->SetCooperativeLevel(g_hWnd, DSSCL_EXCLUSIVE);
  if (FAILED(hRet)) return 2;

  ZeroMemory(&dsbd, sizeof(dsbd));
  dsbd.dwSize      = sizeof(DSBUFFERDESC);
  dsbd.dwFlags     = DSBCAPS_PRIMARYBUFFER;

  hRet = pDS->CreateSoundBuffer(&dsbd, &Primary, NULL);
  if (FAILED(hRet)) return 3;

  ZeroMemory(&WFX, sizeof(WFX));
  WFX.cbSize          = sizeof(WFX);
  WFX.nSamplesPerSec  = 22050;
  WFX.wBitsPerSample  = 16;
  WFX.nChannels       = 1;
  WFX.wFormatTag      = WAVE_FORMAT_PCM;
  WFX.nBlockAlign     = 2;
  WFX.nAvgBytesPerSec = 44100;

  hRet = Primary->SetFormat(&WFX);
  if (FAILED(hRet)) return 4;

  dsbd.dwFlags = DSBCAPS_GETCURRENTPOSITION2;
  dsbd.lpwfxFormat = &WFX;
  dsbd.dwBufferBytes = BUFLEN;
  hRet = pDS->CreateSoundBuffer(&dsbd, &Second, NULL);
  if(FAILED(hRet)) return 5;

  void *p1, *p2;
  DWORD l1, l2;
  hRet = Second->Lock(0, 0, &p1, &l1, &p2, &l2, DSBLOCK_ENTIREBUFFER);
  if(FAILED(hRet)) return 6;
  memset(p1, 0, l1);
  memset(p2, 0, l2);
  Second->Unlock(p1, l1, p2, l2);

  Second->Play(0, 0, DSBPLAY_LOOPING);
  return 0;
}
