﻿/*
pad is a program to quickly encrypt and decrypt files. It is not intended as
a replacement for more sophisticated encryption software, like GPG or PGP.
This program is designed for high-speed encryption of large files or large
numbers of files, where speed is more important than security. That said, it's
not too shabby security-wise, either.

http://www.adammil.net/
Copyright (C) 2011 Adam Milazzo

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

using System;
using System.Collections.Generic;
using System.IO;
using System.Security.Cryptography;
using System.Text;
using AdamMil.Utilities;

namespace pad
{

static class Program
{
  enum Command { Invalid=0, Encrypt, Decrypt, GenerateKey }
  const int KeySize = 32, BlockSize = 16, SaltSize = 16, HashSize = 20, FooterSize = BlockSize+SaltSize+HashSize+sizeof(long);
  const int ScrambleIterations = 1000000;

  static int Main(string[] args)
  {
    if(!ParseArguments(args))
    {
      ShowHelp(false);
      return 2;
    }

    if(command == Command.Invalid) return 0;

    try
    {
      EnsureKey();

      if(command == Command.GenerateKey)
      {
        Console.WriteLine("Saving key.");
        using(FileStream stream = new FileStream(keyFileName, FileMode.Create, FileAccess.Write, FileShare.None))
        {
          stream.SetLength(KeySize);
          stream.Write(key, 0, key.Length);
        }
      }
      else // encrypt or decrypt files
      {
        if(key.Length == 0) Console.WriteLine("WARN: The key is zero bytes long. No data would be altered.");
        else if(!CipherFiles()) return 1;
      }
      return 0;
    }
    catch(Exception ex)
    {
      Console.WriteLine("ERROR: An error occurred. " + ex.Message);
      return 1;
    }
  }

  static string CipherDirectoryName(string name, Aes aes)
  {
    if(command == Command.Encrypt)
    {
      byte[] iv = new byte[BlockSize], rand = new byte[2]; // to avoid ciphering all directory names with the same key, generate an IV
      rng.GetBytes(rand);                                  // based on a 16-bit random number
      iv[0] = rand[0];
      iv[1] = rand[1];
      using(ICryptoTransform transform = aes.CreateEncryptor(key, iv))
      {
        name = CipherFileName(name, transform, false);
        return name == null ? null : BinaryUtility.ToHex(rand[0]) + BinaryUtility.ToHex(rand[1]) + name; // add the IV to the name
      }
    }
    else
    {
      if(((name.Length-4) & (BlockSize*2-1)) != 0) return null;
      byte[] bytes;
      if(!BinaryUtility.TryParseHex(name.Substring(0, 4), false, out bytes)) return null; // extract the IV from the name
      byte[] iv = new byte[BlockSize];
      iv[0] = bytes[0];
      iv[1] = bytes[1];
      using(ICryptoTransform transform = aes.CreateDecryptor(key, iv)) return CipherFileName(name.Substring(4), transform, false);
    }
  }

  static string CipherFileName(string name, ICryptoTransform transform, bool preserveExtensions)
  {
    string extension = null;
    if(preserveExtensions)
    {
      extension = Path.GetExtension(name);
      if(extension.Length != 0) name = Path.GetFileNameWithoutExtension(name);
    }

    if(command == Command.Encrypt)
    {
      int encodedLength = Encoding.UTF8.GetByteCount(name), padding = (BlockSize - (encodedLength & (BlockSize-1))) & (BlockSize-1);
      byte[] bytes = new byte[encodedLength + padding], encryptedBytes = new byte[bytes.Length];
      Encoding.UTF8.GetBytes(name, 0, name.Length, bytes, 0);
      transform.TransformBlock(bytes, 0, bytes.Length, encryptedBytes, 0);
      return BinaryUtility.ToHex(encryptedBytes) + extension;
    }
    else
    {
      if((name.Length & (BlockSize*2-1)) == 0)
      {
        byte[] bytes;
        if(BinaryUtility.TryParseHex(name, false, out bytes))
        {
          byte[] nameBytes = new byte[bytes.Length];
          transform.TransformBlock(bytes, 0, bytes.Length, nameBytes, 0);
          int length = nameBytes.Length;
          while(length > 0 && nameBytes[length-1] == 0) length--;
          if(length > 0)
          {
            name = Encoding.UTF8.GetString(nameBytes, 0, length);
            if(name.IndexOfAny(invalidNameChars) == -1) return name + extension;
          }
        }
      }
      return null;
    }
  }

  static ICryptoTransform CipherFile(string filePath, Aes aes, byte[] buffer)
  {
    ICryptoTransform transform;
    using(FileStream file = new FileStream(filePath, FileMode.Open, FileAccess.ReadWrite, FileShare.None))
    {
      long originalSize, endPosition;
      if(command == Command.Encrypt)
      {
        aes.GenerateIV();
        transform    = aes.CreateEncryptor();
        originalSize = endPosition = file.Length;
      }
      else // if we're decrypting...
      {
        byte[] iv = null, salt, hash;
        originalSize = endPosition = 0;
        if(file.Length >= FooterSize) // and the file could possibly contain the footer...
        {
          file.Seek(-FooterSize, SeekOrigin.End);
          originalSize = ReadInt64(file);
          endPosition  = file.Length - FooterSize;
          // if the file size and original file sizes match sufficiently
          if(((int)endPosition & (BlockSize-1)) == 0 && originalSize <= endPosition && originalSize >= endPosition - (BlockSize-1))
          {
            iv   = Read(file, BlockSize, true); // then read the IV and checksum hash
            salt = Read(file, SaltSize, true);
            hash = Read(file, HashSize, true);
            using(SHA1 sha = SHA1.Create()) // and verify the hash
            {
              sha.TransformBlock(key, 0, key.Length, null, 0);
              sha.TransformBlock(iv, 0, iv.Length, null, 0);
              sha.TransformFinalBlock(salt, 0, salt.Length);
              if(!BinaryUtility.AreEqual(hash, sha.Hash)) iv = null; // if the hash doesn't match, then we don't have the right key
            }
          }
        }

        if(iv == null)
        {
          Console.WriteLine(" Failed! Password is wrong or file is not encrypted.");
          return null;
        }

        transform = aes.CreateDecryptor(key, iv);
        file.Position = 0;
      }

      // encrypt or decrypt all the bytes in the file
      for(long readPosition = file.Position; readPosition < endPosition; )
      {
        // we'll read file data into the first half of the buffer and encrypt or decrypt it into the second half
        int bufferBytes = 0, splitPoint = buffer.Length/2, maxBytes = (int)Math.Min(splitPoint, endPosition-readPosition);
        do
        {
          int read = file.Read(buffer, bufferBytes, maxBytes-bufferBytes);
          if(read == 0)
          {
            if(command == Command.Encrypt) // when encrypting, we need to pad the file data out to the block size
            {
              int padding = (BlockSize - (bufferBytes & (BlockSize-1))) & (BlockSize-1);
              Array.Clear(buffer, bufferBytes, padding);
              bufferBytes += padding;
            }
            break;
          }
          bufferBytes += read;
        } while(bufferBytes < splitPoint);

        transform.TransformBlock(buffer, 0, bufferBytes, buffer, splitPoint);
        file.Position = readPosition;
        file.Write(buffer, splitPoint, bufferBytes);
        readPosition += bufferBytes;
      }

      if(command == Command.Encrypt)
      {
        byte[] salt = new byte[SaltSize]; // the salt just increases the length of the footer enough that it should be infeasible for
        rng.GetBytes(salt);               // anybody to use a lookup table of SHA1 hashes to obtain the original key
        Write(file, originalSize); // append a small footer to help us recover the original file
        file.Write(aes.IV, 0, BlockSize);
        file.Write(salt, 0, SaltSize);
        using(SHA1 sha = SHA1.Create())
        {
          sha.TransformBlock(key, 0, key.Length, null, 0);
          sha.TransformBlock(aes.IV, 0, BlockSize, null, 0);
          sha.TransformFinalBlock(salt, 0, SaltSize);
          file.Write(sha.Hash, 0, HashSize);
        }
      }
      else // if we're decrypting...
      {
        file.SetLength(originalSize); // truncate any extra bytes required by the block cipher
      }
    }

    return transform;
  }

  static bool CipherFiles()
  {
    Console.WriteLine(command == Command.Encrypt ? "Encrypting files..." : "Decrypting files...");
    HashSet<string> visitedFiles = new HashSet<string>();
    Dictionary<string,string> filesToRename = null, directoriesToRename = null;
    byte[] buffer = new byte[8192*BlockSize*2];
    bool fileFound = false, failed = false;

    if(rename) // if we need to rename files...
    {
      filesToRename = new Dictionary<string, string>(); // keep track of files to rename
      if(recursive) directoriesToRename = new Dictionary<string, string>(); // and directories too, if we're ciphering recursively
    }

    Aes aes = Aes.Create();
    aes.Key     = key;
    aes.Padding = PaddingMode.None; // avoid .NET's default behavior of unnecessarily adding an extra block

    foreach(string fileSpec in fileSpecs)
    {
      string directory;
      string[] filePaths = GetFilePaths(fileSpec, out directory);
      if(filePaths == null)
      {
        failed = true;
        continue;
      }

      foreach(string filePath in filePaths) // for each file...
      {
        string normalizedPath = PathUtility.NormalizePath(filePath);
        if(!visitedFiles.Add(normalizedPath)) continue; // if we've processed the file already, skip to the next one

        string fileName = Path.GetFileName(filePath);
        Console.Write(filePath);
        if(!File.Exists(filePath))
        {
          Console.WriteLine(" Failed! File does not exist.");
          failed = true;
        }
        else
        {
          Console.Write("...");
          fileFound = true;

          try
          {
            ICryptoTransform transform = CipherFile(filePath, aes, buffer);
            if(transform == null)
            {
              failed = true;
            }
            else
            {
              // if the file was processed successfully, add it to the files that we need to rename
              if(rename)
              {
                string name = CipherFileName(Path.GetFileName(filePath), transform, preserveExtensions);
                if(name != null) filesToRename.Add(normalizedPath, name);
                // if the file is in a subdirectory, we'll need to rename the subdirectory
                if(recursive && directory != null)
                {
                  string normalizedDirectory = Path.GetDirectoryName(normalizedPath);
                  if(!directoriesToRename.ContainsKey(normalizedDirectory))
                  {
                    directoriesToRename[normalizedDirectory] = CipherDirectoryName(Path.GetFileName(Path.GetDirectoryName(filePath)), aes);
                  }
                }
              }

              transform.Dispose();

              Console.WriteLine(" OK.");
            }
          }
          catch(Exception ex)
          {
            Console.WriteLine(" Failed! " + ex.Message);
            failed = true;
          }
        }
      }
    }

    if(rename && (filesToRename.Count != 0 || recursive && directoriesToRename.Count != 0)) // if we need to rename files...
    {
      RenameFilesAndDirectories(filesToRename, directoriesToRename, ref failed);
    }

    if(!fileFound) Console.WriteLine("WARN: No files found.");
    return !failed;
  }

  static void EnsureKey()
  {
    if(readPassword) // if we need to read a password...
    {
      string password;
      while(true)
      {
        Console.Write("Enter password: ");
        password = ReadPassword();
        Console.Write("Enter password again: ");
        string password2 = ReadPassword();
        if(string.Equals(password, password2, StringComparison.Ordinal)) break;
        Console.WriteLine("Passwords do not match.");
      }

      pwd = Encoding.UTF8.GetBytes(password);
    }

    // now we have our password, if there is one. if a key was loaded from a file and a password was specified, we'll mix the two together
    bool mixKeyWithPassword = key != null && pwd != null && pwd.Length != 0, useRandomScrambleKey = false;

    if(key == null) // if a key was not loaded from a file, generate one
    {
      key = new byte[KeySize];
      if(pwd == null) // if there was no password, generate a random key
      {
        Console.WriteLine("Generating a random key.");
        rng.GetBytes(key);
        useRandomScrambleKey = true; // the key will be scrambled later to obscure any known patterns in the generator
      }
      else // otherwise, create the initial key based on the password
      {
        Console.WriteLine("Generating a password-based key.");
        ExpandPassword(pwd, key);
      }
    }

    if(mixKeyWithPassword)
    {
      Console.WriteLine("Mixing key with password.");
      // if we're not going to scramble the key later (currently, this is always true here), then scramble the password now
      if(!scramble)
      {
        if(pwd.Length != KeySize)
        {
          byte[] newPwd = new byte[KeySize];
          ExpandPassword(pwd, newPwd);
          pwd = newPwd;
        }
        Scramble(pwd, ScrambleIterations, false);
      }

      SimpleXorBytes(key, pwd);
    }
    pwd = null;

    if(scramble) Scramble(key, ScrambleIterations, useRandomScrambleKey); // note: if a key was loaded from a file, scramble will be false
  }

  static void ExpandPassword(byte[] pwd, byte[] initialKey)
  {
    if(pwd.Length > KeySize)
    {
      using(SHA256 sha = SHA256.Create()) Array.Copy(sha.ComputeHash(pwd), initialKey, KeySize);
    }
    else if(pwd.Length != 0)
    {
      for(int i=0; i<initialKey.Length; i += pwd.Length) Array.Copy(pwd, 0, initialKey, i, Math.Min(pwd.Length, initialKey.Length-i));
    }
  }

  static string[] GetFilePaths(string fileSpec, out string directory)
  {
    string[] filePaths;
    directory = null;
    if(fileSpec.Contains('*') || fileSpec.Contains('?')) // if it contains wildcards...
    {
      directory = Path.GetDirectoryName(fileSpec); // get the directory of the file spec
      if(string.IsNullOrEmpty(directory)) // if there is none...
      {
        directory = Environment.CurrentDirectory; // use the current directory
      }
      else if(directory.Contains('*') || directory.Contains('?')) // otherwise, if the directory contains wildcards...
      {
        Console.WriteLine("ERROR: File spec " + fileSpec + " contains a wildcard in a directory name.");
        return null;
      }

      // get the matching files
      filePaths = PathUtility.GetFiles(directory, Path.GetFileName(fileSpec),
                                        recursive ? SearchOption.AllDirectories : SearchOption.TopDirectoryOnly);
      // if we're renaming recursively, keep track of the directory name so we can identify subdirectories
      if(rename && recursive) directory = PathUtility.NormalizePath(directory);
    }
    else if(recursive && Directory.Exists(fileSpec)) // if it's the path to a directory...
    {
      filePaths = Directory.GetFiles(fileSpec, "*", SearchOption.AllDirectories);
      if(rename) directory = PathUtility.NormalizePath(fileSpec);
    }
    else
    {
      filePaths = new string[] { fileSpec };
    }
    return filePaths;
  }

  static bool ParseArguments(string[] args)
  {
    string pwdCommand = null; // the -p or -pf argument used, if any
    int i = 0;
    bool failed = false, showFullHelp = false;
    
    // first parse the command name
    if(args.Length == 0)
    {
      failed = true;
    }
    else
    {
      switch(args[0].ToLowerInvariant())
      {
        case "dec": command = Command.Decrypt; i=1; break;
        case "enc": command = Command.Encrypt; i=1; break;
        case "gen": command = Command.GenerateKey; i=1; break;
      }
    }

    while(i < args.Length)
    {
      string arg = args[i];
      if(arg.Length == 0 || arg[0] != '-' && arg[0] != '/') break;
      i++;
      switch(arg.Substring(1).ToLowerInvariant())
      {
        case "e":
          preserveExtensions = true;
          break;

        case "k":
          if(command == Command.GenerateKey)
          {
            failed = true;
            Console.WriteLine("ERROR: -k cannot be used when generating a key.");
          }
          else if(key != null)
          {
            failed = true;
            Console.WriteLine("ERROR: Only one key can be specified.");
          }
          else if(!ReadKey(args, ref i, out key))
          {
            failed = true;
          }
          else if(key.Length != KeySize)
          {
            failed = true;
            Console.WriteLine("ERROR: The given file is not a valid key.");
          }
          scramble = false;
          break;

        case "n":
          if(command == Command.GenerateKey)
          {
            failed = true;
            Console.WriteLine("ERROR: -k cannot be used when generating a key.");
          }
          rename = true;
          break;

        case "p":
          if(pwdCommand != null)
          {
            failed = true;
            Console.WriteLine("ERROR: " + arg + " cannot be used with " + pwdCommand + ".");
          }
          pwdCommand   = arg;
          readPassword = true;
          break;

        case "pf":
          if(pwd != null)
          {
            failed = true;
            Console.WriteLine("ERROR: Only one password can be specified.");
          }
          else if(pwdCommand != null)
          {
            failed = true;
            Console.WriteLine("ERROR: " + arg + " cannot be used with " + pwdCommand + ".");
          }
          else if(!ReadPasswordFile(args, ref i, out pwd))
          {
            failed = true;
          }
          pwdCommand = arg;
          break;

        case "r":
          if(command == Command.GenerateKey)
          {
            failed = true;
            Console.WriteLine("ERROR: -r cannot be used when generating a key.");
          }
          recursive = true;
          break;

        case "help": case "?": case "-help": showFullHelp = true; break;

        default:
          Console.WriteLine("ERROR: Invalid command: " + arg);
          failed = true;
          break;
      }
    }

    if(preserveExtensions && !rename)
    {
      failed = true;
      Console.WriteLine("ERROR: -e cannot be used without -n.");
    }

    if(command == Command.Encrypt || command == Command.Decrypt)
    {
      if(pwdCommand == null && key == null)
      {
        failed = true;
        Console.WriteLine("ERROR: No key was specified.");
      }
      else if(i == args.Length)
      {
        failed = true;
        Console.WriteLine("ERROR: Expected file specification.");
      }
      else
      {
        fileSpecs = new string[args.Length-i];
        for(int j=0; i<args.Length; j++, i++) fileSpecs[j] = args[i];
      }
    }
    else if(command == Command.GenerateKey)
    {
      if(i != args.Length-1)
      {
        failed = true;
        Console.WriteLine(i == args.Length ? "ERROR: Expected key file name." : "ERROR: Unexpected argument " + args[i+1]);
      }
      else
      {
        keyFileName = args[i];
      }
    }
    else if(!showFullHelp)
    {
      failed = true;
      Console.WriteLine("ERROR: A valid command must be specified first.");
    }

    if(showFullHelp) ShowHelp(true);
    return !failed;
  }

  static byte[] Read(Stream file, int length, bool throwOnEOS)
  {
    byte[] bytes = new byte[length];
    int index = 0;
    while(length != 0)
    {
      int read = file.Read(bytes, index, length);
      if(read == 0)
      {
        if(throwOnEOS) throw new EndOfStreamException();
        else bytes = bytes.Trim(index);
        break;
      }
      index  += read;
      length -= read;
    }
    return bytes;
  }

  static bool ReadFile(string[] args, ref int index, out byte[] array, Func<string,byte[]> reader)
  {
    if(index < args.Length)
    {
      string fileName = args[index++];
      try
      {
        if(!File.Exists(fileName))
        {
          Console.WriteLine("ERROR: " + fileName + " does not exist or cannot be accessed.");
        }
        else
        {
          array = reader(fileName);
          return true;
        }
      }
      catch(Exception ex)
      {
        Console.WriteLine("ERROR: Unable to read " + fileName + ". " + ex.Message);
      }
    }
    array = null;
    return false;
  }

  static bool ReadKey(string[] args, ref int index, out byte[] key)
  {
    return ReadFile(args, ref index, out key, path =>
    {
      using(FileStream stream = File.OpenRead(path)) return Read(stream, KeySize, false);
    });
  }

  static bool ReadPasswordFile(string[] args, ref int index, out byte[] pwd)
  {
    return ReadFile(args, ref index, out pwd, path =>
    {
      using(FileStream stream = File.OpenRead(path))
      {
        if(stream.Length <= KeySize)
        {
          return Read(stream, KeySize, false);
        }
        else
        {
          using(SHA256 sha = SHA256.Create()) return BinaryUtility.Hash(stream, sha);
        }
      }
    });
  }

  static long ReadInt64(Stream file)
  {
    byte[] bytes = Read(file, 8, true);
    return (long)(ReadUInt32(bytes, 0) | ((ulong)ReadUInt32(bytes, 4) << 32));
  }

  static uint ReadUInt32(byte[] array, int index)
  {
    return (uint)(array[index] | (array[index+1]<<8) | (array[index+2]<<16) | (array[index+3]<<24));
  }

  static string ReadPassword()
  {
    StringBuilder sb = new StringBuilder();
    while(true)
    {
      ConsoleKeyInfo info = Console.ReadKey(true);
      if(info.KeyChar == '\n' || info.KeyChar == '\r') break;
      else if(info.KeyChar == '\b' && sb.Length != 0) sb.Remove(sb.Length-1, 1);
      else if(info.KeyChar != 0 && info.KeyChar >= ' ') sb.Append(info.KeyChar);
    }
    Console.WriteLine();
    return sb.ToString();
  }

  static void RenameFilesAndDirectories(Dictionary<string, string> filesToRename, Dictionary<string, string> directoriesToRename,
                                        ref bool failed)
  {
    Console.WriteLine("Renaming files and directories...");

    // rename the files first. (if we renamed directories first, the file paths would be broken)
    foreach(KeyValuePair<string, string> pair in filesToRename)
    {
      try
      {
        File.Move(pair.Key, Path.Combine(Path.GetDirectoryName(pair.Key), pair.Value));
      }
      catch
      {
        failed = true;
        Console.WriteLine("ERROR: Couldn't rename " + pair.Key);
      }
    }

    // and then rename the directories
    if(recursive)
    {
      string[] dirPaths = new string[directoriesToRename.Count];
      directoriesToRename.Keys.CopyTo(dirPaths, 0);
      // sort directories by length, descending, so that we rename subdirectories before parent directories
      Array.Sort(dirPaths, (a, b) => b.Length - a.Length);

      foreach(string directoryPath in dirPaths)
      {
        string newName = directoriesToRename[directoryPath];
        if(newName != null)
        {
          try
          {
            Directory.Move(directoryPath, Path.Combine(Path.GetDirectoryName(directoryPath), newName));
          }
          catch
          {
            failed = true;
            Console.WriteLine("ERROR: Couldn't rename " + directoryPath);
          }
        }
      }
    }
  }

  static unsafe void Scramble(byte[] data, int iterations, bool useRandomKey)
  {
    if(data.Length < 16 || (data.Length & 3) != 0) throw new ArgumentException();

    // encrypt the data using the XXTEA algorithm with minor variations
    uint[] teaKey = new uint[4] { 0x0d3c189e, 0x8f4744ef, 0x61e76946, 0x6f209e14 };
    if(useRandomKey)
    {
      byte[] random = new byte[16];
      rng.GetBytes(random);
      fixed(uint* pKey=teaKey)
      fixed(byte* pRand=random)
      {
        Unsafe.Copy(pRand, pKey, random.Length);
      }
    }

    fixed(uint* key=teaKey)
    fixed(byte* pData=data)
    {
      uint* words = (uint*)pData;
      uint lastWord = (uint)data.Length/4 - 1, sum = 0;

      // combine the last 16 bytes of the plaintext with the key so that the initial key is not a known constant
      key[0] ^= words[lastWord-3];
      key[1] ^= words[lastWord-2];
      key[2] ^= words[lastWord-1];
      key[3] ^= words[lastWord];

      while(--iterations >= 0)
      {
        uint rounds = 6 + 52/(lastWord+1), z = words[lastWord];
        do
        {
          sum += 0x9e3779b9;
          uint e = (sum >> 2) & 3, i, y;
          for(i=0; i < lastWord; i++)
          {
            y = words[i+1];
            z = (words[i] += ((z>>5^y<<2) + (y>>3^z<<4)) ^ ((sum^y) + (key[(i&3)^e] ^ z)));
          }
          y = words[0];
          z = (words[i] += ((z>>5^y<<2) + (y>>3^z<<4)) ^ ((sum^y) + (key[(i&3)^e] ^ z)));
        } while(--rounds != 0);

        // use the last 16 bytes of the encrypted data as the key for next time (in an arbitrary order)
        key[0] = words[lastWord-(sum&3)];
        key[1] = words[lastWord-((sum+1)&3)];
        key[2] = words[lastWord-((sum+2)&3)];
        key[3] = words[lastWord-((sum+3)&3)];
      }
    }
  }

  static void ShowHelp(bool showFullDetails)
  {
    Console.WriteLine(@"USAGE: pad gen [options] <keyfile>
       pad enc [options] <filespec> [<filespec> ...]
       pad dec [options] <filespec> [<filespec> ...]
EXAMPLES: pad gen X:\key
          pad enc -k X:\key D:\movies\*.avi D:\downloads\*
          pad dec -k X:\key D:\movies\foo.avi");
    if(!showFullDetails)
    {
      Console.WriteLine("For more details, use pad -?");
    }
    else
    {
      Console.WriteLine(@"
This program can create a key and encrypt or decrypt files with it.
Encrypting and decrypting are done in-place and the sizes of files are not
changed.

WARNING: This program is not intended as a replacement for more sophisticated
encryption software, like GPG or PGP. This program is designed for high-speed
encrypting of large files or large numbers of files, when speed is more
important than security.

COMMANDS:
gen [options] <keyfile>
  Generates a new key and saves it to the named file. If -p or -pf is used,
  the key will be generated based on a password. Using the same password again
  will allow you to regenerate the key. The key file should be kept secret,
  and it may be valuable to encrypt the key file with a real encryption
  program, like GPG.

  Examples:
  pad gen key             Generates a random key.
  pad gen -p key          Generates a key based on a password.
  pad gen -pf pfile key   Generates a key based on pfile.

enc [options] <filespec> [<filespec> ...]
  Encrypts the specified files. A key must be specified, with -k to use
  an existing key, -p or -pf to create a new key based on a password, or both
  -k and -p or -pf to use an existing key mixed with a password.

  Examples:
  pad enc -k kfile foo     Encrypts foo with a key taken from kfile.
  pad enc -p foo           Encrypts foo with a key generated from a password.
  pad enc -k kfile -p foo  Encrypts foo with a key taken from kfile and mixed
                           with a password.
  pad enc -k kfile -r -n D:\downloads\*
    Recursively encrypts all files in D:\downloads with a key taken from
    kfile, and encrypts the file names as well.
  pad enc -k kfile -r -n -e D:\downloads\*.png
    Recursively encrypts and renames all PNG files while preserving their
    extensions to make them easier to decrypt later.

dec [options] <filespec> [<filespec> ...]
  Decrypts the specified files. A key must be specified, with -k to use
  an existing key, -p or -pf to create a new key based on a password, or both
  -k and -p or -pf to use an existing key mixed with a password.

  Examples: See enc, above.

OPTIONS:
-e            File extensions will be preserved if -n is also used. This can
              make files easier to decrypt if you use a selective wildcard
              to encrypt them.
-k <keyfile>  Specifies that the named key file will be used during
              encrypting or decrypting.
-n            Specifies that the names of files should be encrypted and
              decrypted as well as the contents. If combined with -r,
              subdirectories containing matching files will also be renamed.
-p            When generating a key, the key will be generated using a
              password, which the program will prompt for. If the same
              options are used a second time, the same key will be
              generated. When encrypting or decrypting, if -k is used, the
              password will be mixed with the key.
-pf <file>    When generating a key, the key will be generated using the
              contents of the file as a seed. If the same options are used
              a second time, the same key will be generated. When
              encrypting or decrypting, if -k is used, the password file
              will be mixed with the key.
-r            Encrypts or decrypts files recursively.
-? -help /?   Displays this help message.

FILE SPECIFICATIONS:
A file specification is an absolute or relative path to a file, possibly
containing one or more of the standard wildcards in the file name portion.
(Wildcards must not be used in the directory portion.) An asterisk (*) will
match multiple characters and a question mark will match a single character.
Names with spaces should be enclosed in quotation marks.

If two file specifications reference the same file, the file will only be
processed once.

Good examples: *.png model.* file??.ext c:\path\* ""name with spaces.ext""
Bad example: c:\pictures\spain*\*.jpg");
    }
  }

  static void SimpleXorBytes(byte[] data, byte[] key)
  {
    for(int i=0, j=0; i<data.Length; i++)
    {
      data[i] ^= key[j];
      if(++j == key.Length) j = 0;
    }
  }

  static void Write(byte[] array, int index, uint value)
  {
    array[index]   = (byte)value;
    array[index+1] = (byte)(value>>8);
    array[index+2] = (byte)(value>>16);
    array[index+3] = (byte)(value>>24);
  }

  static void Write(Stream file, long value)
  {
    byte[] bytes = new byte[8];
    Write(bytes, 0, (uint)value);
    Write(bytes, 4, (uint)(value >> 32));
    file.Write(bytes, 0, 8);
  }

  static byte[] key, pwd;
  static string[] fileSpecs;
  static string keyFileName;
  static Command command;
  static bool preserveExtensions, readPassword, recursive, rename, scramble = true;

  static readonly RandomNumberGenerator rng = RandomNumberGenerator.Create();
  static readonly char[] invalidNameChars = Path.GetInvalidFileNameChars();
}

} // namespace pad
