#include "regexp.h"
#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <conio.h>
#include <memory.h>

enum
{ SYM_NONE=0, SYM_ANY='.'
};

static node *  newNode();
static void    delNode(node *n);
static void    delNodeR(node *n);

static void    addEdge(node *from, node *to, char symbol);

static graph * newGraph(char symbol);
static void    delGraph(graph *g);

static void    merge(node *end, node *start);
static void    concat(graph *lhs, graph *rhs);
static void    kleene(graph *n);
static void    or(graph *lhs, graph *rhs);

static char    nextchar();
static int     renumber(node *n, int start=1);

struct set
{ set();
 ~set();

  void add(node *);
  void clear();
  bool isIn(node *);

  int    nPos, nMax;
  node **Arr;
};

static set * eclosure(set *src, set *dest);
static void  eclosure(node *n, set *dest);
static set * move(set *src, char c, set *dest);
static void  move(node *n, char c, set *dest);

node * newNode()
{ node *n   = new node;
  n->state  = 0;
  n->nEdges = 0;
  n->edges  = NULL;
  return n;
}

void delNode(node *n)
{ delete n->edges;
  delete n;
}

void delNodeR(node *n)
{ for(int i=0; i<n->nEdges; i++) delNodeR(n->edges[i].to);
  delNode(n);
}

void addEdge(node *from, node *to, char symbol)
{ assert(from->nEdges < 2);

  if(!from->edges) from->edges = new edge[2];
  from->edges[from->nEdges].symbol = symbol;
  from->edges[from->nEdges].to     = to;
  from->nEdges++;
}

graph * newGraph(char symbol)
{ graph *g = new graph;
  node  *s = newNode(), *e = newNode();
  addEdge(s, e, symbol);
  g->start=s, g->end=e;
  return g;
}

void delGraph(graph *g)
{ delNodeR(g->start);
}

void merge(node *end, node *start)
{ assert(!end->nEdges);
  delete[] end->edges;
  end->edges=start->edges, end->nEdges=start->nEdges;
  delete start;
}

void concat(graph *lhs, graph *rhs)
{ merge(lhs->end, rhs->start);
  lhs->end = rhs->end;
  delete rhs;
}

void kleene(graph *g)
{ node *s=newNode(), *e=newNode();
  addEdge(s, g->start, SYM_NONE);
  addEdge(s, e, SYM_NONE);
  addEdge(g->end, e, SYM_NONE);
  addEdge(g->end, g->start, SYM_NONE);
  g->start=s, g->end=e;
}

void or(graph *lhs, graph *rhs)
{ node *s=newNode(), *e=newNode();
  addEdge(s, lhs->start, SYM_NONE);
  addEdge(s, rhs->start, SYM_NONE);
  addEdge(lhs->end, e, SYM_NONE);
  addEdge(rhs->end, e, SYM_NONE);
  lhs->start=s, lhs->end=e;
  delete rhs;
}

char nextchar()
{ char c;
  return isprint(c=getche()) ? c : 0;
}

set::set()
{ nPos=nMax=0;
  Arr =NULL;
}

set::~set()
{ delete[] Arr;
}

void set::add(node *n)
{ assert(!isIn(n));
  if(nPos==nMax)
  { node **tmp=new node*[nMax=nMax?nMax*2:16];
    if(Arr)
    { memcpy(tmp, Arr, nPos*sizeof(node*));
      delete[] Arr;
    }
    Arr=tmp;
  }
  Arr[nPos++]=n;
}

void set::clear()
{ nPos=0;
}

bool set::isIn(node *n)
{ for(int i=0; i<nPos; i++) if(Arr[i]==n) return true;
  return false;
}

set * eclosure(set *src, set *dest)
{ dest->clear();
  for(int i=0; i<src->nPos; i++) eclosure(src->Arr[i], dest); 
  return dest;
}

void eclosure(node *n, set *dest)
{ if(!dest->isIn(n)) dest->add(n);
  for(int i=0; i<n->nEdges; i++)
    if(n->edges[i].symbol==0) eclosure(n->edges[i].to, dest);
}

set * move(set *src, char c, set *dest)
{ dest->clear();
  for(int i=0; i<src->nPos; i++) move(src->Arr[i], c, dest); 
  return dest;
}

void move(node *n, char c, set *dest)
{ if(n->nEdges && !dest->isIn(n->edges[0].to))
  { char s=n->edges[0].symbol;
    if(!s) return;
    switch(s)
    { case SYM_ANY: break; // accepts any
      default:      if(s!=c) return; break;
    }
    dest->add(n->edges[0].to);
  }
}

int renumber(node *n, int start)
{ if(!n || n->state) return start;
  n->state=start++;
  for(int i=0; i<n->nEdges; i++) start=renumber(n->edges[i].to, start);
  return start;
}

graph * construct()
{ graph *g=NULL, *tmp;
  char   cur;

  while(cur=nextchar())
  { switch(cur)
    { case '(':
        tmp = construct();
        if(g)
        { if(tmp) concat(g, tmp);
        }
        else if(tmp) g=tmp;
        break;

      case '*': kleene(g); break;
      case '|':
        assert(g);
        tmp = construct();
        or(g, tmp);
        break;

      case ')': return g;
      default:
        if(cur=='@') cur=SYM_ANY;
        tmp = newGraph(cur);
        if(g)
        { if(tmp) concat(g, tmp);
        }
        else if(tmp) g=tmp;
        break;
    }
  }
  return g;
}

void prepare(graph *g)
{ renumber(g->start);
}

bool match(char *str, graph *regex)
{ set  *s1=new set, *s2=new set;
  char  c;

  s2->add(regex->start);
  eclosure(s2, s1);
  while(c=*str++) eclosure(move(s1, c, s2), s1);

  for(int i=0; i<s1->nPos; i++) if(s1->Arr[i]==regex->end) return true;
  return false;
}
