#include <windows.h>
#include <time.h>
#include <dplay.h>
#include <assert.h>
#include <stdlib.h>
#include "Debug.h"
#include "Items.h"
#include "Server.h"
#include "Network.h"
#include "Players.h"
#include "PrintQueue.h"
#include "Frame\Externs.h"
#include "Frame\Defines.h"
#include "Vectors\Vector.h"
#include "Entities\Entity.h"
#include "Entities\Player.h"
#include "Network\NetMsgs.h"
#include "Binfile\Binfmem.h"
#include "IntrFace\BasicDP.h"

// a pulse is generated this many times per second
#define VFPS 20
/* distances are squared */
// 120 pixels.. or about 4 body lengths
#define DROPDIST 14400
// 30 pixels.. or about 1 body length
#define PICKUPDIST 900

/* BLACK, DKGRAY, GRAY, LTGRAY, WHITE / DKRED, RED, LTRED, PINK /
   DKGREEN, GREEN, LTGREEN / DKORANGE, ORANGE / DKBLUE, BLUE, LTBLUE, CYAN /
   DKPURPLE, PURPLE, LTPURPLE / DKYELLOW, YELLOW, LTYELLOW
*/
enum Colors
{
   BLACK = MYRGB(0,0,0), WHITE = MYRGB(255,255,255),
   DKGRAY = MYRGB(64,64,64), GRAY = MYRGB(128,128,128), LTGREY = MYRGB(192,192,192), 
   DKRED = MYRGB(128,0,0), RED = MYRGB(192,0,0), LTRED = MYRGB(255,0,0), PINK = MYRGB(255,128,192),
   DKGREEN = MYRGB(96,0,0), GREEN = MYRGB(160,0,0), LTGREEN = MYRGB(224,0,0),
   DKORANGE = MYRGB(128,64,0), ORANGE = MYRGB(255,128,0),
   DKBLUE = MYRGB(128,0,0), BLUE = MYRGB(176,0,0), LTBLUE = MYRGB(224,0,0), CYAN = MYRGB(0,192,192),
   DKPURPLE = MYRGB(128,0,128), PURPLE = MYRGB(192,0,192), LTPURPLE = MYRGB(255,0,255),
   DKYELLOW = MYRGB(128,128,0), YELLOW = MYRGB(224,244,0), LTYELLOW = MYRGB(255,255,0)
};

static HANDLE hThread, hKill;
static DWORD WINAPI PulseThread(LPVOID);
static void OnPulse();

static void ResetSession();                   // delete all players, items, map..

// send to all players
static void SendToAll(void *msg, DWORD msglen, bool guar);
// send to all players except one
static void SendToAllExcept(void *msg, DWORD msglen, bool guar, DWORD skipID);

/* message handlers */
static void HandleLogon(DWORD SenderID);      // handle logon request from client
static void HandleLogoff(DWORD SenderID);     // client is logging off
static void HandlePickUp(DWORD SenderID);     // player is picking something up
static void HandleDrop(DWORD SenderID);       // player is dropping something
static void HandleEquip(DWORD SenderID);      // player equips something
static void HandleDeEquip(DWORD SenderID);    // player deequips something

/* helpers to implement message handlers */
static void DoLogonAccept(DWORD PlayerID);    // accept a player's logon request
static void DoCreateInventory(DWORD PlayerID); // create a player's starting inventory
static void DoLoggedMessage(DWORD NetID, DWORD Color, char const *msg); // send a player a message
static void DoCreateObject(Entities::Item *item); // tell everyone that an item has been created [on the map!]
static void DoRemoveObject(Entities::Item *item); // tell everyone that an item has been removed [from the map!]

// Send buffer guaranteed to be at least 512 bytes
static void EnlargeSendBuffer(DWORD newsize); // enlarge send buffer if needed
static char      *SendBuffer;
static DWORD      SendBufLen;
static BasicDP   *BDP;

CRITICAL_SECTION S_CS;

// stuff for debug info
#ifndef NDEBUG
char *EquipPlaceStr[Entities::EP_EQUIPPLACES] =
{
   "Finger", "Hands", "Head", "Neck", "Torso", "Arms", "Feet", "Weapon"
};
#endif
int SessionStarted(BasicDP *bdp)
{
   ResetSession();
   if (!hThread)
   {
      InitializeCriticalSection(&S_CS);
      SECURITY_ATTRIBUTES sa;
      DWORD id;

      sa.nLength = sizeof(SECURITY_ATTRIBUTES);
      sa.lpSecurityDescriptor = NULL;
      sa.bInheritHandle = FALSE;

      hKill   = CreateEvent(0, 0, 0, 0);
      hThread = CreateThread(&sa, 0, PulseThread, NULL, 0, &id);
      if (!hThread) return 1;
      BDP = bdp;
      DebugOut("Session started successfully\n");
   }
   return 0;
} /* InitPrintQueue */


void SessionEnding()
{
   if (hThread)
   {
      EnterCriticalSection(&S_CS);
      SetEvent(hKill);
      WaitForSingleObject(hThread, INFINITE);
      CloseHandle(hKill);
      CloseHandle(hThread);
      hThread = 0;
      LeaveCriticalSection (&S_CS);
      DeleteCriticalSection(&S_CS);
      DebugOut("Session ended successfully\n");
   }
} /* SessionEnding */


void OnPulse()
{
} /* OnPulse */


void HandleLogon(DWORD SenderID)
{
   MsgLogOn      *msg    = (MsgLogOn *)     RecvBuffer;
   MsgAddPlayer  *outmsg = (MsgAddPlayer *) SendBuffer;
   MPL_OnePlayer *mpl    = (MPL_OnePlayer *)(outmsg+1);
   
   DWORD   msglen    = sizeof(MsgAddPlayer) + sizeof(MPL_OnePlayer) + msg->NameLen;
   DWORD   pid       = AddPlayer((char *)(msg+1), SenderID);
   Entities::Player *newplyr = Players[pid];
   newplyr->SetPos(0, 0, 0);

   outmsg->Type = MT_ADDPLAYER;
   mpl->NameLen = msg->NameLen;
   mpl->X       = newplyr->Position.X;
   mpl->Y       = newplyr->Position.Y;
   mpl->Z       = newplyr->Position.Z;
   mpl->ObjectIndex = Entities::OT_PLAYER;
   strcpy((char *)(mpl+1), (char const *)(msg+1));

   DebugOut("HandleLogon: SenderID(%d), Name(%s), PlayerID(%d)\n", SenderID, (msg+1), pid);

   SendToAllExcept(outmsg, msglen, true, pid);
   DoLogonAccept(pid);

} /* HandleLogon */


void HandleLogoff(DWORD SenderID)
{
   int pid = FindPlayer(SenderID);
   if (pid >= 0)
   {
      Entities::Player *plyr = Players[pid];
      DebugOut("Player %d (%s) Logging off.\n", pid, plyr->Name.c_str());

      MsgRemPlayer *msg = (MsgRemPlayer *)SendBuffer;
      msg->Type = MT_REMPLAYER;
      msg->ID   = pid;

      SendToAllExcept(msg, sizeof(MsgRemPlayer), true, pid);

      RemPlayer(pid);
   }
} /* HandleLogoff */


void HandlePickup(DWORD SenderID)
{
   MsgPickup *msg = (MsgPickup *)RecvBuffer;
   int pid = FindPlayer(SenderID);
   assert(pid != -1);
   Entities::Player *plyr = Players[pid];
   Entities::Item   *item = FindItem(msg->ID);
   assert(item);
   
   DebugOut("Player %d(%s) tried to pickup item %d(%s) - ",
            pid, plyr->Name.c_str(), item->ID, item->Name.c_str());

   if ((item->Position - plyr->Position).Length_Sqr() > PICKUPDIST)
   {
      DoLoggedMessage(SenderID, YELLOW, "It's too far away!");
      DebugOut("Too far!\n");
   }

   DebugOut("Picked up!\n");
   plyr->Drop(item);

   DoRemoveObject(item);
} /* HandlePickup */


void HandleDrop(DWORD SenderID)
{
   int pid = FindPlayer(SenderID);
   assert(pid != -1);

   MsgDrop *msg = (MsgDrop *)RecvBuffer;
   Entities::Player *plyr = Players[pid];
   Entities::Item   *item = FindItem(msg->ID);
   assert(item);
   assert(plyr->Position.Z == msg->Z); // no dropping things into the next level

   Vectors::Vector<int> pos = Vectors::Vector<int>(msg->X, msg->Y, msg->Z);
   DebugOut("Player %d(%s) tried to throw Item %d(%s) to %d,%d,%d - ",
            pid, plyr->Name.c_str(), item->ID, item->Name.c_str(), msg->X, msg->Y, msg->Z);
   if ((pos - plyr->Position).Length_Sqr() > DROPDIST) // too far
   {
      DoLoggedMessage(SenderID, YELLOW, "You can't throw it that far!");
      DebugOut("Too far!\n");
      return;
   }

   DebugOut("Thrown!\n");

   plyr->Drop(item);
   item->Position = pos;
   DoCreateObject(item);
} /* HandleDrop */


void HandleEquip(DWORD SenderID)
{
   MsgEquip *msg = (MsgEquip *)RecvBuffer;
   
   Entities::Item *item   = FindItem(msg->ID);
   int pid = FindPlayer(SenderID);
   assert(pid != -1);
   Entities::Player *plyr = Players[pid];
   
   assert(item);
   plyr->Equip(item, msg->EP);

   DebugOut("Player %d(%s) equipped %s(ID %d) on %s.\n",
            pid, plyr->Name.c_str(), item->Name.c_str(), item->ID, EquipPlaceStr[msg->EP]);
} /* HandleEquip */


void HandleDeEquip(DWORD SenderID)
{
   MsgDeEquip *msg = (MsgDeEquip *)RecvBuffer;
   
   int pid = FindPlayer(SenderID);
   assert(pid != -1);
   Entities::Player *plyr = Players[pid];

   plyr->DeEquip(msg->EP);

   DebugOut("Player %d(%s) deequipped from %s.\n", pid, plyr->Name.c_str(), EquipPlaceStr[msg->EP]);
} /* HandleDeEquip */


void DoLogonAccept(DWORD PlayerID)
{
   DWORD msglen = 0;
   int   i;

   if (NumItems)
   {
      mbinfile file;
      file.open(SendBuffer, 512, file.openrw);
      msglen = Items[0]->Save(&file); // length of a single object
      assert(msglen);
      msglen = (msglen+sizeof(int)*4) * NumItems;
   }
   
   msglen += sizeof(MsgLogOnAccepted) + sizeof(MLO_PlayerList) +
             sizeof(MPL_OnePlayer) * NumPlayers;
   for(i=0;i<NumPlayers;i++) msglen += Players[i]->Name.size() + 1;
   EnlargeSendBuffer(msglen);

   MsgLogOnAccepted *msg = (MsgLogOnAccepted *)SendBuffer;
   
   msg->Type     = MT_LOGONACCEPTED;
   msg->ID       = PlayerID;
   msg->Seed     = 0;
   msg->Depth    = 0;
   msg->numItems = NumItems;

   mbinfile file;
   file.open(msg, msglen, file.openrw);
   file.seek(sizeof(MsgLogOnAccepted));

   for(i=0;i<NumItems;i++)
   {
      int *dat = (int *)((char *)msg + file.tell());
      file.seekcur(sizeof(int)*4);
      dat[0] = Items[i]->ID;
      dat[1] = Items[i]->Position.X;
      dat[2] = Items[i]->Position.Y;
      dat[3] = Items[i]->Position.Z;
      Items[i]->Save(&file);
   }
   
   MLO_PlayerList   *mlo  = (MLO_PlayerList *)((char *)msg + file.tell());
   Entities::Player *plyr = Players[PlayerID];
   mlo->Str        = plyr->Str;
   mlo->Int        = plyr->Int;
   mlo->Dex        = plyr->Def;
   mlo->Spd        = plyr->Spd;
   mlo->Def        = plyr->Def;
   mlo->Attk       = plyr->Attk;
   mlo->MaxHits    = plyr->MaxHits;
   mlo->numPlayers = NumPlayers;

   MPL_OnePlayer *mpl = (MPL_OnePlayer *)(mlo+1);
   for(i=0;i<NumPlayers;i++)
   {
      plyr = Players[i];
      mpl->ObjectIndex = Entities::OT_PLAYER;
      mpl->X           = plyr->Position.X;
      mpl->Y           = plyr->Position.Y;
      mpl->Z           = plyr->Position.Z;
      mpl->NameLen     = plyr->Name.size() + 1;
      strcpy((char *)(mpl+1), plyr->Name.c_str());
      mpl = (MPL_OnePlayer *)((char *)(mpl+1) + mpl->NameLen);
   }

   DebugOut("DoLogonAccept: PlayerID(%d), NumItems(%d), NumPlayers(%d), msglen(%d)\n", PlayerID, NumItems, NumPlayers, msglen);
   
   SendMessage(Players[PlayerID]->NetID, msg, msglen, true);

   DoCreateInventory(PlayerID);
} /* DoLogonAccept */


void DoCreateInventory(DWORD PlayerID)
{
   // for now just create one item from each class
   MsgUpdateInv *msg = (MsgUpdateInv *)SendBuffer;
   DWORD *ids        = (DWORD *)(msg+1);
   DWORD  msglen     = sizeof(MsgUpdateInv);
   Entities::Player *plyr = Players[PlayerID];
   msg->Type    = MT_UPDATEINV;
   msg->numRems = 0;
   msg->numAdds = Entities::IC_ANY;

   mbinfile file;
   file.open(ids, 512, file.openrw);

   DebugOut("DoCreateInventory: Player %d (%s) Rems(%d), Adds(%d)\n",
            PlayerID, Players[PlayerID]->Name.c_str(), msg->numRems, msg->numAdds);
   for(int i=0;i<Entities::IC_ANY;i++)
   {
      Entities::Item *item = Entities::OnlyItemFactory.RandomItem((Entities::ItemType)i);
      *ids++ = item->ID = NewItemID();
      AddItem(item);
      file.seekcur(sizeof(DWORD));
      int len = item->Save(&file);
      msglen += len+sizeof(DWORD);
      ids = (DWORD *)((char *)ids + len);
      plyr->PickUp(item);
      DebugOut(" Item %d(%s), Object(%d)\n", item->ID, item->Name.c_str(), item->ObjectIndex);
   }

   SendMessage(Players[PlayerID]->NetID, msg, msglen, true);
} /* DoCreateInventory */


void DoLoggedMessage(DWORD NetID, DWORD Color, char const *msg)
{
   DWORD len       = strlen(msg);
   DWORD msglen    = sizeof(MsgMessage) + len;
   
   EnlargeSendBuffer(msglen);
   MsgMessage *outmsg = (MsgMessage *)SendBuffer;
   outmsg->Type   = MT_MESSAGE;
   outmsg->Color  = DKYELLOW;
   outmsg->MsgLen = len;
   strcpy((char *)(outmsg+1), msg);

   SendMessage(NetID, outmsg, msglen, true);
} /* DoLoggedMessage */
   

void DoCreateObject(Entities::Item *item)
{
   MsgCreateObject *msg = (MsgCreateObject *)SendBuffer;
   DWORD msglen = sizeof(MsgCreateObject);
   mbinfile file;

   msg->Type = MT_CREATEOBJECT;
   msg->ID   = item->ID;
   msg->X    = item->Position.X;
   msg->Y    = item->Position.Y;
   msg->Z    = item->Position.Z;

   file.open((msg+1), 512, file.openrw);
   msglen += item->Save(&file);

   DebugOut("DoCreateObject: Item %d(%s), (%d,%d,%d) added to map.\n", msg->ID, item->Name.c_str(), msg->X, msg->Y, msg->Z);
   SendToAll(msg, msglen, true);
} /* DoCreateObject */


void DoRemoveObject(Entities::Item *item)
{
   MsgDeleteObject *msg = (MsgDeleteObject *)RecvBuffer;

   msg->Type = MT_DELETEOBJECT;
   msg->ID   = item->ID;

   DebugOut("DoRemoveObject: Item %d(%s) removed from map.\n", msg->ID, item->Name.c_str());
   SendToAll(msg, sizeof(MsgDeleteObject), true);
} /* DoRemoveObject */


void ResetSession()
{
   // Send buffer guaranteed to be at least 512 bytes
   EnlargeSendBuffer(512);
   ClearPlayers();

   DebugOut("Session Reset\n");
} /* ResetSession */


void HandleMessage(DWORD SenderID)
{
   EnterCriticalSection(&S_CS);
   if(SenderID == DPID_SYSMSG)
   {
      DPMSG_GENERIC *msg = (DPMSG_GENERIC *)RecvBuffer;
      switch(msg->dwType)
      {
         case DPSYS_DESTROYPLAYERORGROUP:
            {
               DPMSG_DESTROYPLAYERORGROUP *smsg = (DPMSG_DESTROYPLAYERORGROUP *)msg;
               if (smsg->dwPlayerType == DPPLAYERTYPE_PLAYER) HandleLogoff(smsg->dpId);
            }
            break;
      }
   }
   else
   {
      GenericMessage *msg = (GenericMessage *)RecvBuffer;
      switch(msg->Type)
      {
         case MT_LOGON:
            HandleLogon(SenderID);
            break;

         case MT_LOGOFF:
            HandleLogoff(SenderID);
            break;

         case MT_PICKUP:
            HandlePickup(SenderID);
            break;

         case MT_DROP:
            HandleDrop(SenderID);
            break;

         case MT_EQUIP:
            HandleEquip(SenderID);
            break;

         case MT_DEEQUIP:
            HandleDeEquip(SenderID);
            break;
      }
   }
   LeaveCriticalSection(&S_CS);
} /* HandleMessage */


void SendToAll(void *msg, DWORD msglen, bool guar)
{
   for(int i=0;i<NumPlayers;i++) SendMessage(Players[i]->NetID, msg, msglen, guar);
} /* SendToAll */


void SendToAllExcept(void *msg, DWORD msglen, bool guar, DWORD skipID)
{
   for(int i=0;i<NumPlayers;i++)
   {
      if (i == skipID) continue;
      SendMessage(Players[i]->NetID, msg, msglen, guar);
   }
} /* SendToAllExcept */


void EnlargeSendBuffer(DWORD newsize)
{
   if (newsize > SendBufLen)
      SendBuffer = (char *)realloc(SendBuffer, SendBufLen = newsize);
} /* EnlargeSendBuffer */

      
DWORD WINAPI PulseThread(LPVOID)
{
   LARGE_INTEGER LI;
   __int64 tFreq, tLast, tCur, tNext, tInc;

   QueryPerformanceFrequency(&LI);

   tFreq = LI.QuadPart;
   tInc  = tFreq / VFPS;

   QueryPerformanceCounter(&LI);
   tLast = LI.QuadPart;
   tNext = tLast + tInc;

   srand(time(NULL));

   while(TRUE)
   {
      QueryPerformanceCounter(&LI);
      tCur = LI.QuadPart;
      while (tCur >= tNext)
      {
         OnPulse();
         tNext += tInc;
      }
      if (WaitForSingleObject(hKill, 0) == WAIT_OBJECT_0)
      {
         ExitThread(0);
         return 0;
      }
      Sleep(0);
   }
   return 0;
} /* PulseThread */
