#pragma warning ( 3 : 4706)
/********************************* Game.cpp ***********************************\
* This module contains all the base functions used for the game code.  This    *
* includes setup, teardown, and the main game loop.                            *
\******************************************************************************/

#include <windows.h>
#include "WinApi.h"
#include "Externs.h"
#include "Defines.h"
#include "Game.h"
#include "Init.h"
#include "Debug.h"

#include "IntrFace\BasicDP.h"
#include "Graphics\PrimDraw.h"
#include "Vectors\VectObject.h"
#include "Binfile\binfmem.h"
#include "Binfile\binfstd.h"
#include "Entities\Entity.h"
#include "Maps\Level.h"
#include "OverSeer.h"

// the game state is updated 20 times per second
#define VFPS 20

#ifdef NEED_GRAPHICS
   CGraphics   Cgraph;
#endif
#ifdef NEED_INPUT
   CDInput     Cdinput;
#endif
#ifdef NEED_MOUSE
   CDIDevice  Cdmouse;
#endif
#ifdef NEED_KEYBOARD
   CDIDevice  Cdkeyboard;
#endif
#ifdef NEED_SOUND
   CDSound     Cdsound;
#endif
#ifdef NEED_MUSIC
   CMidas      Cmidas;
#endif

volatile bool g_bNeedRedraw;

static Status UpdateGameState();
static Status UpdateScreen();
static int    LoadGameData();
static int    UnloadGameData();

#ifdef NEED_GRAPHICS
   CDDSurface *BackBuffer;
#endif

static BasicDP  bdp;
static OverSeer over;
static Graphics::PrimDraw PD(&Cgraph);
static Entities::Player Plyr;
static int Xpos, Ypos;
static Maps::Vertex vs[2];
static int NumVs;
Maps::Level L;
Maps::Level::Path *path = NULL;

/* Called VFPS times per second. Update the game state here. Set 'Ret'
   to G_UPDATED if you make a change which requires a screen redraw.
   Alternately, you can return G_CATCHUP after a lengthy operation
   (although you shouldn't really be doing lengthy operations in this
   function).  This will cause the main loop to skip pending state updates
   and catch up to where it is when the function returns. By default, the
   main loop would try to call UpdateGameState() a bunch of times to make
   up for the time lost during the lengthy operation.
*/
#include "Math\GenMath.h"
Status UpdateGameState()
{
   Status Ret = G_RUNNING;

   // Update game state here
   if (ClickHistory.size())
   {
      MouseClick c = *ClickHistory.begin();
      ClickHistory.pop_front();
      if(!c.Down && c.Btn==0)
      {
         if(NumVs == 2)
         {
            vs[0] = vs[1];
            vs[1] = Maps::Vertex(c.X+Xpos, c.Y+Ypos);
         }
         else vs[NumVs++] = Maps::Vertex(c.X+Xpos, c.Y+Ypos);

         if (NumVs == 2)
         {
            path = L.FindPath(vs[0], vs[1]);
            Ret = G_UPDATED;
         }
      }
   }
   
   if(KeyboardBuf[DIK_UP])    Ypos -= 25;
   if(KeyboardBuf[DIK_DOWN])  Ypos += 25;
   if(KeyboardBuf[DIK_LEFT])  Xpos -= 25;
   if(KeyboardBuf[DIK_RIGHT]) Xpos += 25;

   Ret = G_UPDATED;

char buf[512];
if(NumVs)
{
   wsprintf(buf,"%d %d %d %d %d\n",Math::FindAngle(vs[0].X, vs[0].Y, MouseX, MouseY), vs[0].X, vs[0].Y, MouseX, MouseY);
   OutputDebugString(buf);
}

   if (g_bNeedRedraw)
   {
      g_bNeedRedraw = false;
      Ret = G_UPDATED;
   }
#ifdef NEED_INPUT   
   if (g_bInputFailure)
   {
      g_bInputFailure = false;
#ifdef NEED_MOUSE
      Cdinput.Acquire(&Cdmouse);
#endif
#ifdef NEED_KEYBOARD
      Cdinput.Acquire(&Cdkeyboard);
#endif
   }
#endif
   
   return Ret;
} /* UpdateGameState */


Status UpdateScreen()
{
   // Update the backbuffer here, based on the game state
   Cgraph.ClearSurface(BackBuffer);
   L.Draw(&PD, BackBuffer, Xpos, Ypos);
   if (path) for(int i=0;i<path->NumNodes-1;i++) PD.Line(BackBuffer, path->Pos[i].X-Xpos, path->Pos[i].Y-Ypos, path->Pos[i+1].X-Xpos, path->Pos[i+1].Y-Ypos, MYRGB(0,128,255));
   if (NumVs)
   {
      PD.Circle(BackBuffer, vs[0].X-Xpos, vs[0].Y-Ypos, 2,MYRGB(192,192,192));
      if (NumVs == 2) PD.Circle(BackBuffer, vs[1].X-Xpos, vs[1].Y-Ypos, 2,MYRGB(255,255,255));
   }
   PD.Circle(BackBuffer, MouseX, MouseY, 2, MYRGB(0,0,255));

   // Now do the actual blit
#ifdef NEED_GRAPHICS
#ifdef WINDOWED
   if (Cgraph.Blit(&Cgraph.Primary, BackBuffer, NULL, NULL) != CDD_NOERROR)
   {
      if (Cgraph.GetLastDDError() == DDERR_SURFACELOST)
      {
         if (Cgraph.RestoreAll() != CDD_NOERROR) return U_FATAL;
         if (Cgraph.Blit(&Cgraph.Primary, BackBuffer, NULL, NULL) != CDD_NOERROR) return U_FATAL;
      }
      else return U_FATAL;
   }
#else
   Cgraph.WaitForRetrace();
   if (Cgraph.BlitFast(&Cgraph.Primary, BackBuffer, 0, 0, NULL) != CDD_NOERROR)
   {
      if (Cgraph.GetLastDDError() == DDERR_SURFACELOST)
      {
         if (Cgraph.RestoreAll() != CDD_NOERROR) return U_FATAL;
         if (Cgraph.BlitFast(&Cgraph.Primary, BackBuffer, 0, 0, NULL) != CDD_NOERROR) return U_FATAL;
      }
      else return U_FATAL;
   }
#endif
#endif

   return U_OK;
} /* UpdateScreen */


int LoadGameData()
{
#ifdef NEED_GRAPHICS
   if (Cgraph.MakeSurface(DISPLAYWIDTH, DISPLAYHEIGHT, DISPLAYDEPTH,
                          &BackBuffer, CDD_SYSTEM)
       != CDD_NOERROR)
   {
      MsgBox("Error creating back buffer.");
      return G_QUIT;
   }
   Cgraph.ClearSurface(BackBuffer, NULL, MYRGB(0, 0, 0));

#ifdef WINDOWED
   Cgraph.UpdateWindowed();
#endif
#endif

/*
   sbinfile sfile;
   char buf[512];
   strcpy(buf, g_AppPath);
   strcat(buf, "objects.dat");
   if (sfile.open(buf, sfile.openro | sfile.openex)) return 1;
   if (Entities::OnlyObjectFactory.Load(&sfile)) return 1;
   strcpy(buf, g_AppPath);
   strcat(buf, "items.dat");
   if (sfile.open(buf, sfile.openro | sfile.openex)) return 1;
   if (Entities::OnlyItemFactory.Load(&sfile)) return 1;
   Plyr.SetObject(Entities::OT_PLAYER);

   DPSESSIONDESC2 desc;
   GUID AppGUID = { 0x85409482, 0x0623, 0x4626,
                   { 0x26, 0x94, 0x89, 0x36, 0x54, 0x85, 0x94, 0x83 } };

   ZeroMemory(&desc, sizeof(desc));
   desc.dwSize  = sizeof(desc);
   desc.guidApplication  = AppGUID;
*/
   L.Dig(10, 100, 30, 40);

/*   
   bdp.InitDP();
   bdp.InitConn(bdp.FindGUID(DPSPGUID_TCPIP));
   bdp.EnumSessions(&desc, 1500);
   bdp.JoinSession(0);

   over.Init(&Cgraph, &bdp);
   over.LogOn("Adam", 1500);
*/
   return 0;
}


int UnloadGameData()
{
/*
   over.LogOff(1000);
   over.Deinit();
   bdp.DeinitDP();
*/
   return 0;
}

   
int MainGameLoop()
{
   MSG msg;
   LARGE_INTEGER LI;
   __int64 tFreq, tLast, tCur, tNext, tInc;
   Status state;
   bool ScreenUpdated = false;

   if (!QueryPerformanceFrequency(&LI))
   {
      MsgBox("No High-Frequency timer supported.");
      return 1;
   }

   if (LoadGameData()) return 1;

   tFreq = LI.QuadPart;
   tInc  = tFreq / VFPS;

   QueryPerformanceCounter(&LI);
   tLast = LI.QuadPart;
   tNext = tLast + tInc;

   g_bNeedRedraw = TRUE;

   while(TRUE)
   {
      if (PeekMessage(&msg, g_hWnd, 0, 0, PM_REMOVE))
      {
         TranslateMessage(&msg);
         DispatchMessage(&msg);
      }

      if (g_bQuitApp) return UnloadGameData();

      if (g_bFocus)
      {
         QueryPerformanceCounter(&LI);
         tCur = LI.QuadPart;
         while (tCur >= tNext)
         {
            state = UpdateGameState();
            switch(state)
            {
               case G_UPDATED:
                  ScreenUpdated = true; // need to redraw the screen
                  break;

               case G_CATCHUP:
                  tNext = tCur;
                  break;

               case G_QUIT:
                  SendMessage(g_hWnd, WM_CLOSE, 0, 0);
                  break;
            }
            tNext += tInc;
         }
         if (ScreenUpdated)
         {
            state = UpdateScreen();
            ScreenUpdated = false;
            switch(state)
            {
               case U_FATAL:
                  SendMessage(g_hWnd, WM_CLOSE, 0, 0);
                  break;
            }
         }
      }
      else
      {
         Sleep(150);
         g_bNeedRedraw = true;
         QueryPerformanceCounter(&LI);
         tLast = LI.QuadPart;
         tNext = tLast + tInc;
      }
   }
} /* MainGameLoop */
