#pragma warning ( 3 : 4706)
#include <windows.h>
#include "Externs.h"
#include "Defines.h"
#include "Init.h"
#include "Debug.h"

#ifdef NEED_INPUT
using std::list;
#endif

static int   InitGameSound();
static void  DeinitGameSound();

static int   InitGameInput();
static void  DeinitGameInput();

static int   InitGameGraphics();
static void  DeinitGameGraphics();

#ifdef NEED_KEYBOARD
   static void  KBCallback(CDIDevice *dev);
#endif
#ifdef NEED_MOUSE
   static void  MouseCallback(CDIDevice *dev);
#endif


/***** GLOBALS *****/
volatile bool g_bInitDone;

#ifdef NEED_INPUT
   volatile bool g_bInputFailure;
#endif
#ifdef NEED_KEYBOARD
   byte KeyboardBuf[256];
   std::list<KeyPress>   KeyHistory;
#endif
#ifdef NEED_MOUSE
   int MouseX, MouseY, MouseBtn[4];
   std::list<MouseClick> ClickHistory;
#endif
/***** GLOBALS *****/


int InitGame()
{
   int err = 0;
   err = err;

#ifdef NEED_INPUT
   err = InitGameInput();
   if (err)
   {
      MsgBox("Error initializing input.");
      return err;
   }
   g_bInputFailure = false;
#endif

#ifdef NEED_SOUND
   err = InitGameSound();
   if (err)
   {
      MsgBox("Error initializing sound.");
      return err;
   }
#endif

#ifdef NEED_MUSIC
   if (!MIDASsetOption(MIDAS_OPTION_MIXRATE, MIDASMIXRATE)) return 7;
   if (!MIDASsetOption(MIDAS_OPTION_OUTPUTMODE, MIDASOUTMODE)) return 8;
   if (!MIDASsetOption(MIDAS_OPTION_MIXING_MODE, MIDASQUALITY)) return 9;

   if (Cmidas .Init(g_hWnd, &Cdsound, 0))
   {
      DebugFOut2("InitGame:Cmidas.Init - %s", MIDASgetErrorMessage(MIDASgetLastError()));
      MsgBox("Error initializing music.");
      return 10;
   }
#endif

#ifdef NEED_GRAPHICS
   err = InitGameGraphics();
   if (err)
   {
      MsgBox("Error initializing graphics.");
      return err;
   }
#endif

   g_bInitDone = true;
   return 0;
} /* InitGame */


int DeinitGame()
{
   g_bInitDone = false;
   DeinitGameSound();
   DeinitGameInput();
   DeinitGameGraphics();

   return 0;
} /* DeinitGame */


int InitGameSound()
{
#ifdef NEED_SOUND
   if (Cdsound.Init(g_hWnd, NULL, SOUNDMIXRATE, SOUNDBITS, SOUNDCHANS))
   {
      MsgBox("InitGameSound:Cdsound.Init - %s", Cdsound.ErrorToString());
      return 2;
   }
#endif   
   return 0;
} /* InitGameSound */


void DeinitGameSound()
{
   #ifdef NEED_MUSIC
      Cmidas.Deinit();
   #endif
   #ifdef NEED_SOUND
      Cdsound.Deinit();
   #endif
} /* DeinitGameSound */


int InitGameInput()
{
#ifdef NEED_INPUT
   int i;
   GUID guid;

   Cdinput.SetBufSize(INPUTBUFSIZE);

   if (Cdinput.Init(g_hWnd, g_hInst))
   {
      MsgBox("InitGame:Cdinput.Init - %s", Cdinput.ErrorToString());
      return 3;
   }
#endif
   
#ifdef NEED_MOUSE
   Cdmouse.Buffered = TRUE;
   Cdmouse.Callback = MouseCallback;
   if (Cdinput.InitDevice(&Cdmouse, CDI_MOUSE,
                          DISCL_EXCLUSIVE | DISCL_FOREGROUND, guid))
   {
      MsgBox("InitGame:Cdinput.InitDevice(MOUSE) - %s", Cdinput.ErrorToString());
      return 4;
   }
   MouseX = MouseY = 0;
   for(i=0;i<3;i++) MouseBtn[i] = 0;
   Cdinput.Acquire(&Cdmouse);
#endif

#ifdef NEED_KEYBOARD
   Cdkeyboard.Buffered = TRUE;
   Cdkeyboard.Callback = KBCallback;
   if (Cdinput.InitDevice(&Cdkeyboard, CDI_KEYBOARD,
                          DISCL_NONEXCLUSIVE | DISCL_FOREGROUND, guid))
   {
      MsgBox("InitGame:Cdinput.InitDevice(KB) - %s", Cdinput.ErrorToString());
      return 5;
   }
   for(i=0;i<256;i++) KeyboardBuf[i] = 0;
   Cdinput.Acquire(&Cdkeyboard);
#endif

#ifdef NEED_INPUT
   if (Cdinput.InitThread())
   {
      MsgBox("InitGame:Cdinput.InitThread");
      return 6;
   }
#endif

   return 0;
} /* InitGameInput */


void DeinitGameInput()
{
#ifdef NEED_INPUT
   Cdinput.KillThread();
#ifdef NEED_KEYBOARD
   Cdinput.KillDevice(&Cdkeyboard);
#endif
#ifdef NEED_MOUSE
   Cdinput.KillDevice(&Cdmouse);
#endif
   Cdinput.Deinit();
#endif
} /* DeinitGameInput */


int InitGameGraphics()
{
#ifdef NEED_GRAPHICS
   if (Cgraph.Init(g_hWnd))
   {
      MsgBox("InitGame:Cgraph.Init - %s", Cgraph.ErrorToString());
      return 1;
   }

#ifdef WINDOWED   
   Cgraph.UpdateWindowed();
   if (Cgraph.SetMode(DISPLAYWIDTH, DISPLAYHEIGHT, DISPLAYDEPTH, CDD_WINDOWED, CDD_SINGLE))
#else
   if (Cgraph.SetMode(DISPLAYWIDTH, DISPLAYHEIGHT, DISPLAYDEPTH, CDD_FULLSCREEN, CDD_SINGLE))
#endif
   {
      MsgBox("InitGame:Cgraph.SetMode - %s", Cgraph.ErrorToString());
      return 11;
   }
#endif

   return 0;
} /* InitGameGraphics */


void DeinitGameGraphics()
{
#ifdef NEED_GRAPHICS
   Cgraph.Deinit();
#endif
} /* DeinitGameGraphics */


#ifdef NEED_KEYBOARD
void KBCallback(CDIDevice *dev)
{
   DIDEVICEOBJECTDATA dod[INPUTBUFSIZE];
   DWORD i, num = INPUTBUFSIZE;
   HRESULT hRet;
   KeyPress kp;

   if (dev == NULL)
   {
      g_bInputFailure = true;
      return;
   }

   while(TRUE)
   {
      hRet = dev->pDIDev->GetDeviceData(sizeof(DIDEVICEOBJECTDATA), dod, &num, 0);
      if (hRet == DIERR_NOTACQUIRED || hRet == DIERR_INPUTLOST)
      {
         hRet = dev->pDIDev->Acquire();
         if (FAILED(hRet)) break;
      }
      else break;
   }
   if (FAILED(hRet))
   {
      g_bInputFailure = true;
      return;
   }
   else g_bInputFailure = false;
   
   for(i=0;i<num;i++)
   {
      KeyboardBuf[dod[i].dwOfs] = dod[i].dwData & 0x80;
      kp.Key  = dod[i].dwOfs;
      kp.Down = dod[i].dwData & 0x80;
      KeyHistory.push_back(kp);
      if (KeyHistory.size() > KEYHISTORY) KeyHistory.pop_front();
   }
} /* KBCallback */
#endif


#ifdef NEED_MOUSE
void MouseCallback(CDIDevice *dev)
{
   DIDEVICEOBJECTDATA dod[INPUTBUFSIZE];
   DWORD i, num = INPUTBUFSIZE;
   HRESULT hRet;
   MouseClick mc;
   bool bClick = false;

   if (dev == NULL)
   {
      g_bInputFailure = true;
      return;
   }

   while(TRUE)
   {
      hRet = dev->pDIDev->GetDeviceData(sizeof(DIDEVICEOBJECTDATA), dod, &num, 0);
      if (hRet == DIERR_NOTACQUIRED || hRet == DIERR_INPUTLOST)
      {
         hRet = dev->pDIDev->Acquire();
         if (FAILED(hRet)) break;
      }
      else break;
   }
   if (FAILED(hRet))
   {
      g_bInputFailure = true;
      return;
   }
   else g_bInputFailure = false;

   for(i=0;i<num;i++)
   {
      switch(dod[i].dwOfs)
      {
         case DIMOFS_X:
            MouseX += dod[i].dwData;
            break;

         case DIMOFS_Y:
            MouseY += dod[i].dwData;
            break;

         case DIMOFS_BUTTON0:
            mc.Down = MouseBtn[0] = dod[i].dwData & 0x80;
            mc.Btn  = 0;
            bClick  = true;
            break;

         case DIMOFS_BUTTON1:
            mc.Down = MouseBtn[1] = dod[i].dwData & 0x80;
            mc.Btn  = 1;
            bClick  = true;
            break;

         case DIMOFS_BUTTON2:
            mc.Down = MouseBtn[2] = dod[i].dwData & 0x80;
            mc.Btn  = 2;
            bClick  = true;
            break;

         case DIMOFS_BUTTON3:
            mc.Down = MouseBtn[3] = dod[i].dwData & 0x80;
            mc.Btn  = 3;
            bClick  = true;
            break;
      }
      if (bClick)
      {
         mc.X = MouseX;
         mc.Y = MouseY;
         if (mc.X < 0) mc.X = 0;
         if (mc.X >= DISPLAYWIDTH) mc.X = DISPLAYWIDTH - 1;
         if (mc.Y < 0) mc.Y = 0;
         if (mc.Y >= DISPLAYHEIGHT) mc.Y = DISPLAYHEIGHT - 1;
         ClickHistory.push_back(mc);
         if (ClickHistory.size() > CLICKHISTORY) ClickHistory.pop_front();
      }
   }

   if (MouseX < 0) MouseX = 0;
   if (MouseX >= DISPLAYWIDTH) MouseX = DISPLAYWIDTH - 1;
   if (MouseY < 0) MouseY = 0;
   if (MouseY >= DISPLAYHEIGHT) MouseY = DISPLAYHEIGHT - 1;
} /* MouseCallback */
#endif

