#include "Primitives.h"

namespace Graphics
{
   // define the static member SineTab
   Math::Sines &Primitive::SineTab = Math::OnlySineTable;
   
   // BEGIN POINT IMPLEMENTATION

   int Point::Draw(PrimDraw *pd, CDDSurface *dest, int xc, int yc, CDDColor c)
   {
      return pd->Point(dest, xc+New.X, yc+New.Y, c);
   }

   void Point::Rotate(int angle)
   {
      New = Orig;
      if (angle == 0) return;
      RotatePoint(&New, angle);
   };

   void Point::Scale(double scale)
   {
      New = Orig;
      if (scale == 1.0) return;
      ScalePoint(&New, scale);
   };

   int Point::Load(binfile *file)
   {
      ReadPoint(file, &Orig);
      New = Orig;
      return 0;
   }

   int Point::Save(binfile *file)
   {
      WritePoint(file, &Orig);
      return 0;
   }

   Primitive * Point::Make_Copy()
   {
      Point *p = (Point *)Make_New();
      *p = *this;
      return p;
   }

   PrimType Point::MyType()
   {
      return PT_POINT;
   }
   // ------------------------- END POINT IMPLEMENTATION ----------------------


   // BEGIN LINE IMPLEMENTATION

   int Line::Draw(PrimDraw *pd, CDDSurface *dest, int xc, int yc, CDDColor c)
   {
      return pd->Line(dest, xc+P1.X, yc+P1.Y, xc+P2.X, yc+P2.Y, c);
   }

   void Line::Rotate(int angle)
   {
      P1 = oP1, P2 = oP2;
      if (angle == 0) return;
      RotatePoint(&P1, angle);
      RotatePoint(&P2, angle);
   };

   void Line::Scale(double scale)
   {
      P1 = oP1, P2 = oP2;
      if (scale == 1.0) return;
      ScalePoint(&P1, scale);
      ScalePoint(&P2, scale);
   };

   int Line::Load(binfile *file)
   {
      ReadPoint(file, &oP1);
      ReadPoint(file, &oP2);
      P1 = oP1, P2 = oP2;
      return 0;
   }

   int Line::Save(binfile *file)
   {
      WritePoint(file, &oP1);
      WritePoint(file, &oP2);
      return 0;
   }

   Primitive * Line::Make_Copy()
   {
      Line *p = (Line *)Make_New();
      *p = *this;
      return p;
   }

   PrimType Line::MyType()
   {
      return PT_LINE;
   }
   // ------------------------- END LINE IMPLEMENTATION -----------------------


   // BEGIN CIRCLE IMPLEMENTATION

   int Circle::Draw(PrimDraw *pd, CDDSurface *dest, int xc, int yc, CDDColor c)
   {
      return pd->Circle(dest, xc+New.X, yc+New.Y, Rad, c);
   }

   void Circle::Rotate(int angle)
   {
      New = Orig;
      if (angle == 0) return;
      RotatePoint(&New, angle);
   };

   void Circle::Scale(double scale)
   {
      New = Orig;
      Rad = oRad;
      if (scale == 1.0) return;
      ScalePoint(&New, scale);
      Rad = oRad * scale + 0.5;
   };

   int Circle::Load(binfile *file)
   {
      ReadPoint(file, &Orig);
      file->read(&oRad, sizeof(oRad));
      New = Orig, Rad = oRad;
      return 0;
   }

   int Circle::Save(binfile *file)
   {
      WritePoint(file, &Orig);
      file->write(&oRad, sizeof(oRad));
      return 0;
   }

   Primitive * Circle::Make_Copy()
   {
      Circle *p = (Circle *)Make_New();
      *p = *this;
      return p;
   }

   PrimType Circle::MyType()
   {
      return PT_CIRCLE;
   }
   // ------------------------ END CIRCLE IMPLEMENTATION ----------------------


   // BEGIN LINELIST IMPLEMENTATION
   int LineList::Draw(PrimDraw *pd, CDDSurface *dest, int xc, int yc, CDDColor c)
   {
      int i, num = Lines.size() - 1;
      
      for(i=0;i<num;i++)
         pd->Line(dest, Points[Lines[i]].X  +xc, Points[Lines[i]].Y  +yc,
                        Points[Lines[i+1]].X+xc, Points[Lines[i+1]].Y+yc, c);

      return 0;
   }

   void LineList::Rotate(int angle)
   {
      int i, num = Points.size();
      
      Points = oPoints;
      if (angle == 0) return;
      for(i=0;i<num;i++) RotatePoint(&Points[i], angle);
   }

   void LineList::Scale(double scale)
   {
      int i, num = Points.size();
      
      Points = oPoints;
      if (scale == 1.0) return;
      for(i=0;i<num;i++) ScalePoint(&Points[i], scale);
   }

   int LineList::Load(binfile *file)
   {
      int i,num;

      file->read(&num, sizeof(num)); // number of points
      
      Points.resize(num);
      oPoints.resize(num);

      for(i=0;i<num;i++) ReadPoint(file, &oPoints[i]);
      Points = oPoints;

      file->read(&num, sizeof(num)); // number of indices
      Lines.resize(num);

      for(i=0;i<num;i++) file->read(&Lines[i], sizeof(int));
      return 0;
   }

   int LineList::Save(binfile *file)
   {
      int i, num = oPoints.size();

      file->write(&num, sizeof(num));
      for(i=0;i<num;i++) WritePoint(file, &oPoints[i]);

      num = Lines.size();
      file->write(&num, sizeof(num));
      for(i=0;i<num;i++) file->write(&Lines[i], sizeof(int));
      return 0;
   }

   Primitive * LineList::Make_Copy()
   {
      LineList *p = (LineList *)Make_New();
      int i, num = oPoints.size();
      
      p->oPoints.resize(num);
      p->Points.resize(num);
      for(i=0;i<num;i++) p->Points[i] = p->oPoints[i] = oPoints[i];

      num = Lines.size();
      p->Lines.resize(num);
      for(i=0;i<num;i++) p->Lines[i] = Lines[i];

      return p;
   }

   PrimType LineList::MyType()
   {
      return PT_LINELIST;
   }
   // ----------------------- END LINELIST IMPLEMENTATION ---------------------

   
   // BEGIN PRIMFACTORY IMPLEMENTATION

   PrimFactory::PrimFactory()
   {
      Prims[PT_POINT]    = new Point;
      Prims[PT_LINE]     = new Line;
      Prims[PT_CIRCLE]   = new Circle;
      Prims[PT_LINELIST] = new LineList;
   }

   PrimFactory::~PrimFactory()
   {
      delete Prims[PT_POINT];
      delete Prims[PT_LINE];
      delete Prims[PT_CIRCLE];
      delete Prims[PT_LINELIST];
   }

   Primitive * PrimFactory::New(PrimType p)
   {
      if (p >= PT_NUMPRIMS) return NULL;
      return Prims[p]->Make_New();
   }
   // --------------------- END PRIMFACTORY IMPLEMENTATION --------------------

}  // Close namespace
