#ifndef _PRIMITIVES_H
#define _PRIMITIVES_H

#include <vector>
#include "IntrFace\CDDraw.h"
#include "Binfile\Binfile.h"
#include "PrimDraw.h"
#include "Math\TrigTables.h"

namespace Graphics
{
   enum PrimType { PT_POINT, PT_LINE, PT_CIRCLE, PT_LINELIST, PT_NUMPRIMS };

   class Primitive
   {
      public:
         virtual int  Draw(PrimDraw *, CDDSurface *dest, int xc, int yc, CDDColor c) = 0;
         virtual void Rotate(int angle)  = 0;
         virtual void Scale(double scale)   = 0;
         virtual int Load(binfile *file) = 0;
         virtual int Save(binfile *file) = 0;
         virtual Primitive * Make_New()  = 0;
         virtual Primitive * Make_Copy() = 0;
         virtual PrimType    MyType()    = 0;

      protected:
         struct Vertex
         {
            int X, Y;
         };
         inline void RotatePoint(Vertex *pt, int Angle)
         {
            int nx=pt->X, ny=pt->Y;
            pt->X = nx * SineTab.Cos(Angle) - ny * SineTab.Sin(Angle);
            pt->Y = nx * SineTab.Sin(Angle) + ny * SineTab.Cos(Angle);
         };
         inline void ScalePoint(Vertex *pt, double Scale)
         {
            pt->X = pt->X * Scale + 0.5;
            pt->Y = pt->Y * Scale + 0.5;
         };
         void WritePoint(binfile *file, Vertex *pt)
         {
            file->write(&pt->X, sizeof(int));
            file->write(&pt->Y, sizeof(int));
         }
         void ReadPoint(binfile *file, Vertex *pt)
         {
            file->read(&pt->X, sizeof(int));
            file->read(&pt->Y, sizeof(int));
         }

      private:
         static Math::Sines &SineTab;   // The one and only sine table
   };

   class Point : public Primitive
   {
      public:
         int  Draw(PrimDraw *, CDDSurface *dest, int xc, int yc, CDDColor c);
         void Rotate(int angle);
         void Scale (double scale);
         int  Load(binfile *file);
         int  Save(binfile *file);
         Primitive * Make_New()
         {
            return new Point;
         };
         Primitive * Make_Copy();
         PrimType    MyType();

      private:
         Vertex New, Orig;
   };

   class Line : public Primitive
   {
      public:
         int  Draw(PrimDraw *, CDDSurface *dest, int xc, int yc, CDDColor c);
         void Rotate(int angle);
         void Scale (double scale);
         int  Load(binfile *file);
         int  Save(binfile *file);
         Primitive * Make_New()
         {
            return new Line;
         };
         Primitive * Make_Copy();
         PrimType    MyType();

      private:
         Vertex P1, P2;
         Vertex oP1, oP2;
   };

   class Circle : public Primitive
   {
      public:
         int  Draw(PrimDraw *, CDDSurface *dest, int xc, int yc, CDDColor c);
         void Rotate(int angle);
         void Scale (double scale);
         int  Load(binfile *file);
         int  Save(binfile *file);
         Primitive * Make_New()
         {
            return new Circle;
         };
         Primitive * Make_Copy();
         PrimType    MyType();

      private:
         Vertex New, Orig;
         int    Rad, oRad;
   };

   class LineList : public Primitive
   {
      public:
         LineList() { };
         int  Draw(PrimDraw *, CDDSurface *dest, int xc, int yc, CDDColor c);
         void Rotate(int angle);
         void Scale (double scale);
         int  Load(binfile *file);
         int  Save(binfile *file);
         Primitive * Make_New()
         {
            return new LineList;
         }
         Primitive * Make_Copy();
         PrimType    MyType();

      private:
         std::vector<Vertex> Points;
         std::vector<Vertex> oPoints;
         std::vector<int>    Lines;
   };


   class PrimFactory
   {
      public:
         PrimFactory();
         ~PrimFactory();
         Primitive * New(PrimType p);

      private:
         Primitive *Prims[PT_NUMPRIMS];
   };
}

#endif
