#pragma warning(disable : 4786) // identifer >255 chars
#include <math.h>
#include <stdlib.h>
#include <assert.h>
#include <vector>
#include "Level.h"
#include "Frame\Debug.h"
#include "Frame\Defines.h"
#include "Graphics\PrimDraw.h"


namespace Maps
{
   // fail map-making if we try this many times and fail
   #define TIMEOUT 100

   static Vertex FindNewPoint(Sector *s, Line *li);

   inline int Sgn(int n)
   { 
      if(n<0) return -1;
      return n && 1;
   } /* Sgn */


   inline int Min(int a, int b)
   {
      return a<b?a:b;
   } /* Min */


   inline int Max(int a, int b)
   {
      return a>b?a:b;
   } /* Max */


   void Vertex::Rotate(Vertex center)
   { 
      int nx = (Y - center.Y) + center.X;
      Y = -X + center.X + center.Y;
      X = nx;
   } /* Rotate */


   int Vertex::Distance(Vertex v)
   {
      register int d1 = X-v.X, d2 = Y-v.Y;
      return sqrt(d1*d1 + d2*d2);
   } /* Distance */


   int Vertex::Distance_squared(Vertex v)
   {
      register int d1 = X-v.X, d2 = Y-v.Y;
      return d1*d1 + d2*d2;
   } /* Distance_squared */


   int Line::Intersects(Line l)
   { 
      long dx, dy, n, r1, r2;
      Vertex v1 = *V1, v2 = *V2, lv1 = *l.V1, lv2 = *l.V2;

      dx = lv1.X - lv2.X;
      dy = lv2.Y - lv1.Y;
      n  = (lv2.X * lv1.Y) - (lv1.X * lv2.Y);
      r1 = (dy * v1.X) + (dx * v1.Y) + n;
      r2 = (dy * v2.X) + (dx * v2.Y) + n;
      if(Sgn(r1) == Sgn(r2) && r1 && r2) return 0;

      dx = v1.X - v2.X;
      dy = v2.Y - v1.Y;
      n  = (v2.X * v1.Y) - (v1.X * v2.Y);
      r1 = (dy * lv1.X)  + (dx * lv1.Y) + n;
      r2 = (dy * lv2.X)  + (dx * lv2.Y) + n;
      if(Sgn(r1) == Sgn(r2) && r1 && r2) return 0;

      return 1;
   } /* Intersects */


   int Line::Intersects2(Line l)
   {
      if(l.V1 == V1 || l.V2 == V1 || l.V1 == V2 || l.V2 == V2) return 0;
      return Intersects(l);
   } /* Intersects2 */


   void Line::Draw(Graphics::PrimDraw *pd, CDDSurface *s, int xc, int yc)
   {
      pd->Line(s, V1->X-xc, V1->Y-yc, V2->X-xc, V2->Y-yc,
                  Blocking ? MYRGB(255, 255, 255) : MYRGB(128, 0, 0));
   } /* Draw */


   int Line::Length()
   {
      return V1->Distance(*V2);
   } /* Length */


   int Line::Length_squared()
   {
      return V1->Distance_squared(*V2);
   } /* Length_squared */


   void Sector::AddNeighbor(Sector *s)
   {
      int i, j;

      for(i=0; i<3; i++) if(Neighbors[i] == s) return;

      assert(NumNeighbors < 3);

      for(i=0; i<3; i++)
      {
         for(j=0; j<3; j++)
         {
            if(Lines[i] == s->Lines[j])
            {
               Lines[i]->Blocking = false;
               Neighbors[i] = s;
               NumNeighbors++;
               return;
            }
         }
      }
      assert(0);
   } /* AddNeighbor */


   Vertex Sector::FindCentroid()
   { 
      Vertex c;
 
      for(int i=0; i<3; i++)
      {
         c.X += Lines[i]->V1->X + Lines[i]->V2->X;
         c.Y += Lines[i]->V1->Y + Lines[i]->V2->Y;
      }

      c.X /= 6, c.Y /= 6;

      return c;
   } /* FindCentroid */


   Line * Sector::RandomLine(bool blocking)
   {
      int n;

      while(Lines[n=rand()%3]->Blocking != blocking);
      return Lines[n];
   } /* RandomLine */


   int Sector::Overlaps(Sector s)
   { 
      int i;

      for(i=0; i<3; i++)
      {
         for(int j=0; j<3; j++)
         {
            if(Lines[i] != Lines[j] && Lines[i]->Intersects2(*(s.Lines[j]))) return 1;
         }
      }

      return 0;
   } /* Overlaps */


   RECT Sector::BoundingBox()
   {
      RECT bounds = { 0x7FFFFFFF, 0x7FFFFFFF, -0x7FFFFFFF, -0x7FFFFFFF };
      for(int i=0;i<3;i++)
      {
         int xb = Lines[i]->V1->X, yb = Lines[i]->V1->Y;
         if (xb < bounds.left)   bounds.left   = xb;
         if (xb > bounds.right)  bounds.right  = xb;
         if (yb < bounds.top)    bounds.top    = yb;
         if (yb > bounds.bottom) bounds.bottom = yb;
         xb = Lines[i]->V2->X, yb = Lines[i]->V2->Y;
         if (xb < bounds.left)   bounds.left   = xb;
         if (xb > bounds.right)  bounds.right  = xb;
         if (yb < bounds.top)    bounds.top    = yb;
         if (yb > bounds.bottom) bounds.bottom = yb;
      }
      return bounds;
   } /* BoundingBox */


   Level::~Level()
   {
      int i;
      for(i=0;i<NumSectors;i++)  delete Sectors[i];
      for(i=0;i<NumLines;i++)    delete Lines[i];
      for(i=0;i<NumVertices;i++) delete Vertices[i];
      NumSectors = NumLines = NumVertices = 0;
   } /* Level Destructor */


   Vertex * Level::AddVertex(int x, int y)
   {
      for(int i=0; i<NumVertices; i++)
      {
         if(x == Vertices[i]->X && y == Vertices[i]->Y) return Vertices[i];
      }

      if(Vertices.size() < NumVertices+1) Vertices.resize(NumVertices+10);
      return Vertices[NumVertices++] = new Vertex(x, y);
   } /* AddVertex */


   Line * Level::AddLine(Vertex *v1, Vertex *v2)
   {
      for(int i=0; i<NumLines; i++)
      {
         if((v1==Lines[i]->V1 && v2==Lines[i]->V2) ||
            (v1==Lines[i]->V2 && v2==Lines[i]->V1))
            return Lines[i];
      }
      if(Lines.size() < NumLines+1) Lines.resize(NumLines+10);
      return Lines[NumLines++] = new Line(v1, v2);
   } /* AddLine */


   int Level::RemoveVertex(Vertex *v)
   { 
      int i, pos;

      for(pos=0; pos<NumVertices; pos++) if(v == Vertices[pos]) break;
      if (pos == NumVertices) return 1;
   
      for(i=0; i<NumLines; i++)
      {
         if(Lines[i]->V1 == v || Lines[i]->V2 == v) return 1;
      }

      Vertices[pos] = Vertices[--NumVertices];
      delete v;

      return 0;
   } /* RemoveVertex */


   int Level::RemoveLine(Line *l)
   {
      int i, pos;

      for(pos=0; pos<NumLines; pos++) if(l==Lines[pos]) break;
      if (pos == NumLines) return 1;

      for(i=0; i<NumSectors; i++)
      {
         for(int j=0; j<3; j++) if(Sectors[i]->Lines[j]==l) return 1;
      }

      RemoveVertex(l->V1);
      RemoveVertex(l->V2);

      Lines[pos] = Lines[--NumLines];
      delete l;

      return 0;
   } /* RemoveLine */


   Sector * Level::AddSector(Line *l1, Line *l2, Line *l3)
   {
      Sector *neighborList[3];
      Sector  test(l1, l2, l3);
      int i, n=0;

      for(i=0; i<NumSectors; i++)
      {
         int count=0;
         for(int j=0; j<3; j++)
         {
            count += (Sectors[i]->Lines[j] == l1) +
                     (Sectors[i]->Lines[j] == l2) +
                     (Sectors[i]->Lines[j] == l3);
         }
         if(count==3) return Sectors[i];
         else if(count)
         {
            assert(n < 3);
            neighborList[n++] = Sectors[i];
         }
      }
      assert(!NumSectors || n);

      if(Sectors.size() < NumSectors+1) Sectors.resize(NumSectors+10);

      Sector *s = Sectors[NumSectors++] = new Sector(l1, l2, l3);
      s->Centroid = s->FindCentroid();

      for(i=0; i<n; i++)
      {  neighborList[i]->AddNeighbor(s);
         for(int j=0; j<3; j++)
         {
            for(int k=0; k<3; k++)
            {
               if(s->Lines[j]==neighborList[i]->Lines[k])
               {
                 s->Neighbors[j]=neighborList[i];
                 goto Found;
               }
            }
         }
         Found:;
      }

      s->NumNeighbors = n;
      return s;
   } /* AddSector */


   Sector *Level::MakeSector(int x1, int y1, int x2, int y2, int x3, int y3)
   { 
      Vertex *v1 = AddVertex(x1, y1), *v2 = AddVertex(x2, y2);
      Vertex *v3 = AddVertex(x3, y3);

      Line l1(v1, v2), l2(v2, v3), l3(v3, v1);

      int minx = Min(x1, Min(x2, x3)), miny = Min(y1, Min(y2, y3));
      int maxx = Max(x1, Max(x2, x3)), maxy = Max(y1, Max(y2, y3));

      for(int i=0; i<NumLines; i++)
      {
         if((Lines[i]->V1->X<minx && Lines[i]->V2->X<minx) ||
            (Lines[i]->V1->X>maxx && Lines[i]->V2->X>maxx) ||
            (Lines[i]->V1->Y<miny && Lines[i]->V2->Y<miny) ||
            (Lines[i]->V1->Y>maxy && Lines[i]->V2->Y>maxy))
               continue;

         if(l1.Intersects2(*Lines[i]) || l2.Intersects2(*Lines[i]) ||
            l3.Intersects2(*Lines[i]))
         { 
            RemoveVertex(v1);
            RemoveVertex(v2);
            RemoveVertex(v3);
            return 0;
         }
      }

      return AddSector(AddLine(v1, v2), AddLine(v2, v3), AddLine(v3, v1));
   } /* MakeSector */


   Sector * Level::RandomSector(int minNeighbors, int maxNeighbors)
   {
      int n;

      do n=rand()%NumSectors;
      while(Sectors[n]->NumNeighbors > maxNeighbors ||
            Sectors[n]->NumNeighbors < minNeighbors);

      return Sectors[n];
   } /* RandomSector */


   void Level::SmoothEdges()
   {
      for(int i=0; i<NumLines-1; i++)
      {
         for(int j=i+1; j<NumLines; j++)
         {
            int which=-1;
            Vertex *v1, *v2;
            if(!Lines[i]->Blocking || !Lines[j]->Blocking) continue;
            if( Lines[i]->V1==Lines[j]->V1) v1=Lines[i]->V2, v2=Lines[j]->V2, which=0;
            if( Lines[i]->V1==Lines[j]->V2) v1=Lines[i]->V2, v2=Lines[j]->V1, which=1;
            if( Lines[i]->V2==Lines[j]->V1) v1=Lines[i]->V1, v2=Lines[j]->V2, which=0;
            if( Lines[i]->V2==Lines[j]->V2) v1=Lines[i]->V1, v2=Lines[j]->V1, which=1;
            if(which >= 0 && 
               v1->Distance_squared(*v2) <
               Min(Lines[i]->Length_squared(), Lines[j]->Length_squared()))
            {
               MakeSector(Lines[i]->V1->X, Lines[i]->V1->Y,
                          Lines[i]->V2->X, Lines[i]->V2->Y,
                          (which ? Lines[j]->V1 : Lines[j]->V2)->X,
                          (which ? Lines[j]->V1 : Lines[j]->V2)->Y);
            }
         }
      }
   } /* SmoothEdges */

   
   void Level::CheckMinMax(Vertex *v)
   {
      if(v->X < MinX) MinX = v->X;
      if(v->X > MaxX) MaxX = v->X;
      if(v->Y < MinY) MinY = v->Y;
      if(v->Y > MaxY) MaxY = v->Y;
   } /* CheckMinMax */


   void Level::Draw(Graphics::PrimDraw *pd, CDDSurface *s, int xc, int yc)
   {
      for(int i=0; i<NumLines; i++) Lines[i]->Draw(pd, s, xc, yc);
   } /* Draw */

   
   void Level::Dig(int outer, int inner, int roominess, int sectsize)
   { 
      int room = roominess*inner/100;

      MakeSector(-(rand()%sectsize), -(rand()%sectsize),
                   rand()%sectsize,  -(rand()%sectsize),
                   rand()%sectsize,    rand()%sectsize);

      for(int i=0; i<outer; i++)
      {
         for(int j=0; j<inner; j++)
         {
            Vertex  v;
            Sector *s;
            Line   *li;
            int     Count=0;
            do
            {
               s  = RandomSector(0, (j<room) ? 2 : 1);
               li = s->RandomLine(1);
               if (Count++ == TIMEOUT)
               {
                  DebugOut("Map-Maker timed out. Outer: %d, Inner: %d, Roominess: %d, SectorSize: %d, i: %d, j: %d\n", outer, inner, roominess, sectsize, i, j);
                  SmoothEdges();
                  goto DoGrid;
               }
            } while (FindNewPoint(s, li, sectsize, &v));
            MakeSector(v.X, v.Y, li->V1->X, li->V1->Y, li->V2->X, li->V2->Y);
         }
         SmoothEdges();
      }
      DoGrid:
      MakeGrid(30);
   } /* Dig */


   void Level::Verify()
   {  
      for(int i=0; i<NumSectors; i++)
      {
         assert(Sectors[i] && Sectors[i]->Lines[0] && Sectors[i]->Lines[1] && Sectors[i]->Lines[2]);
         for(int j=0; j<3; j++)
         {
            if(!Sectors[i]->Lines[j]->Blocking) assert(Sectors[i]->Neighbors[j]);
            else assert(!Sectors[i]->Neighbors[j]);
            if(Sectors[i]->Neighbors[j])
            {
               bool shared = false;
               for(int k=0; k<3; k++)
               {
                  if(Sectors[i]->Lines[j]==Sectors[i]->Neighbors[j]->Lines[k])
                     shared = true;
               }
               assert(shared);
            }
         }
      }
   } /* Verify */

   
   void Level::MakeGrid(int res)
   {
      ClearGrid();
      int i, num = NumVertices;

      for(i=0;i<num;i++) CheckMinMax(Vertices[i]);
      Xscale = (MaxX - MinX)/res+1, Yscale = (MaxY - MinY)/res+1;

      Grid = new std::vector<Sector *>[res*res];
      GridRes = res;

      num = NumSectors;
      for(i=0;i<num;i++) AddSectToGrid(Sectors[i]);
   } /* MakeGrid */


   void Level::AddSectToGrid(Sector *s)
   {
      RECT bounds = s->BoundingBox();
      int x1 = (bounds.left  - MinX)/Xscale, y1 = (bounds.top    - MinY)/Yscale;
      int x2 = (bounds.right - MinX)/Xscale, y2 = (bounds.bottom - MinY)/Yscale;
      
      for(;y1<=y2;y1++)
      {
         for(int tx = x1;tx<=x2;tx++)
         {
            std::vector<Sector *> *vect = &Grid[y1*GridRes + tx];
            int i, num = vect->size();

            for(i=0;i<num;i++) if ((*vect)[i] == s) goto Next;
            vect->resize(num+1);
            (*vect)[num] = s;
         }
         Next:;
      }
   } /* AddSectToGrid */


   void Level::ClearGrid()
   {
      if (Grid)
      {
         delete[] Grid;
         Grid = NULL;
      }

      MinX = MinY =  0x7FFFFFFF;
      MaxX = MaxY = -0x7FFFFFFF;
   } /* ClearGrid */


   Sector * Level::FindSector(Vertex v)
   {
      if (v.X < MinX || v.Y < MinY || v.X > MaxX || v.Y > MaxY) return NULL;
      int xpos = (v.X - MinX) / Xscale;
      int ypos = (v.Y - MinY) / Yscale;
      std::vector<Sector *> *list = &Grid[ypos * GridRes + xpos];

      int i, num = list->size();
      for(i=0;i<num;i++)
      {
         Sector *s = (*list)[i];
         if(CheckSectorBounds(s, v) && CheckSectorTri(s, v)) return s;
      }
      return NULL;
   } /* FindSector */


   bool Level::CheckSectorBounds(Sector *s, Vertex v)
   {
      RECT bounds = s->BoundingBox();
      return !(v.X < bounds.left || v.X > bounds.right || v.Y < bounds.top || v.Y > bounds.bottom);
   } /* CheckSectorBounds */


   bool Level::CheckSectorTri(Sector *s, Vertex v)
   {
      Line   l(&v, &s->Centroid);

      return !(s->Lines[0]->Intersects(l) || s->Lines[1]->Intersects(l) ||
               s->Lines[2]->Intersects(l));
   } /* CheckSectorTri */


   int FindNewPoint(Sector *s, Line *li, int sectsize, Vertex *out)
   { 
      Vertex scenter = s->Centroid;
      Vertex lcenter((li->V1->X + li->V2->X)/2, (li->V1->Y + li->V2->Y)/2);
      Vertex v1 = *(li->V1), v2 = *(li->V2);
      int    dist=v1.Distance(lcenter);
      if (dist <= sectsize / 10) return 1; // has to be <=
      Line   r(&v1, &v2);
      Vertex result, temp;

      v1.Rotate(lcenter);
      v2.Rotate(lcenter);

      temp = *(r.V1);
      r.V1 = &scenter;

      if(r.Intersects(*li)) result = *(r.V2);
      else result = temp;

      result.X = scenter.X + (result.X - scenter.X) * sectsize/dist;
      result.Y = scenter.Y + (result.Y - scenter.Y) * sectsize/dist;
 
      result.X += (rand() % dist)/2 - dist/4;
      result.Y += (rand() % dist)/2 - dist/4;

      *out = result;
      return 0;
   } /* FindNewPoint */


   Level::Path::~Path()
   {
      if(Pos) delete[] Pos;
   } /* Path Destructor */


   void Level::Push(PathNodeList *list, PathNode *node)
   {
      int pos, num = list->List.size();
      for(pos=0;pos<num;pos++) if (!list->List[pos]) break;
      if (pos == num) list->List.resize(num+20, NULL);
      list->List[pos] = node;
      list->Num++;
   } /* Push */

   Level::PathNode * Level::Pop()
   {
      DWORD  minscore = 0xFFFFFFFF;
      int i=0, mini = 0, num = Open.Num--;
      PathNode *ret;

      for(;num;i++)
      {
         if(ret = Open.List[i])
         {
            if(ret->Score < minscore) mini = i, minscore = ret->Score;
            num--;
         }
      }
      ret = Open.List[mini];
      Open.List[mini] = NULL;
      return ret;
   } /* Pop */

   Level::PathNode * Level::FindNode(PathNodeList *list, Sector *sect)
   {
      int i=0, num = list->Num;
      for(;num;i++)
      {
         if (list->List[i])
         {
            if(list->List[i]->Sector == sect) return list->List[i];
            num--;
         }
      }
      return NULL;
   } /* FindNode */

   int Level::Heuristic(Sector *s1, Sector *s2)
   {
      return s1->Centroid.Distance_squared(s2->Centroid);
   } /* Heuristic */

   void Level::PushNeighbors(PathNode *node, Sector *end)
   {
      Sector *s = node->Sector;
      int i;
      PathNode *newnode;

      for(i=0;i<3;i++)
      {
         if (!s->Lines[i]->Blocking)
         {
            if (FindNode(&Closed, s->Neighbors[i])) continue;
            if ((newnode = FindNode(&Open, s->Neighbors[i])) != NULL &&
                newnode->Score <= node->Score)
            {
               continue;
            }
            
            newnode = new PathNode;
            newnode->Parent = node;
            newnode->Sector = s->Neighbors[i];
            newnode->Cost   = node->Cost + 1;
            newnode->Score  = newnode->Cost + Heuristic(newnode->Sector, end);
            Push(&Open, newnode);
         }
      }
   } /* PushNeighbors */

   Level::Path * Level::ConstructPath(PathNode *node)
   {
      Path *path = new Path;
      int i, num = path->NumNodes = node->Cost + 1;
      path->Pos  = new Vertex[num];

      for(i=num-1;i>=0;i--)
      {
         path->Pos[i] = node->Sector->Centroid;
         node = node->Parent;
      }
      return path;
   } /* ConstructPath */

   Level::Path * Level::FindPath(Sector *start, Sector *end)
   {
      int i, num = Open.Num;
      for(i=0;num;i++)
      {
         if (Open.List[i])
         {
            delete Open.List[i];
            Open.List[i] = NULL;
            num--;
         }
      }
      num = Closed.Num;
      for(i=0;num;i++)
      {
         if (Closed.List[i])
         {
            delete Closed.List[i];
            Closed.List[i] = NULL;
            num--;
         }
      }
      Open.Num = Closed.Num = 0;
      
      PathNode *node = new PathNode;
      node->Parent = NULL;
      node->Sector = start;
      node->Cost   = 0;
      node->Score  = Heuristic(node->Sector, end);
      Push(&Open, node);

      while(Open.Num)
      {
         node = Pop();
         if(node->Sector == end) return ConstructPath(node);
         Push(&Closed, node);
         PushNeighbors(node, end);
      }
      return NULL;
   } /* FindPath(Sector *start, Sector *end) */

   Level::Path * Level::FindPath(Vertex start, Vertex end)
   {
      Sector *s1 = FindSector(start), *s2 = FindSector(end);
      Path   *p;
      if (!s1 || !s2) return NULL;
      if (s1 == s2)
      {
         p       = new Path;
         p->Pos  = new Vertex[2];
         p->NumNodes = 2;
      }
      else p = FindPath(s1, s2);
      assert(p);
      p->Pos[0] = start;
      p->Pos[p->NumNodes-1] = end;
      return p;
   } /* FindPath(Vertex start, Vertex end) */
}

#pragma warning(default : 4786) // identifer >255 chars
