#include <windows.h>
#include <stdlib.h>
#include <assert.h>
#include <time.h>
#include "OverSeer.h"
#include "Frame\Debug.h"
#include "Binfile\BinfMem.h"
#include "Vectors\Vector.h"

#define MAXLOGGEDMSGS   50
#define MAXPLAYERS      1024

OverSeer::OverSeer()
{
   NetID         = 0;
   pDP           = NULL;

   hThread       = hMsgEvent = 0;

   RecvBufLen    = 512;
   SendBufLen    = 128;
   RecvBuffer    = (char *)malloc(RecvBufLen);
   SendBuffer    = (char *)malloc(SendBufLen);

   LoggedOn      = false;

   ScreenUpdated = false;

   ClearPlayers();
   ClearItems();
} /* Constructor */


OverSeer::~OverSeer()
{
   Deinit();
   if (RecvBuffer) free(RecvBuffer);
   if (SendBuffer) free(SendBuffer);
} /* Destructor */


int OverSeer::Init(CDDraw *cd, BasicDP *bdp)
{
   PrimD.SetCDDraw(cd);

   BDP = bdp;
   pDP = bdp->GetDP();

   if (CreateConnection()) return 1;

   return 0;
} /* Init */


void OverSeer::Deinit()
{
   DestroyConnection();
} /* Deinit */


int OverSeer::LogOn(char *name, DWORD msTimeout)
{
   if(!LoggedOn)
   {
      DWORD namelen = strlen(name) + 1;
      DWORD msglen  = sizeof(MsgLogOn) + namelen;
      DWORD timeout;
      
      MsgLogOn *msg = (MsgLogOn *)SendBuffer;

      msg->Type    = MT_LOGON;
      msg->NameLen = namelen;
      memcpy(msg+1, name, namelen);

      if(SendMessage(msg, msglen, true)) return 1;
      
      timeout = clock() + msTimeout;
      while(!LoggedOn)
      {
         if(clock() > timeout) return 2;
         Sleep(60);
      }
   }
   return 0;
} /* LogOn */


void OverSeer::LogOff(DWORD msTimeout)
{
   if(LoggedOn)
   {
      DWORD msglen   = sizeof(MsgLogOff);
      DWORD timeout;
      DWORD NumMsgs=0;

      LoggedOn = false;
      MsgLogOff *msg = (MsgLogOff *)SendBuffer;
      msg->Type = MT_LOGOFF;

      SendMessage(msg, msglen, true);
      timeout = clock() + msTimeout;
      while(clock() < timeout)
      {
         pDP->GetMessageQueue(0, 0, DPMESSAGEQUEUE_SEND, &NumMsgs, NULL);
         if(!NumMsgs) break;
      }
   }
} /* LogOff */


int OverSeer::MoveTo(int x, int y, int z)
{
   assert(myPlayer);

   MsgMoveTo *msg = (MsgMoveTo *)SendBuffer;

   msg->Type = MT_MOVETO;
   msg->X = x;
   msg->Y = y;
   msg->Z = z;

   return SendMessage(msg, sizeof(MsgMoveTo), true);
} /* MoveTo */


int OverSeer::Attack(DWORD id)
{
   assert(myPlayer && !IsItemID(id));

   MsgAttack *msg = (MsgAttack *)SendBuffer;

   msg->Type = MT_ATTACK;
   msg->ID = id;
   return SendMessage(msg, sizeof(MsgAttack), true);
} /* Attack */


int OverSeer::PickUp(DWORD id)
{
   assert(myPlayer && IsItemID(id));

   MsgPickup *msg = (MsgPickup *)SendBuffer;

   msg->Type = MT_PICKUP;
   msg->ID   = id;

   return SendMessage(msg, sizeof(MsgPickup), true);
} /* PickUp */


int OverSeer::Drop(DWORD id)
{
   assert(myPlayer && IsItemID(id));

   MsgDrop *msg = (MsgDrop *)SendBuffer;

   msg->Type = MT_DROP;
   msg->ID   = id;
   msg->X    = myPlayer->Position.X;
   msg->Y    = myPlayer->Position.Y;
   msg->Z    = myPlayer->Position.Z;

   return SendMessage(msg, sizeof(MsgDrop), true);
} /* Drop */


int OverSeer::Equip (DWORD id, Entities::EquipPlace ep)
{
   assert(myPlayer && IsItemID(id) && ep < Entities::EP_EQUIPPLACES && ep >=0 &&
          FindItem(&InvItems, id) != -1);
   
   if (myPlayer->Equip(InvItems.List[FindItem(&InvItems, id)], ep)) return -1;

   DWORD msglen  = sizeof(MsgEquip);
   MsgEquip *msg = (MsgEquip *)SendBuffer;

   msg->Type = MT_EQUIP;
   msg->ID   = id;
   msg->EP   = ep;
   
   return SendMessage(msg, msglen, true);
} /* Equip */


int OverSeer::DeEquip(Entities::EquipPlace ep)
{
   assert(myPlayer && ep < Entities::EP_EQUIPPLACES && ep >=0);

   DWORD msglen    = sizeof(MsgDeEquip);
   MsgDeEquip *msg = (MsgDeEquip *)SendBuffer;

   myPlayer->DeEquip(ep);
   return SendMessage(msg, msglen, true);
} /* DeEquip */


int OverSeer::UseOnTarget(DWORD ObjID, DWORD TargID)
{
   assert(myPlayer && IsItemID(ObjID));

   MsgUseTarg *msg = (MsgUseTarg *)SendBuffer;
   
   msg->Type   = MT_USETARG;
   msg->ObjID  = ObjID;
   msg->TargID = TargID;
   return SendMessage(msg, sizeof(MsgUseTarg), true);
} /* UseOnTarget */


int OverSeer::UseInDirection(DWORD ObjID, int dir)
{
   assert(myPlayer && IsItemID(ObjID));

   MsgUseDir *msg = (MsgUseDir *)SendBuffer;
   
   msg->Type      = MT_USEDIR;
   msg->ObjID     = ObjID;
   msg->Direction = dir;
   return SendMessage(msg, sizeof(MsgUseDir), true);
} /* UseInDirection */


int OverSeer::Draw(CDDSurface *dest)
{
   int i, num = MapItems.LastItem;
   for(i=0;i<num;i++) MapItems.List[i]->Draw(&PrimD, dest, MapItems.List[i]->Position.X, MapItems.List[i]->Position.Y);

   num = LastPlayer;
   for(i=0;i<num;i++) Players[i]->Draw(&PrimD, dest, Players[i]->Position.X, Players[i]->Position.Y);
   
   ScreenUpdated = false;
   return 0;
} /* Draw */


bool OverSeer::NeedRedraw()
{
   return ScreenUpdated;
} /* NeedRedraw */


int OverSeer::CreateConnection()
{
   HRESULT hRet;
   
   DestroyConnection();
   
   if (InitThread()) return 1;

   hRet = pDP->CreatePlayer(&NetID, NULL, hMsgEvent, NULL, 0, 0);
   if (FAILED(hRet)) return 2;

   return 0;
} /* CreateConnection */


void OverSeer::DestroyConnection()
{
   if (NetID && pDP)
   {
      pDP->DestroyPlayer(NetID);
      NetID = 0;
   }
   DeinitThread();

   ClearPlayers();
} /* DestroyConnection */


int OverSeer::InitThread()
{
   SECURITY_ATTRIBUTES sa;
   DWORD id;
   
   DeinitThread();

   sa.nLength = sizeof(SECURITY_ATTRIBUTES);
   sa.lpSecurityDescriptor = NULL;
   sa.bInheritHandle = FALSE;

   hKillEvent = CreateEvent(0, 0, 0, 0);
   hMsgEvent  = CreateEvent(0, 0, 0, 0);
   if(!hKillEvent || !hMsgEvent) return 1;

   InitializeCriticalSection(&OS_CS);

   hThread = CreateThread(&sa, 0, ThreadFunc, this, 0, &id);
   if (hThread == NULL) return 2;

   return 0;
} /* InitThread */


void OverSeer::DeinitThread()
{
   if (hThread)
   {
      SetEvent(hKillEvent);
      WaitForSingleObject(hThread, INFINITE);
      CloseHandle(hThread);
      CloseHandle(hKillEvent);
      CloseHandle(hMsgEvent);
      hThread = NULL;
      DeleteCriticalSection(&OS_CS);
   }
} /* DeinitThread */


DWORD WINAPI OverSeer::ThreadFunc(LPVOID cls)
{
   HANDLE    Events[2];
   DWORD     event;
   OverSeer *over = (OverSeer *)cls;
   char     *ErrStr = "OverSeer::ThreadFunc() Error receiving message: %s\n";

   HRESULT hRet;
   DWORD   SizeNeeded;
   DWORD   SendID, RecvID;

   Events[0] = over->hKillEvent;
   Events[1] = over->hMsgEvent;

   while(TRUE)
   {
      event = WaitForMultipleObjects(2, Events, FALSE, INFINITE);

      switch(event)
      {
         case WAIT_OBJECT_0:     // Kill thread
            ExitThread(0);
            break;

         case WAIT_OBJECT_0 + 1: // Message arriving
            {
               EnterCriticalSection(&over->OS_CS);

               SizeNeeded = over->RecvBufLen;
               DWORD MsgCount;
               hRet = over->BDP->GetDP()->GetMessageCount(over->NetID, &MsgCount);
               if (FAILED(hRet))
               {
                  DebugOut(ErrStr, over->BDP->DPErrorToString(hRet));
                  break;
               }

               while(MsgCount--)
               {
                  hRet = over->BDP->GetDP()->Receive(&SendID, &RecvID, 0,
                                                     over->RecvBuffer, &SizeNeeded);
                  if (FAILED(hRet))
                  {
                     if (hRet == DPERR_BUFFERTOOSMALL)
                     {
                        over->EnlargeRecvBuf(SizeNeeded);
                        if (FAILED(over->BDP->GetDP()->Receive(&SendID, &RecvID, 0,
                                                   over->RecvBuffer, &SizeNeeded)))
                        {
                           DebugOut(ErrStr, over->BDP->DPErrorToString(hRet));
                           break;
                        }
                     }
                     else
                     {
                        DebugOut(ErrStr, over->BDP->DPErrorToString(hRet));
                        break;
                     }
                  }
                  over->HandleMsg(SendID, RecvID);
               }
            }

            LeaveCriticalSection(&over->OS_CS);
            break;
      }
   }
            
   return 0;
} /* ThreadFunc */


void OverSeer::AddPlayer(Entities::Player *player)
{
   int num = Players.size(), last = LastPlayer++;

   if (Players.size() == last) Players.resize(last + 10, NULL);
   Players[last] = player;
} /* AddPlayer */


void OverSeer::RemPlayer(DWORD id)
{
   int i=id, num = --LastPlayer;
   assert(id <= num && myPlayer != Players[id]);
      
   delete Players[id];
   Players[id]  = Players[num];
   Players[num] = NULL;
} /* RemPlayer */


void OverSeer::ClearPlayers()
{
   int i,num = Players.size();
   for(i=0;i<num;i++) delete Players[i];

   Players.resize(0);
   myPlayer   = NULL;
   LastPlayer = 0;
} /* ClearPlayers */


void OverSeer::AddItem(ItemList *list, Entities::Item *item)
{
   assert(FindItem(list, item->ID) == -1);
   std::vector<Entities::Item *> *items = &list->List;
   int last = list->LastItem++;

   if(items->size() == last) items->resize(last + 15, NULL);
   (*items)[last] = item;
} /* AddItem */


void OverSeer::RemItem(ItemList *list, int pos)
{
   assert(pos != -1 && list->LastItem);
   int last = --list->LastItem;
   std::vector<Entities::Item *> *items = &list->List;

   (*items)[pos]  = (*items)[last];
   (*items)[last] = NULL;
} /* RemItem */


void OverSeer::DelItem(ItemList *list, DWORD id)
{
   int pos = FindItem(list, id);
   Entities::Item *item = list->List[pos];
   RemItem(list, pos);
   delete item;
} /* DelItem */

   
int OverSeer::FindItem(ItemList *list, DWORD id)
{
   std::vector<Entities::Item *> *items = &list->List;
   int i, num = list->LastItem;
   Entities::Item *item;

   for(i=0;i<num;i++)
   {
      item = (*items)[i];
      if ((*items)[i]->ID == id) return i;
   }
   return -1;
} /* FindItem */


bool OverSeer::IsItemID(DWORD id)
{
   return id >= MAXPLAYERS;
} /* IsItemID */


void OverSeer::ClearItems()
{
   int i, num = MapItems.LastItem;
   std::vector<Entities::Item *> *items = &MapItems.List;
   MapItems.LastItem = 0;

   for(i=0;i<num;i++) delete (*items)[i];
   items->resize(0);

   num   = InvItems.LastItem;
   InvItems.LastItem = 0;
   items = &InvItems.List;

   for(i=0;i<num;i++) delete (*items)[i];
   items->resize(0);
} /* ClearItems */


int OverSeer::AddLoggedMessage(DWORD color, const char *str)
{
   int num = MessageLog.size();
   if (num < MAXLOGGEDMSGS) MessageLog.resize(num+1);
   else
   {
      delete MessageLog[0];
      for(int i=0;i<MAXLOGGEDMSGS-1;i++) MessageLog[i] = MessageLog[1+1];
   }
   MessageLog[num]        = new LoggedMsg;
   MessageLog[num]->Str   = str;
   MessageLog[num]->Color = color;

   return 0;
} /* AddLoggedMessage */

  
void OverSeer::ClearMessageLog()
{
   int i, num = MessageLog.size();
   for(i=0;i<num;i++) delete MessageLog[i];
   MessageLog.resize(0);
} /* ClearMessageLog */


void OverSeer::ResetState()
{
   ClearPlayers();
   ClearItems();
   ClearMessageLog();
} /* ResetState */


int OverSeer::SendMessage(void *msg, DWORD msglen, bool guaranteed)
{
   HRESULT hRet;
   DWORD flags = DPSEND_ASYNC;

   if (guaranteed) flags |= DPSEND_GUARANTEED;

   hRet = pDP->SendEx(NetID, DPID_SERVERPLAYER, flags, msg, msglen, 0, 0, NULL, NULL);
   if (FAILED(hRet) && hRet != DPERR_PENDING) return hRet;

   return 0;
} /* SendMessage */


int OverSeer::FakeMessage(void *msg, DWORD msglen)
{
   EnlargeRecvBuf(msglen);
   CopyMemory(RecvBuffer, msg, msglen);
   return HandleMsg(DPID_SERVERPLAYER, NetID);
} /* FakeMessage */


void OverSeer::EnlargeRecvBuf(DWORD newsize)
{
   if(newsize > RecvBufLen)
      RecvBuffer = (char *)realloc(RecvBuffer, RecvBufLen = newsize);
} /* EnlargeRecvBuf */


int OverSeer::HandleMsg(DWORD SendID, DWORD RecvID)
{
   if(SendID == DPID_SYSMSG)
   {
      return 1;
   }
   else
   {
      GenericMessage *msg = (GenericMessage *)RecvBuffer;
      switch(msg->Type)
      {
         case MT_UPDATEPLAYERS:
            return HandleUpdatePlayers();
         case MT_MESSAGE:
            return HandlePrintedMessage();
         case MT_CREATEOBJECT:
            return HandleCreateObject();
         case MT_DELETEOBJECT:
            return HandleDeleteObject();
         case MT_ADDPLAYER:
            return HandleAddPlayer();
         case MT_REMPLAYER:
            return HandleRemPlayer();
         case MT_UPDATEINV:
            return HandleUpdateInv();
         case MT_UPDATESTATS:
            return HandleUpdateStats();
         case MT_LOGONACCEPTED:
            return HandleLogonAccept();
         case MT_DROPPED:
            return HandleDropped();
         default:
            return 1;
      }
   }
} /* HandleMsg */


int OverSeer::HandleLogonAccept()
{
   MsgLogOnAccepted *msg = (MsgLogOnAccepted *)RecvBuffer;
   LoggedOn = true;
   mbinfile file;
   int i, num = msg->numItems;

   ResetState();

#pragma message("*** Write InitMap(msg->Seed, msg->Depth);")
   file.open(msg+1, RecvBufLen, file.openro);

   DebugOut("HandleLogonAccept: NumItems(%d)\n", num);

   for(i=0;i<num;i++)
   {
      int x, y, z;
      Entities::Item *item;
      item = new Entities::Item;
      file.read(&item->ID, sizeof(item->ID));
      file.read(&x, sizeof(x));
      file.read(&y, sizeof(y));
      file.read(&z, sizeof(z));
      item->SetPos(x, y, z);
      item->Load(&file);
      DebugOut(" Item %d(%s), (%d,%d,%d)\n", item->ID, item->Name.c_str(), x, y, z);
      AddItem(&MapItems, item);
   }

   MLO_PlayerList *plist = (MLO_PlayerList *)((char *)(msg+1) + file.tell());
   MPL_OnePlayer  *onep  = (MPL_OnePlayer *) (plist+1);

   num = plist->numPlayers;
   assert(num);
   DebugOut("HandleLogonAccept: NumPlayers(%d)\n", num);
   for(i=0;i<num;i++)
   {
      Entities::Player *player;
      player = new Entities::Player;
      player->SetObject(onep->ObjectIndex);
      player->SetPos(onep->X, onep->Y, onep->Z);
      player->SetAccel(0, 0, 0);
      player->Name     = (char *)(onep+1);
      DebugOut(" Player %d: Name(%s), X,Y,Z(%d,%d,%d), Object(%d)\n",
               i, player->Name.c_str(), onep->X, onep->Y, onep->Z, onep->ObjectIndex);
      onep = (MPL_OnePlayer *)((char *)(onep+1) + onep->NameLen);
      AddPlayer(player);
   }

   myPlayer = Players[msg->ID];
   assert(myPlayer);
   myPlayer->SetStr (plist->Str,     myPlayer->ST_ABSOLUTE);
   myPlayer->SetInt (plist->Int,     myPlayer->ST_ABSOLUTE);
   myPlayer->SetDex (plist->Dex,     myPlayer->ST_ABSOLUTE);
   myPlayer->SetSpd (plist->Spd,     myPlayer->ST_ABSOLUTE);
   myPlayer->SetDef (plist->Def,     myPlayer->ST_ABSOLUTE);
   myPlayer->SetAttk(plist->Attk,    myPlayer->ST_ABSOLUTE);
   myPlayer->SetHits(plist->MaxHits, myPlayer->ST_ABSOLUTE);
   myPlayer->CurHits = myPlayer->MaxHits;
   
   DebugOut("HandleLogonAccept: Str(%d), Int(%d), Dex(%d), Spd(%d), Def(%d), Attk(%d), MaxHits(%d)\n",
            plist->Str, plist->Int, plist->Dex, plist->Spd, plist->Def, plist->Attk, plist->MaxHits);
ScreenUpdated = true;
   return 0;
} /* HandleLogonAccept */


int OverSeer::HandleCreateObject()
{
   MsgCreateObject *msg = (MsgCreateObject *)RecvBuffer;
   Entities::Item *item = new Entities::Item;
   mbinfile file;

   file.open(msg+1, RecvBufLen - sizeof(MsgCreateObject), file.openro);
   item->ID = msg->ID;
   item->SetPos(msg->X, msg->Y, msg->Z);
   item->Load(&file);
   DebugOut("HandleCreateObject: ID %d(%s), X,Y,Z(%d, %d, %d) Object(%d)\n",
            msg->ID, item->Name.c_str(), msg->X, msg->Y, msg->Z, item->ObjectIndex);
   AddItem(&MapItems, item);
ScreenUpdated = true;
   return 0;
} /* HandleCreateObject */


int OverSeer::HandleDeleteObject()
{
   MsgDeleteObject *msg = (MsgDeleteObject *)RecvBuffer;
   DelItem(&MapItems, msg->ID);
   DebugOut("HandleDeleteObject: Item %d\n", msg->ID);
ScreenUpdated = true;
   return 0;
} /* HandleDeleteObject */


int OverSeer::HandleUpdateInv()
{
   MsgUpdateInv *msg = (MsgUpdateInv *)RecvBuffer;
   DWORD *IDs        = (DWORD *)(msg+1), id;
   std::vector<Entities::Item *> *list = &InvItems.List;
   Entities::Item *item;
   mbinfile file;
   int i, num = msg->numRems, pos;

   DebugOut("HandleUpdateInv: NumRems(%d)\n", num);
   for(i=0;i<num;i++)
   {
      pos = FindItem(&InvItems, id = *IDs++);
      DebugOut(" Remove item %d\n", id);
      assert(pos != -1);
      myPlayer->Drop((*list)[i]);
      DelItem(&InvItems, pos);
   }

   num = msg->numAdds;
   DebugOut("HandleUpdateInv: NumAdds(%d)\n", num);

   file.open(IDs, num * (sizeof(Entities::Item)+sizeof(DWORD)), file.openro);

   for(i=0;i<num;i++)
   {
      pos = FindItem(&MapItems, id = *IDs++);
      file.seekcur(sizeof(DWORD));
      DebugOut(" Item ID: %d - ", id);
      if(pos == -1)
      {
         item = new Entities::Item;
         item->ID = id;
         int len = item->Load(&file);
         IDs = (DWORD *)((char *)IDs + len);
         AddItem(&InvItems, item);
         myPlayer->PickUp(item);
         DebugOut("Not Found, Item(%s), Object(%d)\n", item->Name.c_str(), item->ObjectIndex);
      }
      else
      {
         AddItem(&InvItems, MapItems.List[pos]);
         RemItem(&MapItems, pos);
         DebugOut("Found, pos(%d)\n", pos);
      }
   }

   return 0;
} /* HandleUpdateInv */


int OverSeer::HandleUpdatePlayers()
{
   MsgUpdatePlayers *msg = (MsgUpdatePlayers *)RecvBuffer;
   MUP_OnePlayer    *mup = (MUP_OnePlayer *)(msg+1);
   DWORD             num = msg->numPlayers;
   Entities::Player *plr;

   DebugOut("HandleUpdatePlayers: NumPlayers(%d)\n", num);
   for(DWORD i=0;i<num;i++,mup++)
   {
      plr = Players[mup->ID];
      plr->SetPos  (mup->pX, mup->pY, mup->pZ);
      plr->SetAccel(mup->vX, mup->vY, mup->vZ);
      plr->RotateObject(mup->Direction);
      DebugOut(" Player %d: Name(%s), Pos(%d,%d,%d), Accel(%d,%d,%d), Dir(%d)\n",
               mup->ID, plr->Name.c_str(), mup->pX, mup->pY, mup->pZ, mup->vX, mup->vY, mup->vZ, mup->Direction);
   }
ScreenUpdated = true;
   return 0;
} /* HandleUpdatePlayers */


int OverSeer::HandleUpdateStats()
{
   MsgUpdateStats *msg = (MsgUpdateStats *)RecvBuffer;

   assert(myPlayer);
   myPlayer->SetStr(msg->Str,      myPlayer->ST_ABSOLUTE);
   myPlayer->SetInt(msg->Int,      myPlayer->ST_ABSOLUTE);
   myPlayer->SetDex(msg->Dex,      myPlayer->ST_ABSOLUTE);
   myPlayer->SetSpd(msg->Spd,      myPlayer->ST_ABSOLUTE);
   myPlayer->SetDef(msg->Def,      myPlayer->ST_ABSOLUTE);
   myPlayer->SetAttk(msg->Attk,    myPlayer->ST_ABSOLUTE);
   myPlayer->SetHits(msg->MaxHits, myPlayer->ST_ABSOLUTE);
   myPlayer->Weight  = msg->Weight;
   myPlayer->CurHits = msg->CurHits;

   DebugOut("HandleUpdateStats: Str(%d), Int(%d), Dex(%d), Spd(%d), Def(%d), Attk(%d), CurHits(%d), MaxHits(%d), Weight(%d)\n",
            msg->Str, msg->Int, msg->Dex, msg->Spd, msg->Def, msg->Attk, msg->CurHits, msg->MaxHits, msg->Weight);

   return 0;
} /* HandleUpdateStats */


int OverSeer::HandleAddPlayer()
{
   MsgAddPlayer  *msg = (MsgAddPlayer *)RecvBuffer;
   MPL_OnePlayer *mop = (MPL_OnePlayer *)(msg+1);
   Entities::Player *player;

   player = new Entities::Player;
   player->SetPos   (mop->X, mop->Y, mop->Z);
   player->SetObject(mop->ObjectIndex);
   player->Name =   (char *)(mop+1);
   AddPlayer(player);

   DebugOut("HandleAddPlayer: Name(%s), X,Y,Z(%d,%d,%d), Object(%d)\n", player->Name.c_str(), mop->X, mop->Y, mop->Z, mop->ObjectIndex);
   return 0;
} /* HandleAddPlayer */


int OverSeer::HandleRemPlayer()
{
   MsgRemPlayer *msg = (MsgRemPlayer *)RecvBuffer;
   DebugOut("HandleRemPlayer: Player %d\n", msg->ID);
   assert(myPlayer != Players[msg->ID]);
   RemPlayer(msg->ID);
   
   return 0;
} /* HandleRemPlayer */


int OverSeer::HandleDropped()
{
   MsgDropped *msg = (MsgDropped *)RecvBuffer;
   DebugOut("HandleDropped:\n");
   LoggedOn = false;
   return 0;
} /* HandleDropped */


int OverSeer::HandlePrintedMessage()
{
   MsgMessage *msg = (MsgMessage *)RecvBuffer;
   DebugOut("HandlePrintedMessage: Color(%d), Msg: %s\n", msg->Color, (msg+1));
   return AddLoggedMessage(msg->Color, (char *)(msg+1));
} /* HandlePrintedMessage */
